// ===================================================================
// Listing 1.  vector.h: Header file for Vector3D class.
// 	Copyright (C) 1992 by Nicholas Wilt.  All rights reserved.
// ===================================================================

// Forward-declare some other classes.
class ostream;

class Vector3D {

public:
    float x, y, z;

    // constructors
    Vector3D() { }
    Vector3D(float X, float Y, float Z) { x = X; y = Y; z = Z; }
    Vector3D(float X) { x = y = z = X; }

    // Function to get at the vector components
    float& operator[] (int inx) {
	if (inx == 0)      return x;
	else if (inx == 1) return y;
	else	       	   return z;
    }

    // Unary operators
    Vector3D Clear() { x = y = z = 0.0; return *this; }
    friend float Magnitude(const Vector3D& v) {
	return sqrt(v.x*v.x + v.y*v.y + v.z*v.z);
    }
    Vector3D operator+ () const;
    Vector3D operator- () const;
    friend Vector3D Normalize(const Vector3D& A);

    // Assignment operators
    Vector3D& operator+= (const Vector3D& A);
    Vector3D& operator-= (const Vector3D& A);
    Vector3D& operator*= (const Vector3D& A);
    Vector3D& operator*= (float A);
    Vector3D& operator/= (float A);

    // Binary operators
    friend Vector3D operator+ (const Vector3D& A, const Vector3D& B);
    friend Vector3D operator- (const Vector3D& A, const Vector3D& B);
    friend Vector3D operator* (const Vector3D& A, const Vector3D& B);
    friend Vector3D operator* (const Vector3D& A, float B);
    friend Vector3D operator* (float A, const Vector3D& B);
    friend Vector3D operator/ (const Vector3D& A, const Vector3D& B);

    friend Vector3D operator/ (const Vector3D& A, float B);
    friend int operator== (const Vector3D& A, const Vector3D& B);
    friend float DotProd(const Vector3D& A, const Vector3D& B);
    friend Vector3D CrossProd (const Vector3D& A, const Vector3D& B);

    friend ostream& operator<< (ostream& s, const Vector3D& A);
};

inline Vector3D Normalize(const Vector3D& A)
{
    return A / Magnitude(A);
}

inline float DotProd (const Vector3D& A, const Vector3D& B)
{
    return A.x * B.x + A.y * B.y + A.z * B.z;
}

inline Vector3D Vector3D::operator+ () const
{
    return *this;
}

inline Vector3D Vector3D::operator- () const
{
    return Vector3D(-x, -y, -z);
}

inline Vector3D& Vector3D::operator+= (const Vector3D& A)
{
    x += A.x;  y += A.y;  z += A.z;
    return *this;
}

inline Vector3D& Vector3D::operator-= (const Vector3D& A)
{
    x -= A.x;  y -= A.y;  z -= A.z;
    return *this;
}

inline Vector3D& Vector3D::operator*= (float A)
{
    x *= A;  y *= A;  z *= A;
    return *this;
}

inline Vector3D& Vector3D::operator/= (float A)
{
    x /= A;  y /= A;  z /= A;
    return *this;
}

inline Vector3D& Vector3D::operator*= (const Vector3D& A)
{
    x *= A.x;  y *= A.y;  z *= A.z;
    return *this;
}

inline Vector3D operator+ (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.x + B.x, A.y + B.y, A.z + B.z);
}

inline Vector3D operator- (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.x - B.x, A.y - B.y, A.z - B.z);
}

inline Vector3D operator* (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.x * B.x, A.y * B.y, A.z * B.z);
}

inline Vector3D operator* (const Vector3D& A, float B)
{
    return Vector3D(A.x * B, A.y * B, A.z * B);
}

inline Vector3D operator* (float A, const Vector3D& B)
{
    return B * A;
}

inline Vector3D operator/ (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.x / B.x, A.y / B.y, A.z / B.z);
}

inline Vector3D operator/ (const Vector3D& A, float B)
{
    return Vector3D(A.x / B, A.y / B, A.z / B);
}

// Might replace floating-point == with comparisons of
// magnitudes, if needed.
inline int operator== (const Vector3D& A, const Vector3D& B)
{
    return (A.x == B.x) && (A.y == B.y) && (A.z == B.z);
}

inline Vector3D CrossProd (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.y * B.z - A.z * B.y,
  		    A.z * B.x - A.x * B.z,
		    A.x * B.y - A.y * B.x);
}

inline ostream&
operator<< (ostream& s, const Vector3D& A)
{
    return s << "(" << A.x << ", " << A.y << ", " << A.z << ")";
}


