;****************************************************************************
; CLRSCR clears the screen or a specified region of the screen using the
; attribute specified.  Its syntax is:
;
;       CLRSCR attr [row1 col1 row2 col2]
;
; where "attr" is the attribute to be used, "row1" and "column1" are
; the row and column numbers of the upper left corner of the region to
; be cleared, and "row2" and "col2" are the row and column numbers of the
; lower right corner.  Row and column numbers are zero-based, so the upper
; left corner of the screen is 0,0.  If row and column numbers are omitted,
; the entire screen is cleared.
;****************************************************************************

code            segment
                assume  cs:code,ds:code
                org     100h
begin:          jmp     short main

msg1            db      13,10,"Syntax: CLRSCR attr [row1 col1 "
                db      "row2 col2",13,10,"$"
msg2            db      13,10,"Invalid parameter",13,10,"$"
attr            db      ?
row1            db      0
col1            db      0
row2            db      24
col2            db      ?

;****************************************************************************
; Procedure MAIN
;****************************************************************************

main            proc    near
                cld                             ;Clear direction flag
                mov     si,81h                  ;Point SI to command line
                call    findchar                ;Advance to first character
                jnc     parse                   ;Branch if found

error1:         mov     dx,offset msg1          ;Display error message
error2:         mov     ah,09h
                int     21h
                mov     ax,4C01h                ;Exit with ERRORLEVEL=1
                int     21h
;
; Parse the command line for attribute and row and column entries.
;
parse:          call    hex2bin                 ;Get attribute value and
                mov     dx,offset msg2          ;  save it
                jc      error2
                mov     attr,bl

                call    findchar                ;Advance to next character
                jnc     readcoords              ;Branch if found
;
; Determine the coordinates for the lower left corner of the screen.
;
                mov     ah,0Fh                  ;Get the number of columns
                int     10h                     ;  displayed (minus 1)
                dec     ah
                mov     col2,ah

                mov     ah,12h                  ;Check for an EGA with
                mov     bl,10h                  ;  video function 12H
                int     10h
                cmp     bl,10h                  ;Branch if no EGA detected
                je      clrscr
                mov     ax,40h                  ;Get number of rows from
                mov     es,ax                   ;  the BIOS Data Area
                mov     al,es:[84h]
                mov     row2,al
                jmp     short clrscr
;
; Read row and column coordinates from the command line
;
readcoords:     call    asc2bin                 ;Read "row1" parameter
                mov     dx,offset msg2
                jc      error2
                mov     row1,al
                call    findchar
                jc      error1

                call    asc2bin                 ;Read "col1" parameter
                mov     dx,offset msg2
                jc      error2
                mov     col1,al
                call    findchar
                jc      error1

                call    asc2bin                 ;Read "row2" parameter
                mov     dx,offset msg2
                jc      error2
                mov     row2,al
                call    findchar
                jc      error1

                call    asc2bin                 ;Read "col2" parameter
                mov     dx,offset msg2
                jc      error2
                mov     col2,al
;
; Clear the region.
;
clrscr:         mov     ax,0600h                ;Use DOS function 06H
                mov     bh,attr                 ;BH = attribute
                mov     ch,row1                 ;CH = upper left row
                mov     cl,col1                 ;CL = upper left column
                mov     dh,row2                 ;DH = lower right row
                mov     dl,col2                 ;DL = lower right column
                int     10h                     ;Do it

                mov     ax,4C00h                ;Exit with ERRORLEVEL=0
                int     21h
main            endp

;****************************************************************************
; FINDCHAR advances SI to the next non-space or non-comma character.
; On return, carry set indicates EOL was encountered.
;****************************************************************************

findchar        proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Loop if space
                je      findchar
                cmp     al,2Ch                  ;Loop if comma
                je      findchar
                dec     si                      ;Point SI to the character
                cmp     al,0Dh                  ;Exit with carry set if end
                je      eol                     ;  of line is reached

                clc                             ;Clear carry and exit
                ret

eol:            stc                             ;Set carry and exit
                ret
findchar        endp

;****************************************************************************
; HEX2BIN converts a hex number entered in ASCII form into a binary
; value in BL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

hex2bin         proc    near
                sub     ah,ah                   ;Initialize registers
                sub     bx,bx

h2b_loop:       lodsb                           ;Get a character
                cmp     al,20h                  ;Exit if space
                je      h2b_exit
                cmp     al,2Ch                  ;Exit if comma
                je      h2b_exit
                cmp     al,0Dh                  ;Exit if carriage return
                je      h2b_exit

                cmp     al,"0"                  ;Check for digits "0"
                jb      h2b_error               ;  through "9"
                cmp     al,"9"
                ja      h2b2

                sub     al,30h                  ;ASCII => binary
h2b1:           mov     cl,4                    ;Multiply BX by 16 and
                shl     bx,cl                   ;  add the latest
                add     bx,ax                   ;  digit
                cmp     bx,255                  ;Error if sum > 255
                ja      h2b_error
                jmp     h2b_loop                ;Loop back for more

h2b2:           and     al,0DFh                 ;Capitalize the letter
                cmp     al,"A"                  ;Check range and exit if
                jb      h2b_error               ;  not "A" through "F"
                cmp     al,"F"
                ja      h2b_error               
                sub     al,37h                  ;ASCII => binary
                jmp     h2b1                    ;Finish and loop back

h2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

h2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
hex2bin         endp

;****************************************************************************
; ASC2BIN converts a decimal number entered in ASCII form into a binary
; value in AL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

asc2bin         proc    near
                sub     ax,ax                   ;Initialize registers
                sub     bh,bh
                mov     dl,10

a2b_loop:       mov     bl,[si]                 ;Get a character
                inc     si
                cmp     bl,20h                  ;Exit if space
                je      a2b_exit
                cmp     bl,2Ch                  ;Exit if comma
                je      a2b_exit
                cmp     bl,0Dh                  ;Exit if carriage return
                je      a2b_exit

                cmp     bl,"0"                  ;Error if character is not
                jb      a2b_error               ;  a number
                cmp     bl,"9"
                ja      a2b_error

                mul     dl                      ;Multiply the value in AL by
                jc      a2b_error               ;  10 and exit on overflow
                sub     bl,30h                  ;ASCII => binary
                add     ax,bx                   ;Add latest value to AX
                cmp     ax,255                  ;Error if sum > 255
                jna     a2b_loop                ;Loop back for more

a2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

a2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
asc2bin         endp

code            ends
                end     begin
