;****************************************************************************
; DRAWBOX draws a box on the screen.  Its syntax is:
;
;       DRAWBOX attr width height [style]
;
; where "attr" specifies the attribute to be used, "width" specifies the
; box width in columns, "height" specifies the height of the in rows, and
; "style" is an optional parameter that specifies whether a single or
; double-line box is to be drawn (1=single, 2=double).  If "style" is
; omitted, DRAWBOX defaults to a single-line box.  The upper left corner
; of the box is drawn at the current cursor position.  The cursor can be
; positioned with the separate SETPOS utility.
;****************************************************************************

code            segment
                assume  cs:code,ds:code
                org     100h
begin:          jmp     short main

msg1            db      13,10,"Syntax: DRAWBOX attr width height "
                db      "[style]",13,10,"$"
msg2            db      13,10,"Invalid parameter",13,10,"$"

attr            db      ?
boxwidth        db      ?,0
boxheight       db      ?,0

row1            db      ?
col1            db      ?
row2            db      ?
col2            db      ?

upleft          db      218
upright         db      191
loleft          db      192
loright         db      217
horizontal      db      196
vertical        db      179

altchars        db      201,187,200,188,205,186

;****************************************************************************
; Procedure MAIN
;****************************************************************************

main            proc    near
                cld                             ;Clear direction flag
                mov     si,81h                  ;Point SI to command line
                call    findchar                ;Advance to first character
                jnc     parse                   ;Branch if found

error1:         mov     dx,offset msg1          ;Display error message
error2:         mov     ah,09h
                int     21h
                mov     ax,4C01h                ;Exit with ERRORLEVEL=1
                int     21h
;
; Parse the command line.
;
parse:          call    hex2bin                 ;Read "attr"
                mov     dx,offset msg2
                jc      error2
                mov     attr,bl
                call    findchar                ;Advance to next character
                jc      error1

                call    asc2bin                 ;Read "width"
                mov     dx,offset msg2
                jc      error2
                sub     al,2                    ;Normalize "width"
                jc      error2                  ;Error if less than 2
                mov     boxwidth,al
                call    findchar                ;Advance to next character
                jc      error1

                call    asc2bin                 ;Read "height"
                mov     dx,offset msg2
                jc      error2
                sub     al,2                    ;Normalize "height"
                jc      error2                  ;Error if less than 2
                mov     boxheight,al
                call    findchar                ;Advance to next character
                jc      drawbox                 ;Branch if there is none

                call    asc2bin                 ;Read "style"
                mov     dx,offset msg2
                jc      error2
                cmp     al,1                    ;Error if greater than 2
                jb      error2                  ;  or less than 1
                cmp     al,2
                ja      error2
                dec     al                      ;Branch if "style"=1
                jz      drawbox

                mov     si,offset altchars      ;Copy alternate graphics
                mov     di,offset upleft        ;  characters to table
                mov     cx,6
                rep     movsb
;
; Draw the box.
;
drawbox:        mov     ah,0Fh                  ;Get active page number
                int     10h                     ;  in BH
                mov     ah,03h                  ;Get current cursor position
                int     10h                     ;  in DH and DL

                mov     row1,dh                 ;Compute corner coordinates
                mov     col1,dl                 ;  of box and store them
                add     dh,boxheight            ;  away
                inc     dh
                mov     row2,dh
                add     dl,boxwidth
                inc     dl
                mov     col2,dl

                mov     ah,09h                  ;Draw upper left corner
                mov     al,upleft
                mov     bl,attr
                mov     cx,1
                int     10h

                mov     dh,row1                 ;Draw upper horizontal
                mov     dl,col1
                inc     dl
                mov     cx,word ptr boxwidth
                call    drawhorizontal

                mov     ah,02h                  ;Draw upper right corner
                mov     dl,col2
                int     10h
                mov     ah,09h
                mov     al,upright
                mov     cx,1
                int     10h

                inc     dh                      ;Draw right vertical
                mov     cx,word ptr boxheight
                call    drawvertical

                mov     ah,02h                  ;Draw lower right corner
                mov     dh,row2
                int     10h
                mov     ah,09h
                mov     al,loright
                mov     cx,1
                int     10h

                mov     dl,col1                 ;Draw lower horizontal
                inc     dl
                mov     cx,word ptr boxwidth
                call    drawhorizontal

                mov     ah,02h                  ;Draw lower left corner
                dec     dl
                int     10h
                mov     ah,09h
                mov     al,loleft
                mov     cx,1
                int     10h

                mov     dh,row1                 ;Draw left vertical
                inc     dh
                mov     cx,word ptr boxheight
                call    drawvertical

                mov     ah,02h                  ;Home the cursor
                dec     dh
                int     10h

                mov     ax,4C00h                ;Exit with ERRORLEVEL=0
                int     21h
main            endp

;****************************************************************************
; DRAWHORIZONTAL draws a horizontal line at the cursor position passed in
; DH and DL.  Length is specified in CX, attribute in BL, and page number in
; BH.  The line is drawn left to right.
;****************************************************************************

drawhorizontal  proc    near
                jcxz    dh_exit                 ;Exit if length=0
                mov     ah,02h                  ;Position the cursor
                int     10h
                mov     ah,09h                  ;Draw horizontal
                mov     al,horizontal
                int     10h
dh_exit:        ret
drawhorizontal  endp

;****************************************************************************
; DRAWVERTICAL draws a vertical line at the cursor position passed in
; DH and DL.  Length is specified in CX, attribute in BL, and page number
; in BH.  The line is drawn top to bottom.
;****************************************************************************

drawvertical    proc    near
                jcxz    dv_exit                 ;Exit if length=0
dv_loop:        mov     ah,02h                  ;Position the cursor
                int     10h
                push    cx                      ;Save CX
                mov     ah,09h                  ;Draw one character
                mov     al,vertical
                mov     cx,1
                int     10h
                inc     dh
                pop     cx                      ;Retrieve CX
                loop    dv_loop                 ;Loop until done
dv_exit:        ret
drawvertical    endp

;****************************************************************************
; FINDCHAR advances SI to the next non-space or non-comma character.
; On return, carry set indicates EOL was encountered.
;****************************************************************************

findchar        proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Loop if space
                je      findchar
                cmp     al,2Ch                  ;Loop if comma
                je      findchar
                dec     si                      ;Point SI to the character
                cmp     al,0Dh                  ;Exit with carry set if end
                je      eol                     ;  of line is reached

                clc                             ;Clear carry and exit
                ret

eol:            stc                             ;Set carry and exit
                ret
findchar        endp

;****************************************************************************
; HEX2BIN converts a hex number entered in ASCII form into a binary
; value in BL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

hex2bin         proc    near
                sub     ah,ah                   ;Initialize registers
                sub     bx,bx

h2b_loop:       lodsb                           ;Get a character
                cmp     al,20h                  ;Exit if space
                je      h2b_exit
                cmp     al,2Ch                  ;Exit if comma
                je      h2b_exit
                cmp     al,0Dh                  ;Exit if carriage return
                je      h2b_exit

                cmp     al,"0"                  ;Check for digits "0"
                jb      h2b_error               ;  through "9"
                cmp     al,"9"
                ja      h2b2

                sub     al,30h                  ;ASCII => binary
h2b1:           mov     cl,4                    ;Multiply BX by 16 and
                shl     bx,cl                   ;  add the latest
                add     bx,ax                   ;  digit
                cmp     bx,255                  ;Error if sum > 255
                ja      h2b_error
                jmp     h2b_loop                ;Loop back for more

h2b2:           and     al,0DFh                 ;Capitalize the letter
                cmp     al,"A"                  ;Check range and exit if
                jb      h2b_error               ;  not "A" through "F"
                cmp     al,"F"
                ja      h2b_error               
                sub     al,37h                  ;ASCII => binary
                jmp     h2b1                    ;Finish and loop back

h2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

h2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
hex2bin         endp

;****************************************************************************
; ASC2BIN converts a decimal number entered in ASCII form into a binary
; value in AL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

asc2bin         proc    near
                sub     ax,ax                   ;Initialize registers
                sub     bh,bh
                mov     dl,10

a2b_loop:       mov     bl,[si]                 ;Get a character
                inc     si
                cmp     bl,20h                  ;Exit if space
                je      a2b_exit
                cmp     bl,2Ch                  ;Exit if comma
                je      a2b_exit
                cmp     bl,0Dh                  ;Exit if carriage return
                je      a2b_exit

                cmp     bl,"0"                  ;Error if character is not
                jb      a2b_error               ;  a number
                cmp     bl,"9"
                ja      a2b_error

                mul     dl                      ;Multiply the value in AL by
                jc      a2b_error               ;  10 and exit on overflow
                sub     bl,30h                  ;ASCII => binary
                add     ax,bx                   ;Add latest value to AX
                cmp     ax,255                  ;Error if sum > 255
                jna     a2b_loop                ;Loop back for more

a2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

a2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
asc2bin         endp

code            ends
                end     begin
