;****************************************************************************
; PC-DIAL is a popup phone directory and dialer that lets you store names,
; company names, and phone numbers and dial the phone with the touch of a
; key provided you have a modem.  Its syntax is
;
;       PC-DIAL [[d:][path]filename] [/B=size] [/M]
;
; where "filename" is the name of the data file to load at start-up,
; "size" is the size of the data buffer (in kilobytes), and /M runs the
; program in monochrome mode on color video adapaters -- useful for lap-
; tops with color adapters but LCD or gas plasma screens.  Once PC-DIAL
; is installed, pressing Alt-Rt Shift pops it up.
;****************************************************************************

defaultsize     equ     16384                   ;Default buffer size

border_color    equ     [bx]                    ;Color table equates
window_color_1  equ     [bx+1]
window_color_2  equ     [bx+2]
hilite_color_1  equ     [bx+3]
hilite_color_2  equ     [bx+4]
shadow_color    equ     [bx+5]
menu_color1     equ     [bx+6]
menu_color2     equ     [bx+7]
title_color     equ     [bx+8]
error_color     equ     [bx+9]

;****************************************************************************
; Program code and data.
;****************************************************************************

code            segment
                assume  cs:code
                org     100h
begin:          jmp     init

signature       db      0,1,2,"PC-DIAL",2,1,0   ;Program signature
prog_id         db      ?                       ;Multiplex ID number

int08h          dd      ?                       ;Interrupt 08H vector
int09h          dd      ?                       ;Interrupt 09H vector
int10h          dd      ?                       ;Interrupt 10H vector
int13h          dd      ?                       ;Interrupt 13H vector
int1Bh          dd      ?                       ;Interrupt 1Bh vector
int23h          dd      ?                       ;Interrupt 23H vector
int24h          dd      ?                       ;Interrupt 24H vector
int25h          dd      ?                       ;Interrupt 25H vector
int26h          dd      ?                       ;Interrupt 26H vector
int28h          dd      ?                       ;Interrupt 28H vector
int2Fh          dd      ?                       ;Interrupt 2FH vector

dosversion      dw      ?                       ;DOS version number
bufferptr       dw      offset databuffer       ;Address of current record
buffertop       dw      offset databuffer       ;Address of top of data
bufferlimit     dw      offset databuffer+defaultsize
buffersize      dw      defaultsize             ;Buffer size (in bytes)
filename        db      128 dup (0)             ;Data file name
inbuffer        db      128 dup (0)             ;String input buffer
progfile        db      128 dup (0)             ;Program file name
egaflag         db      1                       ;1=EGA/VGA/XGA installed
popupflag       db      0                       ;Non-zero if window is up
requestflag     db      0                       ;Non-zero if hotkey pressed
errorflag       dd      ?                       ;Critical error flag address
indos           dd      ?                       ;InDOS flag address
intflags        db      0                       ;User interrupt flags
oldpsp          dw      ?                       ;Active process ID
records         dw      0                       ;Number of records
recordno        dw      0                       ;Current record number
recordlength    dw      0                       ;Length of deleted record
del_record      db      74 dup (0)              ;Last record deleted
searchtext      db      27 dup (0)              ;Text to search for
counter         db      0                       ;Internal countdown timer
ss_register     dw      ?                       ;Contents of SS register
sp_register     dw      ?                       ;Contents of SP register

color_table     db      1Bh,3Fh,1Fh,1Eh         ;Color video attributes
                db      4Fh,07h,7Fh,70h
                db      1Fh,4Fh

mono_table      db      70h,07h,70h,70h         ;Monochrome video attributes
                db      07h,07h,0Fh,0Fh
                db      70h,70h

key_table       db      "QWERTYUIOP",0,0,0,0    ;Alt key character codes
                db      "ASDFGHJKL",0,0,0,0,0
                db      "ZXCVBNM"

menu_cols       db      1,11,22,31,42,53,63,72

color_ptr       dw      offset color_table      ;Pointer to color table
video_segment   dw      0B800h                  ;Video buffer segment
video_offset    dw      ?                       ;Video buffer offset
small_cursor    dw      0607h                   ;Small cursor shape
big_cursor      dw      0407h                   ;Bug cursor shape
line_length     dw      ?                       ;Bytes per video line
cursor_mode     dw      ?                       ;Cursor mode
cursor_pos      dw      ?                       ;Cursor position
video_page      db      ?                       ;Video page number
window_start    db      ?                       ;Starting row of window
columns         dw      0                       ;Number of columns in window
rows            dw      0                       ;Number of rows in window

menutext        db      " F1-Setup  F2-Search  F3-Next  F4-Insert "
                db      " F5-Delete  F6-Paste  F7-Save  F8-Open ",0

title1          db      " Name ",0
title2          db      " Company ",0
title3          db      " Phone ",0

search_prompt   db      "Search For:",0
save_prompt     db      "Save As:",0
open_prompt     db      "File Name:",0
keep_prompt     db      "Save changes to disk (Y/N)?",0
setup_prompt    db      "Save To:",0

errtxt1         db      "No matching records were found",0
errtxt2         db      "There's not enough room for another record",0
errtxt3         db      "You must delete a record before you can paste it",0
errtxt4         db      "There's not enough room to paste the record",0
errtxt5         db      "There are no records to save",0
errtxt6         db      "Invalid path or file name.  File was not saved.",0
errtxt7         db      "An error occurred while the file was being saved",0
errtxt8         db      "Disk full error.  File was only partially saved.",0
errtxt9         db      "Invalid path or file name",0
errtxt10        db      "The file was not found or could not be opened",0
errtxt11        db      "The file is too large to fit in the data buffer",0
errtxt12        db      "The file was not created by PC-DIAL",0
errtxt13        db      "An error occurred while the file was being read",0
errtxt14        db      "There's not enough room to edit this record",0
errtxt15        db      "Either your modem is off or there is no modem "
                db      "attached to COMx",0
errtxt16        db      "The address must be a valid hex number",0
errtxt17        db      "The file was not found or could not be opened for "
                db      "writing",0
errtxt18        db      "An error occurred as the setup information was "
                db      "being written",0
errtxt19        db      "There is no COM port at ",4 dup (0)

msg1a           db      "Wait until the phone begins to ring.  Then pick",0
msg1b           db      "up the handset and press any key to disconnect the",0
msg1c           db      "modem from the line.",0

config1         db      "COM Port",0
config2         db      "COM Port Address",0
config3         db      "Communications Setting",0
config4         db      "Initialization Command",0
config5         db      "Dial Command",0
config6         db      "Dial Command Suffix",0
config7         db      "Disconnect Command",0
config8         db      "Dialing Prefix",0

divisor_data    dw      384,384,96,96,48,48     ;Baud rate data values used
                dw      24,24,12,12             ;  to initialize the UART

format_data     db      03h,1Ch,03h,1Ch,03h     ;Data format values used
                db      1Ch,03h,1Ch,03h,1Ch     ;  to initialize the UART

comm_table      db      "300-N81 ",0
                db      "300-E71 ",0
                db      "1200-N81",0
                db      "1200-E71",0
                db      "2400-N81",0
                db      "2400-E71",0
                db      "4800-N81",0
                db      "4800-E71",0
                db      "9600-N81",0
                db      "9600-E71",0

addr_table      db      "3F8",0                 ;COM port addresses
                db      "2F8",0
                db      "3E8",0
                db      "2E8",0

initcmd         db      "ATS11=55M1",7 dup (0)  ;Modem init command
dialcmd         db      "ATDT",13 dup (0)       ;Dial command
suffix          db      ";",16 dup (0)          ;Dial command suffix
hangcmd         db      "ATH0",13 dup (0)       ;Disconnext command
prefix          db      17 dup (0)              ;Dial prefix
comport         db      0                       ;COM port number
commptr         db      2                       ;Pointer into COMM_TABLE
setup_index     db      0                       ;Setup window index
comstring       db      "COM",0
endcmd          db      13,0

text_table      dw      offset initcmd
                dw      offset dialcmd
                dw      offset suffix
                dw      offset hangcmd
                dw      offset prefix

xerror_ax       dw      ?                       ;Storage array for
xerror_bx       dw      ?                       ;  extended error
xerror_cx       dw      ?                       ;  information
xerror_dx       dw      ?
xerror_si       dw      ?
xerror_di       dw      ?
xerror_ds       dw      ?
xerror_es       dw      ?
                dw      3 dup (0)

;****************************************************************************
; TIMER_INT handles interrupt 08H.
;****************************************************************************

timer_int       proc    far
                pushf                           ;Push FLAGS
                call    cs:[int08h]             ;Call previous handler

                cmp     cs:[counter],0          ;Decrement internal counter
                je      timer0                  ;  if it's not 0 (used by
                dec     cs:[counter]            ;  the DIAL procedure)

timer0:         cmp     cs:[requestflag],0      ;Exit if REQUESTFLAG is
                je      timer_exit              ;  not set

                push    di                      ;Save DI and ES
                push    es
                cmp     cs:[intflags],0         ;Exit if a user interrupt
                jne     timer1                  ;  flag is set
                les     di,cs:[indos]           ;Exit if the InDOS flag
                cmp     byte ptr es:[di],0      ;  is non-zero
                jne     timer1
                les     di,cs:[errorflag]       ;Exit if the critical error
                cmp     byte ptr es:[di],0      ;  flag is non-zero
                jne     timer1
                mov     cs:[requestflag],0      ;Clear REQUESTFLAG
                mov     cs:[popupflag],1        ;Set POPUPFLAG
                call    popup                   ;Pop up the window
                mov     cs:[popupflag],0        ;Clear POPUPFLAG
                jmp     short timer2            ;Exit

timer1:         dec     cs:[requestflag]        ;Decrement counter
timer2:         pop     es                      ;Restore DI and ES
                pop     di
timer_exit:     iret                            ;Return from interrupt
timer_int       endp

;****************************************************************************
; KB_INT handles interrupt 09H.
;****************************************************************************

kb_int          proc    far
                pushf                           ;Push FLAGS
                call    cs:[int09h]             ;Call previous handler

                cmp     cs:[popupflag],1        ;Exit now if POPUPFLAG
                je      kb_exit                 ;  is already set

                push    ax                      ;Save AX and ES
                push    es
                mov     ax,40h                  ;Point ES to the BIOS
                mov     es,ax                   ;  Data Area
                mov     al,es:[17h]             ;Get keyboard flags
                and     al,0Fh                  ;Zero the upper 4 bits
                cmp     al,09h                  ;Is Alt-Rt Shift pressed?
                pop     es                      ;Restore AX and ES
                pop     ax
                jne     kb_exit                 ;No, then branch

                push    di                      ;Save DI and ES
                push    es
                cmp     cs:[intflags],0         ;Exit if a user interrupt
                jne     kb1                     ;  flag is set
                les     di,cs:[indos]           ;Exit if the InDOS flag
                cmp     byte ptr es:[di],0      ;  is non-zero
                jne     kb1
                les     di,cs:[errorflag]       ;Exit if the critical error
                cmp     byte ptr es:[di],0      ;  flag is non-zero
                jne     kb1
                mov     cs:[requestflag],0      ;Clear REQUESTFLAG
                mov     cs:[popupflag],1        ;Set POPUPFLAG
                call    popup                   ;Pop up the window
                mov     cs:[popupflag],0        ;Clear POPUPFLAG
                jmp     short kb2               ;Exit

kb1:            mov     requestflag,18          ;Set request flag
kb2:            pop     es                      ;Restore DI and ES
                pop     di
kb_exit:        iret                            ;Return from interrupt
kb_int          endp

;****************************************************************************
; VIDEO_INT handles interrupt 10H.
;****************************************************************************

video_int       proc    far
                pushf                           ;Save FLAGS
                or      cs:[intflags],01h       ;Set video interrupt flag
                popf                            ;Retrieve FLAGS
                pushf                           ;Push FLAGS
                call    cs:[int10h]             ;Call previous handler
                pushf                           ;Save FLAGS
                and     cs:[intflags],0FEh      ;Clear video interrupt flag
                popf                            ;Retrieve FLAGS
                ret     2                       ;Return with FLAGS intact
video_int       endp

;****************************************************************************
; DISK_INT handles interrupt 13H.
;****************************************************************************

disk_int        proc    far
                pushf                           ;Save FLAGS
                or      cs:[intflags],02h       ;Set disk flag
                popf                            ;Retrieve FLAGS
                pushf                           ;Push FLAGS
                call    cs:[int13h]             ;Call previous handler
                pushf                           ;Save FLAGS
                and     cs:[intflags],0FDh      ;Clear disk flag
                popf                            ;Retrieve FLAGS
                ret     2                       ;Return with FLAGS intact
disk_int        endp

;****************************************************************************
; ERROR_INT handles interrupt 24H (and 1BH and 23H).
;****************************************************************************

error_int       proc    far
                mov     al,03h                  ;Fail the call
error_exit:     iret                            ;Return from interrupt
error_int       endp

;****************************************************************************
; ABS_READ_INT handles interrupt 25H.
;****************************************************************************

abs_read_int    proc    far
                pushf                           ;Save FLAGS
                or      cs:[intflags],04h       ;Set disk flag
                popf                            ;Retrieve FLAGS
                call    cs:[int25h]             ;Call previous handler
                pushf                           ;Save FLAGS
                and     cs:[intflags],0FBh      ;Clear disk flag
                popf                            ;Retrieve FLAGS
                ret                             ;Return with FLAGS on stack
abs_read_int    endp

;****************************************************************************
; ABS_WRITE_INT handles interrupt 26H.
;****************************************************************************

abs_write_int   proc    far
                pushf                           ;Save FLAGS
                or      cs:[intflags],08h       ;Set disk flag
                popf                            ;Retrieve FLAGS
                call    cs:[int26h]             ;Call previous handler
                pushf                           ;Save FLAGS
                and     cs:[intflags],0F7h      ;Clear disk flag
                popf                            ;Retrieve FLAGS
                ret                             ;Return with FLAGS on stack
abs_write_int   endp

;****************************************************************************
; DOSIDLE_INT handles interrupt 28H.
;****************************************************************************

dosidle_int     proc    far
                pushf                           ;Push FLAGS
                call    cs:[int28h]             ;Call previous handler

                cmp     cs:[requestflag],0      ;Exit if REQUESTFLAG is
                je      dosidle_exit            ;  not set

                push    di                      ;Save DI and ES
                push    es
                cmp     cs:[intflags],0         ;Exit if a user interrupt
                jne     dosidle2                ;  flag is set
                les     di,cs:[errorflag]       ;Exit if the critical error
                cmp     byte ptr es:[di],0      ;  flag is non-zero
                jne     dosidle2
                mov     cs:[requestflag],0      ;Clear REQUESTFLAG
                mov     cs:[popupflag],1        ;Set POPUPFLAG
                call    popup                   ;Pop up the window
                mov     cs:[popupflag],0        ;Clear POPUPFLAG
dosidle2:       pop     es                      ;Restore DI and ES
                pop     di
dosidle_exit:   iret                            ;Return from interrupt
dosidle_int     endp

;****************************************************************************
; MPLEX_INT handles interrupt 2FH.  If, on entry, AH is set to PC-DIAL's
; multiplex ID number, MPLEX_INT uses the value in AL as a function code.
; The functions supported are:
;
;    00H    Returns FFH in AL to indicate the program is installed
;           and returns the address of its signature in ES:DI.
;****************************************************************************

mplex_int       proc    far
                pushf                           ;Save FLAGS register
                cmp     ah,cs:[prog_id]         ;Branch if AH holds the
                je      mplex2                  ;  multiplex ID
                popf                            ;Restore FLAGS
                jmp     cs:[int2Fh]             ;Pass the interrupt on
;
; Function 00H verifies that the program is installed.
;               
mplex2:         popf                            ;Restore FLAGS
                or      al,al                   ;Branch if function code
                jnz     mplex3                  ;  is other than 00H
                mov     al,0FFh                 ;Set AL to FFH
                push    cs                      ;Point ES:DI to the program
                pop     es                      ;  signature
                mov     di,offset signature
mplex3:         iret                            ;Return from interrupt
mplex_int       endp

;****************************************************************************
; POPUP pops up the PC-DIAL window.
;****************************************************************************

popup           proc    near
                cli                             ;Disable interrupts
                mov     cs:[ss_register],ss     ;Save the SS register
                mov     cs:[sp_register],sp     ;Save the SP register
                push    cs                      ;Switch to the internal
                pop     ss                      ;  stack
                mov     sp,offset mystack
                sti                             ;Enable interrupts
                cld                             ;Clear direction flag
                push    ax                      ;Save registers
                push    bx
                push    cx
                push    dx
                push    si
                push    bp
                push    ds
                mov     ax,cs                   ;Point DS to the code
                mov     ds,ax                   ;  segment
                assume  ds:code
;
; Save extended error information available from DOS.
;
                cmp     dosversion,030Ah        ;Skip if not 3.10 or later
                jb      save_psp
                push    ds                      ;Save DS
                mov     ah,59h                  ;Get extended error
                sub     bx,bx                   ;  information
                int     21h
                mov     cs:[xerror_ds],ds       ;Save return value of DS
                pop     ds                      ;Set DS to code segment again
                mov     xerror_ax,ax            ;Save return values in
                mov     xerror_bx,bx            ;  XERROR array
                mov     xerror_cx,cx
                mov     xerror_dx,dx
                mov     xerror_si,si
                mov     xerror_di,di
                mov     xerror_es,es
;
; Save the ID for the active process and make this the active one.
;
save_psp:       mov     ah,51h                  ;Get the PSP address of
                int     21h                     ;  the current process
                mov     oldpsp,bx               ;Save it
                mov     ah,50h                  ;Make this the active
                mov     bx,cs                   ;  process by calling
                int     21h                     ;  DOS function 50H
;
; Vector interrupts 1BH, 23H, and 24H to internal routines.
;
                mov     ax,351Bh                ;Hook interrupt 1BH
                int     21h
                mov     word ptr int1Bh,bx
                mov     word ptr int1Bh[2],es
                mov     ax,251Bh
                mov     dx,offset error_exit
                int     21h

                mov     ax,3523h                ;Hook interrupt 23H
                int     21h
                mov     word ptr int23h,bx
                mov     word ptr int23h[2],es
                mov     ax,2523h
                mov     dx,offset error_exit
                int     21h

                mov     ax,3524h                ;Hook interrupt 24H
                int     21h
                mov     word ptr int24h,bx
                mov     word ptr int24h[2],es
                mov     ax,2524h
                mov     dx,offset error_int
                int     21h
;
; Get information about the video environment.
;
                mov     ax,40h                  ;Point ES to the BIOS Data
                mov     es,ax                   ;  Area
                mov     al,es:[49h]             ;Get video mode number in AL
                cmp     al,2                    ;Continue if mode number is
                je      popup1                  ;  2, 3, or 7
                cmp     al,3
                je      popup1
                cmp     al,7
                je      popup1
popup_exit:     jmp     end_popup               ;Exit if it's not

popup1:         mov     ax,es:[4Ah]             ;Get number of columns in AX
                cmp     ax,80                   ;Exit if less than 80 are
                jb      popup_exit              ;  displayed
                shl     ax,1                    ;Compute bytes per video line
                mov     line_length,ax          ;Store it in LINE_LENGTH

                mov     ax,es:[4Eh]             ;Get offset address of the
                mov     video_offset,ax         ;  video buffer
                test    byte ptr es:[63h],40h   ;Branch if this is a color
                jnz     popup2                  ;  video mode
                mov     color_ptr,offset mono_table     ;Prepare for mono-
                mov     video_segment,0B000h            ;  chrome if not
                mov     small_cursor,0B0Ch
                mov     big_cursor,080Ch

popup2:         mov     window_start,24         ;Initialize WINDOW_START
                cmp     egaflag,0               ;Branch if this is a CGA
                je      popup3                  ;  or MDA
                mov     al,es:[84h]             ;Get number of rows displayed
                mov     window_start,al         ;Store it in WINDOW_START
popup3:         sub     window_start,11         ;Compute window location

                mov     ax,es:[60h]             ;Get the cursor mode
                mov     cursor_mode,ax          ;Store it
                mov     dx,es:[63h]             ;Place CRTC address in DX
                mov     bl,es:[62h]             ;Get the active page
                sub     bh,bh                   ;  number in BX
                mov     video_page,bl           ;Save it
                shl     bx,1                    ;Multiply it by 2
                mov     ax,es:[bx+50h]          ;Get the cursor address
                mov     cursor_pos,ax           ;Store it
;
; Save the screen and display the popup window.
;               
                mov     ch,window_start         ;Load CX with the address of
                sub     cl,cl                   ;  the upper left corner
                mov     dh,ch                   ;Load DX with the address of
                add     dh,11                   ;  the lower right corner
                mov     dl,79
                push    cx                      ;Save CX and DX
                push    dx
                mov     ax,cs                   ;Point ES:DI to the primary
                mov     es,ax                   ;  screen buffer
                mov     di,offset screenbuffer1
                call    saveregion              ;Save the screen
                pop     dx                      ;Restore CX and DX
                pop     cx

                push    cx                      ;Save CX and DX again
                push    dx
                dec     dh
                mov     bx,color_ptr            ;Load AH with the border's
                mov     ah,border_color         ;  color attribute
                call    drawbox                 ;Draw the window border
                pop     dx                      ;Restore CX and DX
                pop     cx

                push    cx                      ;Save CX and DX yet again
                push    dx
                add     cx,0101h                ;Blank the interior of
                sub     dx,0201h                ;  the window using
                mov     bh,window_color_1       ;  interrupt 10H,
                mov     ax,0600h                ;  function 06H in
                int     10h                     ;  the video BIOS

                mov     dh,window_start         ;Draw the highlight bar
                add     dh,5                    ;  across the center of
                mov     dl,1                    ;  the window
                mov     bx,color_ptr
                mov     al,hilite_color_1
                mov     cx,78
                call    write_attr

                mov     ah,border_color         ;Draw the top of the left
                mov     al,0C2h                 ;  inside vertical bar
                mov     dh,window_start
                mov     dl,29
                call    write_char

                mov     dh,window_start         ;Draw the top of the right
                mov     dl,58                   ;  inside vertical bar 
                call    write_char

                mov     al,0C1h                 ;Draw the bottom of the
                mov     dh,window_start         ;  left inside vertical
                add     dh,10                   ;  bar
                mov     dl,29
                call    write_char

                mov     dh,window_start         ;Draw the bottom of the
                add     dh,10                   ;  right inside vertical
                mov     dl,58                   ;  bar
                call    write_char

                push    ax                      ;Draw the left inside
                mov     dh,window_start         ;  vertical bar
                inc     dh
                mov     dl,29
                call    compute_address
                mov     di,ax
                mov     es,video_segment
                pop     ax
                mov     cx,9
                call    drawvertical

                push    ax                      ;Draw the right inside
                mov     dh,window_start         ;  vertical bar
                inc     dh
                mov     dl,58
                call    compute_address
                mov     di,ax
                pop     ax
                mov     cx,9
                call    drawvertical

                mov     ah,title_color          ;Display "Name" title
                mov     dh,window_start
                mov     dl,12
                mov     si,offset title1
                call    write_string

                mov     dh,window_start         ;Display "Company" title
                mov     dl,39
                mov     si,offset title2
                call    write_string

                mov     dh,window_start         ;Display "Phone" title
                mov     dl,65
                mov     si,offset title3
                call    write_string

                mov     dh,window_start         ;Display the menu at the
                add     dh,11                   ;  bottom of the window
                sub     dl,dl
                push    dx
                mov     bx,color_ptr
                mov     ah,menu_color2
                mov     si,offset menutext
                call    write_string
                pop     dx
                mov     al,menu_color1
                mov     si,offset menu_cols
                mov     cx,8
popup4:         push    cx
                push    dx
                mov     dl,[si]
                inc     si
                mov     cx,3
                call    write_attr
                pop     dx
                pop     cx
                loop    popup4

                mov     si,bufferptr            ;Point SI to data
                call    showpage                ;Display the data

                mov     ah,01h                  ;Hide the cursor
                mov     cx,2000h
                int     10h
;
; Process keystrokes with the window displayed.
;
getkey:         call    readkey                 ;Read a keystroke
                or      al,al                   ;Branch if it's an extended
                jz      extended                ;  code

                cmp     al,13                   ;Process the Enter key
                jne     escape
                call    dial
                jmp     getkey

escape:         cmp     al,27                   ;Process the Esc key
                jne     tab
                jmp     close

tab:            cmp     al,9                    ;Process the Tab key
                jne     getkey
                call    edit
                cmp     ax,4800h                ;Branch if Up-Arrow was
                je      up                      ;  pressed
                cmp     ax,5000h                ;Branch if Down-Arrow was
                je      down                    ;  pressed
                jmp     getkey
;
; Process extended keycodes.
;
extended:       cmp     ah,59                   ;Process F1 (Setup)
                jne     f2
                call    setup
                jmp     getkey

f2:             cmp     ah,60                   ;Process F2 (Search)
                jne     f3
                call    search
                jmp     getkey

f3:             cmp     ah,61                   ;Process F3 (Next)
                jne     f4
                call    next
                jmp     getkey

f4:             cmp     ah,62                   ;Process F4 (Insert)
                jne     f5
                call    insert
                jmp     getkey

f5:             cmp     ah,63                   ;Process F5 (Delete)
                jne     f6
                call    delete
                jmp     getkey

f6:             cmp     ah,64                   ;Process F6 (Paste)
                jne     f7
                call    paste
                jmp     getkey

f7:             cmp     ah,65                   ;Process F7 (Save)
                jne     f8
                call    save
                jmp     getkey

f8:             cmp     ah,66                   ;Process F8 (Open)
                jne     up
                call    open
                jmp     getkey

up:             cmp     ah,72                   ;Process the Up-Arrow key
                jne     down
                cmp     recordno,0              ;Ignore if we're already
                je      up1                     ;  at the top
                dec     recordno                ;Decrement the record number
                mov     al,1                    ;Compute the address of the
                mov     di,bufferptr            ;  record previous to this
                call    findlast                ;  one
                mov     bufferptr,di            ;Save it in BUFFERPTR
                mov     si,di                   ;Transfer it to SI
                call    showpage                ;Display the page
up1:            jmp     getkey                  ;Return to the input loop

down:           cmp     ah,80                   ;Process the Down-Arrow key
                jne     pgup
                mov     ax,recordno             ;Ignore if we're already
                cmp     ax,records              ;  at the bottom
                je      down1
                inc     recordno                ;Increment the record number
                mov     al,1                    ;Compute the address of the
                mov     di,bufferptr            ;  record following this
                call    findnext                ;  one
                mov     bufferptr,di            ;Save it in BUFFERPTR
                mov     si,di                   ;Transfer it so SI
                call    showpage                ;Display the page
down1:          jmp     getkey                  ;Return to the input loop

pgup:           cmp     ah,73                   ;Process the PgUp key
                jne     pgdn
                cmp     recordno,0              ;Ignore if we're already at
                je      pgup2                   ;  the top
                mov     cx,recordno             ;Compute the new record
                sub     cx,9                    ;  number by subtracting
                cmp     cx,0                    ;  9 and adjusting to 0
                jge     pgup1                   ;  if necessary
                mov     cx,0
pgup1:          mov     ax,recordno             ;Compute the relative
                sub     ax,cx                   ;  displacement
                mov     recordno,cx             ;Store the new record number
                mov     di,bufferptr            ;Retrieve the buffer pointer
                call    findlast                ;Find the address of the new
                mov     bufferptr,di            ;  record and save it
                mov     si,di                   ;Transfer address to SI
                call    showpage                ;Display the page
pgup2:          jmp     getkey                  ;Return to the input loop

pgdn:           cmp     ah,81                   ;Process the PgDn key
                jne     homekey
                mov     ax,recordno             ;Ignore it if we're already
                cmp     ax,records              ;  at the bottom
                je      pgdn2
                add     ax,9                    ;Compute the new record
                cmp     ax,records              ;  number by adding 9 and
                jbe     pgdn1                   ;  adjusting to the highest
                mov     ax,records              ;  number if necessary
pgdn1:          mov     cx,recordno             ;Compute the relative
                mov     recordno,ax             ;  displacement
                sub     ax,cx
                mov     di,bufferptr            ;Retrieve the buffer pointer
                call    findnext                ;Find the address of the new
                mov     bufferptr,di            ;  record and save it
                mov     si,di                   ;Transfer address to SI
                call    showpage                ;Display the page
pgdn2:          jmp     getkey                  ;Return to the input loop

homekey:        cmp     ah,71                   ;Process the Home key
                jne     endkey
                cmp     recordno,0              ;Ignore it if we're already
                je      homekey1                ;  at the top
                mov     recordno,0              ;Set record number to 0
                mov     si,offset databuffer    ;Reset SI and BUFFERPTR
                mov     bufferptr,si
                call    showpage                ;Display the page
homekey1:       jmp     getkey                  ;Return to the input loop

endkey:         cmp     ah,79                   ;Process the End key
                jne     altchar
                mov     ax,records              ;Ignore it if we're already
                cmp     ax,recordno             ;  at the bottom
                je      endkey1
                mov     recordno,ax             ;Determine the address of
                call    findrec                 ;  the final record
                mov     bufferptr,di            ;Save it in BUFFERPTR
                mov     si,di                   ;Transfer address to SI
                call    showpage                ;Display the page
endkey1:        jmp     getkey                  ;Return to the input loop

altchar:        cmp     ah,16                   ;Branch if scan code is less
                jb      altchar1                ;  than 16 (Alt-Q)
                cmp     ah,50                   ;Also branch if it's greater
                ja      altchar1                ;  than 50 (Alt-M)
                mov     al,ah                   ;Transfer key code to AL
                sub     al,16                   ;Subtract 16 from it
                mov     bx,offset key_table     ;Point BX to table of codes
                xlat                            ;Get the ALth byte from it
                or      al,al                   ;Exit if AL holds 0
                jz      altchar1
                call    qsearch                 ;Do a quick search
altchar1:       jmp     getkey                  ;Return to the input loop
;
; Close the window.
;
close:          pop     dx                      ;Retrieve window coordinates
                pop     cx
                mov     si,offset screenbuffer1 ;Point SI to saved video data
                call    restregion              ;Restore the screen
                mov     ah,02h                  ;Restore the cursor position
                mov     bh,video_page
                mov     dx,cursor_pos
                int     10h
                mov     ah,01h                  ;Restore the cursor mode
                mov     cx,cursor_mode
                int     10h
;
; Reactivate the old PSP, restore interrupt vectors, and restore extended
; error information before exiting.
;
                mov     ah,50h                  ;Reactivate the PSP that was
                mov     bx,oldpsp               ;  current when this process
                int     21h                     ;  took over

                assume  ds:nothing
                mov     ax,251Bh                ;Restore interrupt 1BH
                lds     dx,cs:[int1Bh]          ;  vector
                int     21h
                mov     ax,2523h                ;Restore interrupt 23H
                lds     dx,cs:[int23h]          ;  vector
                int     21h
                mov     ax,2524h                ;Restore interrupt 24H
                lds     dx,cs:[int24h]          ;  vector
                int     21h

                cmp     cs:[dosversion],030Ah   ;Skip if not 3.10 or later
                jb      end_popup
                mov     ax,cs                   ;Make sure DS points to
                mov     ds,ax                   ;  this segment
                mov     ax,5D0Ah                ;Restore information with
                mov     dx,offset xerror_ax     ;  DOS function 5Dh
                int     21h
;
; Restore register values, restore the stack, and exit.
;
end_popup:      pop     ds
                pop     bp
                pop     si
                pop     dx
                pop     cx
                pop     bx
                pop     ax
                cli
                mov     ss,cs:[ss_register]
                mov     sp,cs:[sp_register]
                sti
                ret
popup           endp

;****************************************************************************
; READKEY reads a keycode from the keyboard buffer.  While it waits for
; a key to be pressed, READKEY generates interrupt 28H over and over to
; give other TSRs a chance to pop up.
;****************************************************************************

readkey         proc    near
                assume  ds:code
                mov     ah,01h                  ;Check keyboard buffer and
                int     16h                     ;  branch if it contains
                jnz     rkey1                   ;  a keycode
                int     28h                     ;Execute an INT 28H and
                jmp     readkey                 ;  loop back
rkey1:          mov     ah,00h                  ;Read the keycode and exit
                int     16h
                ret
readkey         endp

;****************************************************************************
; SAVEREGION saves a region of video memory to a user-specified buffer.
; On entry, CX holds the row and column address of the region's upper left
; corner, DX holds the row and column address of the lower right corner,
; and ES:DI points to the buffer where data will be stored.
;****************************************************************************

saveregion      proc    near
                sub     dh,ch                   ;Compute the number of rows
                inc     dh                      ;  that will be read and
                mov     byte ptr rows,dh        ;  buffered
                sub     dl,cl                   ;Then compute the number of
                inc     dl                      ;  columns
                mov     byte ptr columns,dl

                mov     dx,cx                   ;Place starting address in DX
                call    compute_address         ;Compute the memory address
                mov     si,ax                   ;Transfer result to SI
                push    ds                      ;Save DS
                mov     ds,video_segment        ;Point DS to the video buffer
                assume  ds:nothing

                mov     cx,rows                 ;Load CX with row count
sr_loop:        push    cx                      ;Save count
                push    si                      ;Save starting address
                mov     cx,columns              ;Load CX with column count
                rep     movsw                   ;Buffer one row of text
                pop     si                      ;Retrieve SI
                add     si,line_length          ;Point it to the next row
                pop     cx                      ;Rerieve count
                loop    sr_loop                 ;Loop until all rows are done

                pop     ds                      ;Restore DS and exit
                assume  ds:code
                ret
saveregion      endp

;****************************************************************************
; RESTREGION restores a region of video memory from a user-specified buffer.
; On entry, CX holds the row and column address of the region's upper left
; corner, DX holds the row and column address of the lower right corner,
; and DS:SI points to the buffer the data will be retrieved from.
;****************************************************************************

restregion      proc    near
                sub     dh,ch                   ;Compute the number of rows
                inc     dh                      ;  that will be written
                mov     byte ptr rows,dh        ;
                sub     dl,cl                   ;Then compute the number of
                inc     dl                      ;  columns
                mov     byte ptr columns,dl

                mov     dx,cx                   ;Place starting address in DX
                call    compute_address         ;Compute the memory address
                mov     di,ax                   ;Transfer result to DI
                mov     es,video_segment        ;Point ES to the video buffer

                mov     cx,rows                 ;Load CX with row count
rr_loop:        push    cx                      ;Save count
                push    di                      ;Save starting address
                mov     cx,columns              ;Load CX with column count
                rep     movsw                   ;Buffer one row of text
                pop     di                      ;Retrieve DI
                add     di,line_length          ;Point it to the next row
                pop     cx                      ;Rerieve count
                loop    rr_loop                 ;Loop until all rows are done
                ret
restregion      endp

;****************************************************************************
; DRAWBOX draws a box in text mode using single-line graphics characters.
; On entry, CX holds the row and column address of the upper left corner,
; and DX holds the row and column address of the lower right corner.  AH
; holds the video attribute used to draw the box.
;****************************************************************************

drawbox         proc    near
                sub     dh,ch                   ;Compute the number of rows
                dec     dh                      ;  inside the box
                mov     byte ptr rows,dh
                sub     dl,cl                   ;Then compute the number of
                dec     dl                      ;  columns
                mov     byte ptr columns,dl

                push    ax                      ;Save video attribute
                mov     dx,cx                   ;Place starting address in DX
                call    compute_address         ;Compute the memory address
                mov     di,ax                   ;Transfer result to DI
                mov     es,video_segment        ;Point ES to the video buffer
                pop     ax                      ;Retrieve video attribute
                push    di                      ;Save video buffer address

                mov     al,0DAh                 ;Draw the upper left corner
                stosw                           ;  of the box
                mov     al,0C4h                 ;Draw the upper horizontal
                mov     cx,columns
                rep     stosw
                mov     al,0BFh                 ;Draw the upper right corner
                stosw                           ;  of the box

                sub     di,2                    ;Point DI to the end of the
                add     di,line_length          ;  second row of the box
                mov     cx,rows                 ;Draw right vertical
                call    drawvertical
                mov     al,0D9h                 ;Draw the lower right corner
                stosw                           ;  of the box

                pop     di                      ;Retrieve address
                add     di,line_length          ;Point DI to the second row
                mov     cx,rows                 ;Draw left vertical
                call    drawvertical
                mov     al,0C0h                 ;Draw the lower left corner
                stosw                           ;  of the box

                mov     al,0C4h                 ;Draw the lower horizontal
                mov     cx,columns
                rep     stosw
                ret
drawbox         endp

;****************************************************************************
; DRAWVERTICAL draws a vertical line.  On entry, ES:DI points to the
; location in the video buffer, AH holds the video attribute, and CX
; holds the length of the line in rows.
;****************************************************************************

drawvertical    proc    near
                mov     al,0B3h                 ;Load AL with ASCII code
dv_loop:        stosw                           ;Write one character
                sub     di,2                    ;Point DI to the character
                add     di,line_length          ;  cell on the next row
                loop    dv_loop                 ;Loop until done
                ret
drawvertical    endp

;****************************************************************************
; WRITE_STRING writes an ASCIIZ string at the row and column address
; specified in DH and DL.  On entry, AH contains the video attribute and
; DS:SI points to the string.
;****************************************************************************

write_string    proc    near
                push    ax                      ;Save video attribute
                call    compute_address         ;Compute the memory address
                mov     di,ax                   ;Transfer result to DI
                mov     es,video_segment        ;Point ES to the video buffer
                pop     ax                      ;Retrieve video attribute

ws_loop:        lodsb                           ;Get a character
                or      al,al                   ;Exit if it's zero
                jz      ws_exit
                stosw                           ;Display it
                jmp     ws_loop                 ;Loop back for more
ws_exit:        ret
write_string    endp

;****************************************************************************
; WRITE_CHAR writes a character and attribute at the row and column
; address specified in DH and DL.  On entry, AH holds the video attribute
; and AL holds the character's ASCII code.
;****************************************************************************

write_char      proc    near
                push    ax                      ;Save video attribute
                call    compute_address         ;Compute the memory address
                mov     di,ax                   ;Transfer result to DI
                mov     es,video_segment        ;Point ES to the video buffer
                pop     ax                      ;Retrieve video attribute
                stosw                           ;Write the character and
                ret                             ;  attribute and exit
write_char      endp

;****************************************************************************
; SHOW_STRING writes an ASCIIZ string at the row and column address
; specified in DH and DL.  On entry, DS:SI points to the string.  Video
; attributes are preserved.
;****************************************************************************

show_string     proc    near
                call    compute_address         ;Compute the memory address
                mov     di,ax                   ;Transfer result to DI
                mov     es,video_segment        ;Point ES to the video buffer
ss_loop:        lodsb                           ;Get a character
                or      al,al                   ;Exit if it's zero
                jz      ss_exit
                stosb                           ;Display it
                inc     di                      ;Skip attribute byte
                jmp     ss_loop                 ;Loop back for more
ss_exit:        ret
show_string     endp

;****************************************************************************
; SHOW_CHAR writes a character only (no attribute) at the row and column
; address specified in DH and DL.  On entry, the character is in AL.
;****************************************************************************

show_char       proc    near
                push    ax                      ;Save the character
                call    compute_address         ;Compute the memory address
                mov     di,ax                   ;Transfer it to DI
                pop     ax                      ;Retrieve the character
                mov     es,video_segment        ;Point ES to the video buffer
                stosb                           ;Display the character
                ret
show_char       endp

;****************************************************************************
; WRITE_ATTR writes the attribute in AL to the row and column address
; passed in DH and DL.  On entry, CX holds the number of consecutive
; attribute bytes to write.
;****************************************************************************

write_attr      proc    near
                push    ax                      ;Save video attribute
                push    cx                      ;Save count
                call    compute_address         ;Compute the memory address
                mov     di,ax                   ;Transfer result to DI
                inc     di                      ;Point DI to attribute bytes
                mov     es,video_segment        ;Point ES to the video buffer
                pop     cx                      ;Retrieve count
                pop     ax                      ;Retrieve video attribute
wa_loop:        stosb                           ;Write one attribute
                inc     di                      ;Advance DI to next one
                loop    wa_loop                 ;Loop until done
wa_exit:        ret
write_attr      endp

;****************************************************************************
; COMPUTE_ADDRESS returns in AX the video buffer address that corresponds
; to the row and column number passed in DH and DL.  Before this procedure
; is called, LINE_LENGTH must contain the number of bytes per video line
; and VIDEO_OFFSET must contain the offset within the video buffer of the
; current video page.
;****************************************************************************

compute_address proc    near
                mov     cl,dl                   ;Save DL in CL
                mov     al,dh                   ;Get starting row in AL
                cbw                             ;Convert byte to word in AX
                mul     line_length             ;Multiply by bytes per row
                mov     dl,cl                   ;Load DL with column number
                shl     dx,1                    ;Multiply starting column by 2
                add     ax,dx                   ;Add it to AX
                add     ax,video_offset         ;Add video buffer offset
                ret
compute_address endp

;****************************************************************************
; STRLEN returns the length of the ASCIIZ string pointed to by DS:SI in CX.
;****************************************************************************

strlen          proc    near
                push    si                      ;Save SI
                sub     cx,cx                   ;Initialize count
strlen1:        lodsb                           ;Get a character
                or      al,al                   ;Exit if it's zero
                jz      strlen_exit
                inc     cx                      ;Otherwise increment the
                jmp     strlen1                 ;  count and loop back
strlen_exit:    pop     si                      ;Restore SI and exit
                ret
strlen          endp

;****************************************************************************
; STRCAT appends the ASCIIZ string pointed to by ES:DI to the one at DS:SI.
;****************************************************************************

strcat          proc    near
                call    strlen                  ;Compute the string length
                add     si,cx                   ;Add the result to SI
                xchg    si,di                   ;Exchange SI and DI
                call    copyz                   ;Append the string
                ret
strcat          endp

;****************************************************************************
; EDIT_STRING permits a string to be edited.  On entry, DS:SI points to
; the string, DH and DL identify the row and column where the string is
; displayed, and CX holds the maximum number of characters that will be
; accepted.  If AL is 0 when the routine is called, the cursor is
; positioned at the beginning of the string; if it's 1, the cursor is
; positioned at the end.  If BH is 0 when the routine is called, Tab and
; Shift-Tab are ignored; if BH is 1, Tab and Shift-Tab will terminate input
; If BL is 0 when the routine is called, Up- and Down-Arrow are ignored;  if
; BL is 1, Up- and Down-Arrow will terminate input.
;
; On return, CX holds the length of the string, not including the string
; delimiter at the end, and AX holds the key code for the key that was
; pressed to terminate input.
;
; NOTE: The values of DX (the current row and column numbers) and SI (the
; pointer to the current location in the string) are maintained throughout.
;****************************************************************************

chars           dw      ?                       ;Character count
maxchars        dw      ?                       ;Maximum number of characters
stringstart     dw      ?                       ;String starting address
stringend       dw      ?                       ;String ending address
insflag         db      ?                       ;0=Insert off, 1=Insert on
tabflag         db      ?                       ;0=Ignore Tab and Shift-Tab
updnflag        db      ?                       ;0=Ignore Up- and Down-Arrow

edit_string     proc    near
                mov     maxchars,cx             ;Initialize pointers and
                mov     stringstart,si          ;  variables
                mov     tabflag,bh
                mov     updnflag,bl
                mov     insflag,0
                push    ax
                call    strlen
                pop     ax
                mov     chars,cx
                mov     stringend,cx
                add     stringend,si
                or      al,al
                jz      edstr1
                add     dl,cl
                add     si,cx
edstr1:         mov     ah,02h                  ;Position the cursor at
                mov     bh,video_page           ;  the beginning or end
                int     10h                     ;  of the string
                mov     ah,01h                  ;Display the cursor
                mov     cx,small_cursor
                int     10h
;
; Wait for a keypress and process conventional key codes.
;
editkey:        call    readkey                 ;Get a key code
                or      al,al                   ;Branch if it's not an
                jnz     ekey1                   ;  extended code
                jmp     ekey10                  ;Jump if it is

ekey1:          cmp     al,8                    ;Process the Backspace key
                jne     ekey2
                cmp     si,stringstart
                je      editkey
                mov     ah,02h
                mov     bh,video_page
                dec     dl
                int     10h
                dec     si
                jmp     ekey16a

ekey2:          cmp     al,9                    ;Process the Tab key
                jne     ekey3
                cmp     tabflag,0
                je      editkey
ekey2a:         jmp     edstr_exit

ekey3:          cmp     al,13                   ;Process the Enter key
                je      ekey2a

ekey4:          cmp     al,27                   ;Process the Esc key
                je      ekey2a

ekey5:          cmp     insflag,0               ;Process a character key
                jne     ekey6                   ;Branch if the character
                cmp     si,stringend            ;  will be inserted
                je      ekey6
                mov     [si],al
                push    dx
                call    show_char
                pop     dx
ekey5a:         inc     dl
                mov     ah,02h
                mov     bh,video_page
                int     10h
                inc     si
                jmp     editkey

ekey6:          mov     cx,chars                ;Insert a character at the
                cmp     cx,maxchars             ;  current cursor location
                je      editkey
                mov     cx,stringend
                sub     cx,si
                inc     cx
                push    ax
                push    si
                mov     si,stringend
                mov     di,si
                inc     di
                mov     bx,cs
                mov     es,bx
                std
                rep     movsb
                cld
                pop     si
                pop     ax
                mov     [si],al
                inc     stringend
                inc     chars
                push    dx
                push    si
                call    show_string
                pop     si
                pop     dx
                jmp     ekey5a
;
; Process extended key codes.
;
ekey10:         cmp     ah,15                   ;Process Shift-Tab
                jne     ekey11
                cmp     tabflag,0
                je      ekey11a
                jmp     edstr_exit

ekey11:         cmp     ah,71                   ;Process the Home key
                jne     ekey12
                mov     cx,si
                sub     cx,stringstart
                mov     ah,02h
                mov     bh,video_page
                sub     dl,cl
                int     10h
                mov     si,stringstart
ekey11a:        jmp     editkey

ekey12:         cmp     ah,75                   ;Process the Left-Arrow key
                jne     ekey13
                cmp     si,stringstart
                je      ekey12a
                mov     ah,02h
                mov     bh,video_page
                dec     dl
                int     10h
                dec     si
ekey12a:        jmp     editkey

ekey13:         cmp     ah,77                   ;Process the Right-Arrow key
                jne     ekey14
                cmp     si,stringend
                je      ekey13a
                mov     ah,02h
                mov     bh,video_page
                inc     dl
                int     10h
                inc     si
ekey13a:        jmp     editkey

ekey14:         cmp     ah,79                   ;Process the End key
                jne     ekey15
                mov     cx,stringend
                sub     cx,si
                mov     ah,02h
                mov     bh,video_page
                add     dl,cl
                int     10h
                mov     si,stringend
                jmp     editkey

ekey15:         cmp     ah,82                   ;Process the Ins key
                jne     ekey16
                xor     insflag,1
                mov     cx,small_cursor
                cmp     insflag,0
                je      ekey15a
                mov     cx,big_cursor
ekey15a:        mov     ah,01h
                int     10h
                jmp     editkey

ekey16:         cmp     ah,83                   ;Process the Del key
                jne     ekey17
ekey16a:        mov     cx,stringend
                sub     cx,si
                jcxz    ekey16b
                mov     di,si
                mov     bx,cs
                mov     es,bx
                push    si
                inc     si
                rep     movsb
                pop     si
                push    dx
                push    si
                call    show_string
                mov     byte ptr es:[di],32
                pop     si
                pop     dx
                dec     chars
                dec     stringend
ekey16b:        jmp     editkey

ekey17:         cmp     updnflag,0              ;Loop back if Arrow keys
                je      ekey16b                 ;  are ignored
                cmp     ah,72                   ;Terminate input if Up-
                je      edstr_exit              ;  Arrow was pressed
                cmp     ah,80                   ;Terminate input if Down-
                jne     ekey16b                 ;  Arrow was pressed
;
; Hide the cursor, place the string length in CX, and exit.
;
edstr_exit:     push    ax                      ;Save exit character
                mov     ah,01h                  ;Hide the cursor
                mov     ch,20h
                int     10h
                mov     si,stringstart          ;Compute the length of
                call    strlen                  ;  the string
                pop     ax                      ;Retrieve exit character
                ret
edit_string     endp

;****************************************************************************
; COPYZ copies an ASCIIZ string from one location to another.  On entry,
; DS:SI points to the string and ES:DI points to its new location.
;****************************************************************************

copyz           proc    near
                lodsb                           ;Get a character
                stosb                           ;Copy it to the destination
                or      al,al                   ;Loop back for more if it
                jnz     copyz                   ;  wasn't a zero
                ret
copyz           endp

;****************************************************************************
; INSTRING returns carry clear if the string pointed to by DS:SI is a
; subset of the string pointed to by DS:DI, carry set if it's not.  The
; string comparisons are performed without regard to case.
;****************************************************************************

strlength       dw      ?

instring        proc    near
                push    si                      ;Save substring address
                mov     si,di                   ;Compute the length of the
                call    strlen                  ;  primary string
                mov     dx,cx                   ;Save it in DX
                pop     si                      ;Compute the length of the
                call    strlen                  ;  substring
                mov     strlength,cx            ;Save it
                sub     dx,cx                   ;Subtract it from DX
                inc     dx                      ;Increment DX
                mov     cx,dx                   ;Transfer result to CX
                cmp     cx,1                    ;No match if the result is
                jl      instr6                  ;  less than 1 (signed)

instr1:         push    cx
                push    si
                push    di
                mov     cx,strlength

instr2:         lodsb                           ;Get character from string 1
                cmp     al,"a"                  ;Capitalize it if necessary
                jb      instr3
                cmp     al,"z"
                ja      instr3
                and     al,0DFh
instr3:         mov     ah,[di]                 ;Get character from string 2
                inc     di                      ;Advance DI to next character
                cmp     ah,"a"                  ;Capitalize it if necessary
                jb      instr4
                cmp     ah,"z"
                ja      instr4
                and     ah,0DFh
instr4:         cmp     ah,al
                jne     instr5
                loop    instr2
                jmp     instr7

instr5:         pop     di
                inc     di
                pop     si
                pop     cx
                loop    instr1

instr6:         stc                             ;Set carry and exit
                ret

instr7:         add     sp,6                    ;Clean off the stack
                clc                             ;Clear carry and exit
                ret
instring        endp

;****************************************************************************
; SHOWPAGE displays a page of data with record number RECORDNO in the
; center of the window.  On entry, DS:SI must point to the address of
; record number RECORDNO.
;****************************************************************************

nullrec         db      0,0,0
blanklines      dw      ?
linesback       db      ?
lines           dw      ?

showpage        proc    near
                mov     dh,window_start         ;Set DH to the starting
                inc     dh                      ;  row number
                mov     blanklines,0            ;Initialize variables
                mov     linesback,4
                mov     lines,9
                cmp     records,0               ;Exit if there are no
                je      sp_exit                 ;  records to display

                mov     cx,4                    ;Compute the number of blank
                sub     cx,recordno             ;  lines at the top of the
                cmp     cx,0                    ;  window
                jle     sp2                     ;Branch if it's less than 0

                sub     linesback,cl            ;Adjust LINESBACK
                sub     lines,cx                ;Adjust LINES
                push    si                      ;Save address in SI
sp1:            push    cx                      ;Save CX and DX
                push    dx
                mov     si,offset nullrec       ;Point SI to null record
                call    showline                ;Blank one line
                pop     dx                      ;Retrieve DX
                inc     dh                      ;Increment line number
                pop     cx                      ;Retrieve count in CX
                loop    sp1                     ;Loop until done
                pop     di                      ;Restore SI into DI

sp2:            mov     al,linesback            ;Compute the address of the
                mov     di,bufferptr            ;  first record to be
                call    findlast                ;  displayed
                mov     si,di

                mov     cx,recordno             ;Compute the number of blank
                add     cx,5                    ;  lines at the bottom of
                sub     cx,records              ;  the window
                cmp     cx,0                    ;Branch if the result is
                jle     sp3                     ;  less than 0
                mov     blanklines,cx           ;Save it
                sub     lines,cx                ;Adjust LINES

sp3:            mov     cx,lines                ;Place line count in CX
sp4:            push    cx                      ;Save CX and DX
                push    dx
                call    showline                ;Display one line
                pop     dx                      ;Retrieve DX
                inc     dh                      ;Increment line number
                pop     cx                      ;Retrieve count in CX
                loop    sp4                     ;Loop until done

                mov     cx,blanklines           ;Retrieve blank line count
                jcxz    sp_exit                 ;Exit if zero
sp5:            push    cx                      ;Save CX and DX
                push    dx
                mov     si,offset nullrec       ;Point SI to null record
                call    showline                ;Display one blank line
                pop     dx                      ;Retrieve DX
                inc     dh                      ;Increment line number
                pop     cx                      ;Retrieve count in CX
                loop    sp5                     ;Loop until done
sp_exit:        ret
showpage        endp

;****************************************************************************
; SHOWLINE displays a line of data.  On entry, DS:SI points to the record
; and DH identifies the line on which it will be displayed.
;****************************************************************************

showline        proc    near
                mov     dl,2                    ;Place column number in DL
                call    compute_address         ;Compute the address
                mov     di,ax                   ;Transfer it to DI
                mov     es,video_segment        ;Point ES to video
                mov     cx,26                   ;Display the name field
                call    showfield
                add     di,6                    ;Advance DI
                mov     cx,26                   ;Display the company field
                call    showfield
                add     di,6                    ;Advance DI
                mov     cx,18                   ;Display the phone field
                call    showfield
                ret
showline        endp

;****************************************************************************
; SHOWFIELD displays one zero-delimited field.  On entry, ES:DI points
; to the address in the video buffer, DS:SI points to the data, and CX
; holds the length of the field.  If the length of the text string is
; less than the length of the field, the extra character cells are
; filled with spaces.  On exit, SI points to the start of the next field.
;****************************************************************************

showfield       proc    near
                lodsb                           ;Get a character
                or      al,al                   ;Exit if it's zero
                jz      sf2
                stosb                           ;Display it
                inc     di                      ;Skip attribute byte
                loop    showfield               ;Loop if CX isn't zero
sf1:            lodsb                           ;Get a character
                or      al,al                   ;Loop back it's not zero
                jnz     sf1
sf2:            jcxz    sf_exit                 ;Exit if CX is zero
                mov     al,20h                  ;Place space in AL
sf3:            stosb                           ;Write one space
                inc     di                      ;Skip attribute byte
                loop    sf3                     ;Loop until done
sf_exit:        ret
showfield       endp

;****************************************************************************
; FINDREC returns in DI the address of the record whose number is passed
; in AX.
;****************************************************************************

three           db      3,0

findrec         proc    near
                mov     di,offset databuffer    ;Point DI to the data
                mov     bx,cs                   ;Point ES to the segment
                mov     es,bx                   ;  where the data is stored
                mul     word ptr three          ;Multiply AX by 3
                mov     cx,ax                   ;Transfer result to CX
                sub     al,al                   ;Zero AL
fr1:            push    cx                      ;Save count
                mov     cx,255                  ;Initialize CX
                repne   scasb                   ;Search for the next 0
                pop     cx                      ;Retrieve count
                loop    fr1                     ;Loop until done
                ret
findrec         endp

;****************************************************************************
; FINDLAST computes the address of a record relative to a specified record
; whose address is passed in DI.  On entry, the value of AL specifies the
; relative position of the desired record.  On exit, DI holds its address.
;****************************************************************************

findlast        proc    near
                mul     three                   ;Multiply AL by 3
                mov     cx,ax                   ;Transfer result to CX
                inc     cx                      ;Increment it
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                dec     di                      ;Decrement DI
                sub     al,al                   ;Zero AL
                std                             ;Set direction flag
fl1:            push    cx                      ;Save counter
                mov     cx,255                  ;Reset CX
                repne   scasb                   ;Search for a 0
                pop     cx                      ;Retrieve counter
                loop    fl1                     ;Loop until done
                add     di,2                    ;Add 2 for record address
                cld                             ;Clear direction flag
                ret                             ;Return to caller
findlast        endp

;****************************************************************************
; FINDNEXT computes the address of a record relative to a specified record
; whose address is passed in DI.  On entry, the value of AL specifies the
; relative position of the desired record.  On exit, DI holds its address.
;****************************************************************************

findnext        proc    near
                mul     three                   ;Multiply AL by 3
                mov     cx,ax                   ;Transfer result to CX
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                sub     al,al                   ;Zero AL
fn1:            push    cx                      ;Save counter
                mov     cx,255                  ;Reset CX
                repne   scasb                   ;Search for a 0
                pop     cx                      ;Retrieve counter
                loop    fn1                     ;Loop until done
                ret                             ;Return to caller
findnext        endp

;****************************************************************************
; QSEARCH displays the next record that starts with the ASCII code in AL.
;****************************************************************************

qsearch         proc    near
                mov     ah,al                   ;Transfer code to AH
                or      ah,20h                  ;Convert it to lowercase
                mov     bx,recordno             ;Initialize BX and DI for
                mov     di,bufferptr            ;  the search
                mov     cx,cs                   ;Points ES to the segment
                mov     es,cx                   ;  where data is stored

qs1:            push    ax                      ;Save ASCII codes
                call    nextrec                 ;Advance to the next record
                pop     ax                      ;Retrieve ASCII codes
                cmp     bx,recordno             ;Exit if we've reached the
                je      qsearch_exit            ;  current record
                cmp     [di],al                 ;Branch if the name field
                je      qs2                     ;  starts with the code in
                cmp     [di],ah                 ;  AH or AL
                je      qs2
                jmp     qs1                     ;Loop back if it doesn't

qs2:            mov     recordno,bx             ;Store the record number
                mov     bufferptr,di            ;Store the address also
                mov     si,di                   ;Transfer address to SI
                call    showpage                ;Display the record
qsearch_exit:   ret
qsearch         endp

;****************************************************************************
; NEXTREC accepts a record number in BX and the corresponding record
; address in DI, then returns the next record number and its address in
; the same registers.  On entry, ES must point to the segment where data
; is stored.
;****************************************************************************

nextrec         proc    near
                inc     bx                      ;Increment record number
                cmp     bx,records              ;Branch if it's equal to
                jae     reset                   ;  the number of records

                sub     al,al                   ;Zero AL
                mov     cx,255                  ;Initialize CX
                repne   scasb                   ;Advance to the next zero...
                repne   scasb                   ;And the next...
                repne   scasb                   ;And the next
                ret                             ;Return to caller

reset:          sub     bx,bx                   ;Set record number to 0
                mov     di,offset databuffer    ;Point DI to the base
                ret                             ;Return to caller
nextrec         endp

;****************************************************************************
; SETUP lets the program be configured interactively.
;****************************************************************************

nine            db      9
changed         db      ?
exitcode        dw      ?
spaces          db      27 dup (32),0

setup           proc    near
                mov     changed,0               ;Initialize changed flag
                mov     bx,color_ptr            ;Place attributes in AH,
                mov     ah,border_color         ;  AL, and BH
                mov     al,window_color_2
                mov     bh,shadow_color
                mov     ch,window_start         ;Load CX with the window's
                sub     ch,4                    ;  starting coordinates
                mov     cl,17
                mov     dh,ch                   ;Load DX with the window's
                add     dh,9                    ;  ending coordinates
                mov     dl,62
                push    cx                      ;Save window coordinates
                push    dx
                push    cs                      ;Point ES:DI to the buffer
                pop     es                      ;  for screen data
                mov     di,offset screenbuffer2
                call    openwindow              ;Display the window

                mov     dh,window_start         ;Compute row and column
                sub     dh,3                    ;  address to begin
                mov     dl,19                   ;  writes
                push    dx                      ;Save row and column
                mov     si,offset config1       ;Point DS:SI to the text
                call    show_string             ;Display "COM Port" string
                pop     dx                      ;Retrieve row and column

                inc     dh                      ;Display "COM Port Address"
                push    dx
                mov     si,offset config2
                call    show_string
                pop     dx

                inc     dh                      ;Display "Communications
                push    dx                      ;  Setting"
                mov     si,offset config3
                call    show_string
                pop     dx

                inc     dh                      ;Display "Initialization
                push    dx                      ;  Command"
                mov     si,offset config4
                call    show_string
                pop     dx

                inc     dh                      ;Display "Dial Command"
                push    dx
                mov     si,offset config5
                call    show_string
                pop     dx

                inc     dh                      ;Display "Dial Command
                push    dx                      ;  Suffix"
                mov     si,offset config6
                call    show_string
                pop     dx

                inc     dh                      ;Display "Disconnect Command"
                push    dx
                mov     si,offset config7
                call    show_string
                pop     dx

                inc     dh                      ;Display "Dialing Prefix"
                mov     si,offset config8
                call    show_string

                mov     dh,window_start         ;Display the COM port
                sub     dh,3                    ;  number
                mov     dl,45
                push    dx
                mov     si,offset comstring
                call    show_string
                mov     al,comport
                add     al,31h
                stosb
                pop     dx

                inc     dh                      ;Display COM port address
                push    dx
                mov     al,comport
                cbw
                shl     ax,1
                shl     ax,1
                add     ax,offset addr_table
                mov     si,ax
                call    show_string
                pop     dx

                inc     dh                      ;Display communications
                push    dx                      ;  setting
                mov     al,commptr
                mul     nine
                add     ax,offset comm_table
                mov     si,ax
                call    show_string
                pop     dx

                inc     dh                      ;Display initialization
                push    dx                      ;  command
                mov     si,offset initcmd
                call    show_string
                pop     dx

                inc     dh                      ;Display dial command
                push    dx
                mov     si,offset dialcmd
                call    show_string
                pop     dx

                inc     dh                      ;Display dial command
                push    dx                      ;  suffix
                mov     si,offset suffix
                call    show_string
                pop     dx

                inc     dh                      ;Display disconnect
                push    dx                      ;  command
                mov     si,offset hangcmd
                call    show_string
                pop     dx

                inc     dh                      ;Display dialing prefix
                push    dx
                mov     si,offset prefix
                call    show_string
                pop     dx
;
; Take input from the keyboard.
;
setup1:         mov     bx,color_ptr            ;Draw the selection bar at
                mov     al,hilite_color_2       ;  the location addressed
                mov     cx,44                   ;  by SETUP_INDEX
                mov     dh,window_start
                add     dh,setup_index
                sub     dh,3
                mov     dl,18
                call    write_attr

setup2:         call    readkey                 ;Get a keystroke
                or      al,al                   ;Branch if it's an extended
                jnz     setup2z                 ;  key code
                jmp     setup3
setup2z:        cmp     al,27                   ;Exit if Esc was pressed
                jne     setup2a
                jmp     setup_exit
setup2a:        cmp     al,9                    ;Branch if Tab, Enter, or
                je      setup2b                 ;  spacebar was pressed
                cmp     al,13
                je      setup2b
                cmp     al,32
                jne     setup2                  ;Ignore anything else

setup2b:        cmp     setup_index,0           ;Branch if SETUP_INDEX is
                je      setup2c                 ;  either 0 or 2
                cmp     setup_index,2
                jne     setup2d
setup2c:        jmp     setup6a

setup2d:        cmp     setup_index,1           ;Branch if SETUP_INDEX is
                je      get_address             ;  other than 1
                jmp     get_string
;
; Input a COM port address.
;
get_address:    mov     al,comport              ;Compute the address of the
                cbw                             ;  string that denotes the
                shl     ax,1                    ;  COM port address
                shl     ax,1
                add     ax,offset addr_table
                push    ax                      ;Save it
                mov     si,ax                   ;Transfer it to AX
                mov     di,offset inbuffer      ;Point DI to input buffer
                mov     bx,cs                   ;Point ES to this segment
                mov     es,bx
                call    copyz                   ;Copy string to input buffer

getaddr1:       mov     al,01h                  ;Prepare for call to
                mov     bx,01h                  ;  EDIT_STRING
                mov     cx,3
                mov     dh,window_start
                sub     dh,2
                mov     dl,45
                mov     si,offset inbuffer
                call    edit_string             ;Let the string be edited
                mov     exitcode,ax             ;Save the exit code

                cmp     al,27                   ;Branch if ESC was not
                jne     getaddr2                ;  pressed
                mov     dh,window_start         ;Redisplay the old COM
                sub     dh,2                    ;  port address and exit
                mov     dl,45
                push    dx
                mov     si,offset spaces+24
                call    show_string
                pop     dx
                pop     si
                call    show_string
                jmp     setup2

getaddr2:       mov     changed,1               ;Set CHANGED flag
                mov     si,offset inbuffer      ;Convert the address just
                call    hex2bin                 ;  entered to binary
                jnc     getaddr3                ;Branch if call succeeded

                mov     ah,01h                  ;Turn off the cursor
                mov     ch,20h
                int     10h
                mov     si,offset errtxt16      ;Point SI to error message
                mov     di,offset screenbuffer3 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
                jmp     getaddr1                ;Reenter the input loop

getaddr3:       pop     di                      ;Retrieve string address
                push    di                      ;Save it on the stack again
                mov     bx,cs                   ;Point ES to this segment
                mov     es,bx           
                call    bin2hex                 ;Convert binary to ASCIIZ
                mov     dh,window_start         ;Blank the string currently
                sub     dh,2                    ;  displayed
                mov     dl,45
                push    dx
                mov     si,offset spaces+24
                call    show_string
                pop     dx                      ;Then redisplay the ASCII
                pop     si                      ;  equivalent of the number
                call    show_string             ;  just entered and exit
                cmp     exitcode,4800h
                je      getstr2
                cmp     exitcode,5000h
                je      getstr3
                jmp     setup2
;
; Input a modem command string.
;
get_string:     mov     bl,setup_index          ;Compute offset address of
                sub     bl,3                    ;  the string to be edited
                sub     bh,bh
                shl     bx,1
                mov     si,[bx+offset text_table]
                push    si                      ;Save it
                mov     di,offset inbuffer      ;Point DI to input buffer
                mov     bx,cs                   ;Point ES to this segment
                mov     es,bx
                call    copyz                   ;Copy string to input buffer

                mov     al,01h                  ;Prepare for call to
                mov     bx,01h                  ;  EDIT_STRING
                mov     cx,16
                mov     dh,window_start
                sub     dh,3
                add     dh,setup_index
                mov     dl,45
                mov     si,offset inbuffer
                call    edit_string             ;Let the string be edited

                pop     di                      ;Retrieve its address in DI
                cmp     al,27                   ;Branch if the Esc key was
                jne     getstr1                 ;  not pressed
                mov     dh,window_start         ;Otherwise redisplay the
                sub     dh,3                    ;  original string and
                add     dh,setup_index          ;  return to the input
                mov     dl,45                   ;  loop
                push    dx
                push    di
                mov     si,offset spaces+11
                call    show_string
                pop     si
                pop     dx
                call    show_string
                jmp     setup2

getstr1:        mov     changed,1               ;Set CHANGED flag
                mov     bx,ax                   ;Save exit code in BX
                mov     si,offset inbuffer      ;Copy the new string to the
                mov     ax,cs                   ;  original string address
                mov     es,ax
                call    copyz
                cmp     bx,4800h                ;Branch if input was termin-
getstr2:        je      setup3a                 ;  ated with Up-Arrow
                cmp     bx,5000h                ;Also branch if it was term-
getstr3:        je      setup4a                 ;  inated with Down-Arrow
                jmp     setup2                  ;Return to input loop
;
; Process the Up- and Down-Arrow keys.
;
setup3:         cmp     ah,72                   ;Process the Up-Arrow key
                jne     setup4
setup3a:        mov     bx,color_ptr            ;Erase the selection bar
                mov     al,window_color_2
                mov     cx,44
                mov     dh,window_start
                add     dh,setup_index
                sub     dh,3
                mov     dl,18
                call    write_attr
                dec     setup_index             ;Decrement index
                cmp     setup_index,0FFFFh
                jne     setup3b
                mov     setup_index,7
setup3b:        jmp     setup1

setup4:         cmp     ah,80                   ;Process the Down-Arrow key
                jne     setup5
setup4a:        mov     bx,color_ptr            ;Erase the selection bar
                mov     al,window_color_2
                mov     cx,44
                mov     dh,window_start
                add     dh,setup_index
                sub     dh,3
                mov     dl,18
                call    write_attr
                inc     setup_index             ;Increment index
                cmp     setup_index,8
                jne     setup4b
                mov     setup_index,0
setup4b:        jmp     setup1
;
; Process the Left- and Right-Arrow keys.
;
setup5:         cmp     ah,75                   ;Process the Left-Arrow key
                jne     setup6
                mov     dh,window_start         ;Initialize DX with row and
                sub     dh,3                    ;  column number
                mov     dl,45
                cmp     setup_index,0           ;Branch if SETUP_INDEX is
                jne     setup5a                 ;  other than 0
                mov     changed,1               ;Set CHANGED flag
                dec     comport                 ;Decrement COM port number
                cmp     comport,0FFFFh
                jne     setup6b
                mov     comport,3
                jmp     short setup6b           ;Go display the setting

setup5a:        cmp     setup_index,2           ;Branch if SETUP_INDEX is
                jne     setup6c                 ;  other than 2
                mov     changed,1               ;Set CHANGED flag
                dec     commptr                 ;Decrement the pointer to
                cmp     commptr,0FFh            ;  communications settings
                jne     setup6e
                mov     commptr,9
                jmp     short setup6e           ;Go display the new setting

setup6:         cmp     ah,77                   ;Process the Right-Arrow key
                jne     setup6c
setup6a:        mov     dh,window_start         ;Initialize DX with row and
                sub     dh,3                    ;  column number
                mov     dl,45
                cmp     setup_index,0           ;Branch if SETUP_INDEX is
                jne     setup6d                 ;  other than 0
                mov     changed,1               ;Set CHANGED flag
                inc     comport                 ;Increment the COM port
                cmp     comport,4               ;  number
                jne     setup6b
                mov     comport,0
setup6b:        push    dx                      ;Display the COM port number
                add     dl,3
                mov     al,comport
                add     al,31h
                call    show_char
                pop     dx
                inc     dh                      ;Display COM port address
                push    dx
                mov     si,offset spaces+24
                call    show_string
                pop     dx
                mov     al,comport
                cbw
                shl     ax,1
                shl     ax,1
                add     ax,offset addr_table
                mov     si,ax
                call    show_string
setup6c:        jmp     setup2                  ;Return to input loop

setup6d:        cmp     setup_index,2           ;Branch if SETUP_INDEX is
                jne     setup6f                 ;  other than 2
                mov     changed,1               ;Set CHANGED flag
                inc     commptr                 ;Increment the pointer to
                cmp     commptr,10              ;  communications settings
                jne     setup6e
                mov     commptr,0
setup6e:        mov     al,commptr              ;Display the new settings
                mul     nine
                add     ax,offset comm_table
                mov     si,ax
                add     dh,2
                call    show_string
setup6f:        jmp     setup2                  ;Return to input loop
;
; Save the changes, close the window, and exit.
;
setup_exit:     cmp     changed,0               ;Exit now if there were
                jne     setup9                  ;  no changes
                jmp     setup_close

setup9:         mov     bx,color_ptr            ;Place attributes in AH,
                mov     ah,border_color         ;  AL, and BH
                mov     al,window_color_2
                mov     bh,shadow_color
                mov     ch,window_start         ;Load CX with the window's
                add     ch,4                    ;  starting coordinates
                mov     cl,5
                mov     dh,ch                   ;Load DX with the window's
                add     dh,2                    ;  ending coordinates
                mov     dl,74
                push    cx                      ;Save window coordinates
                push    dx
                push    cs                      ;Point ES:DI to the buffer
                pop     es                      ;  for screen data
                mov     di,offset screenbuffer3
                call    openwindow              ;Display the window

                mov     dh,window_start         ;Compute row and column
                add     dh,5                    ;  address for message
                mov     dl,26
                mov     si,offset keep_prompt   ;Prompt "Save changes to
                call    show_string             ;  disk (Y/N)?"

setup10:        call    readkey                 ;Get an answer to the
                or      al,al                   ;  question
                jz      setup10                 ;Loop back on extended key
                cmp     al,27                   ;Branch if the Esc key was
                jne     setup10a                ;  not pressed
                pop     dx                      ;Close the prompt window
                pop     cx                      ;  by restoring what was
                add     dx,0101h                ;  under it
                mov     si,offset screenbuffer3
                call    restregion
                jmp     setup2                  ;Return to setup window
setup10a:       and     al,0DFh                 ;Capitalize the response
                cmp     al,"Y"                  ;Continue if it's "Y"
                je      setup11
                cmp     al,"N"                  ;Exit if it's "N" or loop
                jne     setup10                 ;  back if it's not
                jmp     setup17

setup11:        mov     dh,window_start         ;Compute row and column
                add     dh,5                    ;  address
                mov     dl,26
                mov     si,offset spaces        ;Erase the prompt from the
                call    show_string             ;  window
                mov     dh,window_start
                add     dh,5
                mov     dl,7
                push    dx
                mov     si,offset setup_prompt  ;Display "Save To:" prompt
                call    show_string

                mov     si,offset progfile      ;Point SI to the file name
                call    strlen                  ;Compute the string length
                cmp     cx,57                   ;Branch if it's <= 57
                jna     setup12
                mov     progfile,0              ;Erase it if it's not
setup12:        mov     di,offset inbuffer      ;Point DI to input buffer
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                call    copyz                   ;Copy file name to buffer

                pop     dx                      ;Display the file name
                add     dl,9                    ;  used the last time
                push    dx                      ;  changes were saved
                mov     si,offset progfile
                call    show_string
                pop     dx                      ;Prepare for call to
                mov     al,01h                  ;  EDIT_STRING
                mov     bx,00h
                mov     cx,57
                mov     si,offset inbuffer
                call    edit_string             ;Input the file name
                mov     exitcode,ax             ;Save the exit code
                pop     dx                      ;Close the prompt window
                pop     cx                      ;  by restoring what was
                add     dx,0101h                ;  under it
                mov     si,offset screenbuffer3
                call    restregion
                cmp     byte ptr exitcode,27    ;Loop back if input was
                je      setup_return            ;  ended with the Esc key

                mov     si,offset inbuffer      ;Convert the file name
                mov     di,offset progfile      ;  into a fully qualified
                mov     ax,cs                   ;  file name
                mov     es,ax
                mov     ah,60h
                int     21h
                mov     si,offset errtxt9       ;Branch if an error
                jc      setup_err               ;  occurred

                call    savesetup               ;Save the configuration
                jnc     setup_close             ;  changes

                mov     si,offset errtxt17      ;Point SI to error message
                cmp     al,01H
                je      setup_err
                mov     si,offset errtxt18
setup_err:      mov     di,offset screenbuffer3 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
setup_return:   jmp     setup2                  ;Go back for more

setup_close:    pop     dx                      ;Close the window by
                pop     cx                      ;  restoring what was
                add     dx,0101h                ;  under it
                mov     si,offset screenbuffer2
                call    restregion
setup_end:      ret

setup17:        pop     dx                      ;Close the prompt window
                pop     cx                      ;  by restoring what was
                add     dx,0101h                ;  under it
                mov     si,offset screenbuffer3
                call    restregion
                jmp     setup_close
setup           endp

;****************************************************************************
; SEARCH searches for the first record containing a specified string.
;****************************************************************************

recaddress      dw      ?
searchcount     dw      ?

search          proc    near
                mov     si,offset searchtext    ;Point SI to search string
                mov     di,offset inbuffer      ;Point DI to input buffer
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                call    copyz                   ;Copy the search string

                mov     ch,26                   ;Prepare for call to
                mov     cl,41                   ;  PROMPT_WINDOW
                mov     si,offset search_prompt
                mov     di,offset inbuffer
                call    prompt_window           ;Input the search string

                cmp     al,13                   ;Exit if input wasn't ended
                jne     setup_end               ;  with the Enter key
                mov     si,offset inbuffer      ;Point SI to input buffer
                mov     di,offset searchtext    ;Point DI to search string
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                call    copyz                   ;Copy the search string

                mov     si,offset searchtext    ;Point SI to search string
                call    strlen                  ;Compute its length in CX
                or      cx,cx                   ;Exit if there is no
                jz      search_exit             ;  search string
                cmp     records,0               ;Exit if there are no
                je      search_exit             ;  records to search

search1:        mov     bx,recordno             ;Initialize BX and DI
                mov     di,bufferptr
                mov     cx,records              ;Initialize counter
                mov     searchcount,cx

search2:        call    nextrec                 ;Advance to the next record
                mov     recaddress,di           ;Save the record address
                mov     si,offset searchtext    ;Point SI to search string
                push    di
                call    instring                ;Compare the search string to
                pop     di                      ;  the Name field and branch
                jnc     search3                 ;  if there's a match
                sub     al,al                   ;Find the beginning of the
                mov     cx,255                  ;  Company field
                repne   scasb
                mov     si,offset searchtext    ;Point SI to search string
                push    di
                call    instring                ;Search the field
                pop     di
                jnc     search3                 ;Branch if there's a match
                sub     al,al                   ;Find the beginning of the
                mov     cx,255                  ;  Phone field
                repne   scasb
                mov     si,offset searchtext    ;Point SI to search string
                push    di
                call    instring                ;Search the field
                pop     di
                jnc     search3                 ;Branch if there's a match

                dec     searchcount             ;Decrement counter
                jz      search4                 ;Exit if it has reached zero
                mov     di,recaddress           ;Retrieve record address
                jmp     search2                 ;Continue the search

search3:        mov     recordno,bx             ;Store the record number
                mov     di,recaddress           ;Retrieve the record address
                mov     bufferptr,di            ;Store the address also
                mov     si,di                   ;Transfer address to SI
                call    showpage                ;Display the record
                ret

search4:        mov     si,offset errtxt1       ;Point SI to error message
                mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
search_exit:    ret
search          endp

;****************************************************************************
; NEXT searches for the next record containing a specified string.
;****************************************************************************

next            proc    near
                cmp     records,0               ;Exit if there are no
                je      next_exit               ;  records to search
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                cmp     searchtext,0            ;Branch if the search string
                je      next1                   ;  isn't null
                jmp     search1
next1:          mov     bx,recordno             ;Place record number if BX
                mov     di,bufferptr            ;Place record address in DI
                call    nextrec                 ;Find the next record
                mov     bufferptr,di            ;Save its address
                mov     recordno,bx             ;Save its number
                mov     si,di                   ;Transfer address to SI
                call    showpage                ;Display the next record
next_exit:      ret
next            endp

;****************************************************************************
; INSERT inserts a new (blank) record at the current position.
;****************************************************************************

insert          proc    near
                mov     ax,recordno             ;Exit if we're just past
                cmp     ax,records              ;  the final record
                je      ins_exit

                mov     ax,bufferlimit          ;Compute the number of bytes
                sub     ax,buffertop            ;  left in the buffer
                cmp     ax,3                    ;Branch if it's more than 3
                ja      insert1
                mov     si,offset errtxt2       ;Point SI to error message
                mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
                ret                             ;  and exit

insert1:        mov     si,buffertop            ;Point SI to data to move
                dec     si
                mov     di,si                   ;Point DI to destination
                add     di,3                    ;  (three bytes higher)
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                mov     cx,buffertop            ;Compute number of bytes
                sub     cx,bufferptr            ;  to move in CX
                std                             ;Set direction flag
                rep     movsb                   ;Perform the move
                cld                             ;Clear direction flag
                add     buffertop,3             ;Adjust BUFFERTOP
                inc     records                 ;Increment record count
                mov     si,bufferptr            ;Load SI with BUFFERPTR
                mov     byte ptr [si],0         ;Insert three zeroes
                mov     byte ptr [si+1],0
                mov     byte ptr [si+2],0
                call    showpage                ;Display the page
ins_exit:       ret
insert          endp

;****************************************************************************
; PASTE inserts the most recently deleted record.
;****************************************************************************

paste           proc    near
                cmp     recordlength,0          ;Branch if there is a
                jne     paste1                  ;  record to paste
                mov     si,offset errtxt3       ;Point SI to error message
                mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
                ret                             ;  and exit

paste1:         mov     cx,bufferlimit          ;Compute the number of bytes
                sub     cx,buffertop            ;  left in the buffer
                cmp     cx,recordlength         ;Branch if it's more than the
                ja      paste2                  ;  length of the paste record
                mov     si,offset errtxt4       ;Point SI to error message
                mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
                ret                             ;  and exit

paste2:         mov     si,buffertop            ;Point SI to data to move
                dec     si
                mov     di,si                   ;Point DI to destination
                add     di,recordlength
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                mov     cx,buffertop            ;Compute number of bytes
                sub     cx,bufferptr            ;  to move in CX
                std                             ;Set direction flag
                rep     movsb                   ;Perform the move
                cld                             ;Clear direction flag
                mov     ax,recordlength         ;Get record length in AX
                add     buffertop,ax            ;Adjust BUFFERTOP
                inc     records                 ;Increment record count
                mov     si,offset del_record    ;Point SI to deleted record
                mov     di,bufferptr            ;Point DI to current record
                mov     cx,recordlength         ;Load CX with record length
                rep     movsb                   ;Insert the deleted record
                mov     si,bufferptr            ;Point SI to current record
                call    showpage                ;Display the page
paste_exit:     ret
paste           endp

;****************************************************************************
; DELETE deletes the current record.
;****************************************************************************

delete          proc    near
                cmp     records,0               ;Exit if there are no
                je      del_exit                ;  records
                mov     ax,recordno             ;Exit if we're just past
                cmp     ax,records              ;  the final record
                je      del_exit

                mov     di,bufferptr            ;Point DI to current record
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                sub     al,al                   ;Zero AL
                mov     cx,255                  ;Initialize CX
                repne   scasb                   ;Skip past the next
                repne   scasb                   ;  three zeroes
                repne   scasb
                push    di                      ;Save next record address
                mov     si,bufferptr            ;Point SI to current record
                mov     cx,di                   ;Compute length of record to
                sub     cx,si                   ;  delete in CX
                mov     recordlength,cx         ;Save the result
                mov     di,offset del_record    ;Point DI to record buffer
                rep     movsb                   ;Save the deleted record
                pop     si                      ;Retrieve next record address
                mov     di,bufferptr            ;Point DI to current record
                mov     cx,buffertop            ;Compute number of bytes to
                sub     cx,si                   ;  move in CX
                rep     movsb                   ;Delete the current record
                mov     cx,recordlength         ;Subtract record length
                sub     buffertop,cx            ;  from BUFFERTOP
                dec     records                 ;Decrement record count
                jz      delete1                 ;Branch if RECORDS is 0
                mov     si,bufferptr            ;Point SI to current record
                call    showpage                ;Display the page
                ret

delete1:        mov     dh,window_start         ;Load row number in DH
                add     dh,5
                mov     si,offset nullrec       ;Point SI to null record
                call    showline                ;Delete the current line
del_exit:       ret
delete          endp

;****************************************************************************
; SAVE saves the current data to disk.
;****************************************************************************

save            proc    near
                cmp     records,0               ;Branch if RECORDS is not
                jne     save1                   ;  equal to 0
                mov     si,offset errtxt5       ;Point SI to error message
                mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
                ret                             ;  and exit

save1:          mov     si,offset filename      ;Point SI to file name
                call    strlen                  ;Compute the string length
                cmp     cx,57                   ;Branch if it's less than or
                jna     save2                   ;  equal to 57
                mov     filename,0              ;Erase it if it's longer
save2:          mov     di,offset inbuffer      ;Point DI to input buffer
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                call    copyz                   ;Copy the file name

                mov     ch,57                   ;Prepare for call to
                mov     cl,69                   ;  PROMPT_WINDOW
                mov     si,offset save_prompt
                mov     di,offset inbuffer
                call    prompt_window           ;Input the file name

                cmp     al,13                   ;Exit if input wasn't ended
                jne     save_exit               ;  with the Enter key
                mov     ah,60h
                mov     si,offset inbuffer      ;Point SI to input buffer
                mov     di,offset filename      ;Point DI to file name
                mov     bx,cs                   ;Point ES to this segment
                mov     es,bx
                int     21h                     ;Make fully qualified name
                mov     si,offset errtxt6       ;Error if the call failed
                jc      save4

                call    savefile                ;Save the data
                jnc     save_exit               ;Branch if call succeeded

                mov     si,offset errtxt6       ;Point SI to error message
                cmp     al,01H
                je      save4
                mov     si,offset errtxt7
                cmp     al,02H
                je      save4
                mov     si,offset errtxt8
save4:          mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
save_exit:      ret
save            endp

;****************************************************************************
; OPEN opens a new data file.
;****************************************************************************

open            proc    near
                mov     si,offset filename      ;Point SI to file name
                call    strlen                  ;Compute the string length
                cmp     cx,57                   ;Branch if it's less than or
                jna     open1                   ;  equal to 57
                mov     filename,0              ;Erase it if it's longer
open1:          mov     di,offset inbuffer      ;Point DI to input buffer
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                call    copyz                   ;Copy the file name

                mov     ch,57                   ;Prepare for call to
                mov     cl,69                   ;  PROMPT_WINDOW
                mov     si,offset open_prompt
                mov     di,offset inbuffer
                call    prompt_window           ;Input the file name

                cmp     al,13                   ;Exit if input wasn't ended
                jne     open_exit               ;  with the Enter key
                mov     ah,60h
                mov     si,offset inbuffer      ;Point SI to input buffer
                mov     di,offset filename      ;Point DI to file name
                mov     bx,cs                   ;Point ES to this segment
                mov     es,bx
                int     21h                     ;Make fully qualified name
                mov     si,offset errtxt9       ;Error if the call failed
                jc      open3

                call    readfile                ;Save the data
                jc      open2                   ;Branch if call succeeded

                mov     recordno,0              ;Display the new file
                mov     si,offset databuffer
                mov     bufferptr,si
                call    showpage
open_exit:      ret

open2:          mov     si,offset errtxt10      ;Point SI to error message
                cmp     al,01H
                je      open3
                mov     si,offset errtxt11
                cmp     al,02H
                je      open3
                mov     si,offset errtxt12
                cmp     al,03h
                je      open3
                mov     si,offset errtxt13
open3:          mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
                ret
open            endp

;****************************************************************************
; EDIT allows the current record to be edited.  On return, AX holds the
; key code for the key that terminated input.
;****************************************************************************

field_addr      dw      offset inbuffer         ;Field addresses
                dw      offset inbuffer+32
                dw      offset inbuffer+64
field_col       dw      2,31,60                 ;Field column numbers
field_len       dw      26,26,18                ;Field lengths
field_num       dw      ?                       ;Field number
rec_length      dw      ?                       ;Record length
next_record     dw      ?                       ;Address of next record

edit            proc    near
                mov     field_num,0             ;Initialize field number
                mov     ax,recordno             ;Branch if the pointer is
                cmp     ax,records              ;  highlighting a record
                jne     edit2

                mov     ax,bufferlimit          ;Compute the number of bytes
                sub     ax,buffertop            ;  left in the buffer in AX
                cmp     ax,73                   ;Error if there's not enough
                ja      edit1                   ;  room to edit the record

edit_error:     mov     si,offset errtxt14      ;Point SI to error message
                mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
                sub     ax,ax                   ;Zero the return code
                ret
;
; Prepare the input buffer to receive input.
;
edit1:          mov     byte ptr inbuffer,0     ;Zero the three bytes where
                mov     byte ptr inbuffer[32],0 ;  editing will begin and
                mov     byte ptr inbuffer[64],0 ;  branch ahead
                jmp     short edit3

edit2:          mov     si,bufferptr            ;Point SI to current record
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                call    strlen                  ;Compute the length of the
                mov     rec_length,cx           ;  string in the Name field
                mov     di,offset inbuffer      ;Copy the string to the
                call    copyz                   ;  input buffer
                call    strlen                  ;Compute the length of the
                add     rec_length,cx           ;  string in the next field
                mov     di,offset inbuffer+32   ;Copy the string to the
                call    copyz                   ;  input buffer
                call    strlen                  ;Compute the length of the
                add     rec_length,cx           ;  string in the Phone field
                add     rec_length,3            ;Add 3 to account for zeroes
                mov     di,offset inbuffer+64   ;Copy the string to the
                call    copyz                   ;  input buffer
                mov     next_record,si          ;Save the next record address
                mov     cx,73                   ;Subtract the record size
                sub     cx,rec_length           ;  from 73 in CX
                mov     ax,bufferlimit          ;Compute the number of bytes
                sub     ax,buffertop            ;  left in the buffer in AX
                cmp     ax,cx                   ;Error if there's not enough
                jb      edit_error              ;  room to edit the record
;
; Let the record be edited.
;
edit3:          mov     bx,field_num            ;Get the field number in BX
                mov     si,offset field_len     ;Get the length of the field
                mov     cx,[si+bx]              ;  in CX
                mov     dh,window_start         ;Get the starting row number
                add     dh,5                    ;  in DH
                mov     si,offset field_col     ;Get the starting column
                mov     dl,[si+bx]              ;  number in DL
                mov     si,offset field_addr    ;Get the field address in SI
                mov     si,[si+bx]
                mov     al,00h
                mov     bx,0101h
                call    edit_string             ;Edit the field

                mov     bp,ax                   ;Save the exit code
                cmp     al,13                   ;Exit if Enter was pressed
                je      edit5
                cmp     al,27                   ;Exit if Esc was pressed
                je      edit5a
                cmp     ax,4800h                ;Exit if Up-Arrow was
                je      edit5                   ;  pressed
                cmp     ax,5000h                ;Exit if Down-Arrow was
                je      edit5                   ;  pressed

                cmp     al,9                    ;Branch if Tab was not
                jne     edit4                   ;  pressed
                add     field_num,2             ;Move the cursor to the
                cmp     field_num,6             ;  next field if input was
                jne     edit3                   ;  terminated with the Tab
                mov     field_num,0             ;  key
                jmp     edit3
edit4:          sub     field_num,2             ;Move the cursor to the
                cmp     field_num,0             ;  previous field if input
                jnl     edit3                   ;  was terminated with
                mov     field_num,4             ;  Shift-Tab
                jmp     edit3
;
; Move the records above the new one up or down in memory.
;
edit5:          mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                mov     si,offset inbuffer      ;Point SI to first field
                call    strlen                  ;Compute the string length
                mov     bx,cx                   ;Transfer it to BX
                mov     si,offset inbuffer+32   ;Point SI to second field
                call    strlen                  ;Compute the string length
                add     bx,cx                   ;Add it to BX
                mov     si,offset inbuffer+64   ;Point SI to third field
                call    strlen                  ;Compute the string length
                add     bx,cx                   ;Add it to BX
                add     bx,3                    ;Add 3 for record length
                mov     cx,bx                   ;Transfer the result to CX
                mov     ax,recordno             ;Branch if this record will
                cmp     ax,records              ;  not be appended to the end
                jne     edit6

                cmp     cx,3                    ;Exit if the input buffer
                ja      edit5b                  ;  is empty
edit5a:         jmp     short edit_exit
edit5b:         inc     records                 ;Increment record count
                add     buffertop,cx            ;Adjust BUFFERTOP
                jmp     short edit8             ;Go add the new record

edit6:          cmp     cx,rec_length           ;Branch if the new record is
                jae     edit7                   ;  longer than the old one
                mov     ax,rec_length           ;Compute length difference in
                sub     ax,cx                   ;  AX
                mov     si,next_record          ;Point SI to the data to move
                mov     di,si                   ;Point DI to the new
                sub     di,ax                   ;  location below it
                mov     cx,buffertop            ;Compute the number of bytes
                sub     cx,next_record          ;  to move
                rep     movsb                   ;Move the data
                sub     buffertop,ax            ;Adjust BUFFERPTR
                jmp     short edit8             ;Go add the new record

edit7:          sub     cx,rec_length           ;Compute length difference
                jcxz    edit8                   ;Branch if difference is 0
                mov     si,buffertop            ;Point SI to the top of the
                dec     si                      ;  data in the buffer
                mov     di,si                   ;Point DI to the new
                add     di,cx                   ;  location above it
                mov     ax,cx                   ;Transfer difference to AX
                mov     cx,buffertop            ;Compute the number of bytes
                sub     cx,next_record          ;  to move
                std                             ;Set direction flag
                rep     movsb                   ;Move the data
                cld                             ;Clear direction flag
                add     buffertop,ax            ;Adjust BUFFERTOP
;
; Copy the new record to the data buffer and exit.
;
edit8:          mov     si,offset inbuffer      ;Point SI to first field
                mov     di,bufferptr            ;Point DI to data buffer
                call    copyz                   ;Copy the first field
                mov     si,offset inbuffer+32   ;Point SI to second field
                call    copyz                   ;Copy it
                mov     si,offset inbuffer+64   ;Point SI to third field
                call    copyz                   ;Copy it
edit_exit:      mov     si,bufferptr            ;Point SI to current record
                call    showpage                ;Display the page
                mov     ax,bp                   ;Retrieve the return code
                ret
edit            endp

;****************************************************************************
; DIAL dials the phone number currently displayed.
;****************************************************************************

uart_addr       dw      ?

dial            proc    near
                cmp     records,0               ;Exit if there are no
                jne     dial1                   ;  records
dial_abort:     ret
dial1:          mov     ax,records              ;Also exit if the pointer
                cmp     ax,recordno             ;  is at the end of the
                je      dial_abort              ;  data

                mov     al,comport              ;Get the UART's base
                cbw                             ;  address
                shl     ax,1
                shl     ax,1
                add     ax,offset addr_table
                mov     si,ax
                call    hex2bin
                mov     uart_addr,ax            ;Save it
                mov     dx,ax                   ;Transfer it to DX
;
; Initialize the serial port.
;
                add     dx,3                    ;Point DX to line control
                mov     bl,commptr              ;Retrieve format byte from
                sub     bh,bh                   ;  table
                mov     al,[bx+offset format_data]
                push    ax                      ;Save it
                or      al,80h                  ;Set bit 7 (DLAB)
                out     dx,al                   ;Write it to the UART
                sub     dx,3                    ;Point DX to LSB divisor
                shl     bx,1                    ;Double the table pointer
                mov     al,[bx+offset divisor_data]     ;Output the divisor
                out     dx,al                           ;  LSB to the UART
                inc     dx                      ;Point DX to MSB divisor
                mov     al,[bx+offset divisor_data+1]   ;Output the divisor
                out     dx,al                           ;  MSB to the UART
                pop     ax                      ;Retrieve format byte
                add     dx,2                    ;Point DX to line control
                out     dx,al                   ;Output data to the UART
;
; Make sure there's a serial port at this address.
;
                mov     bl,al                   ;Copy format byte to BL
                mul     bl                      ;Dummy opcode for I/O delay
                in      al,dx                   ;Read line control register
                cmp     al,bl                   ;Are the values the same?
                je      dial2                   ;Yes, then continue
                mov     al,comport              ;Compute address of COM port
                cbw                             ;  address
                shl     ax,1
                shl     ax,1
                add     ax,offset addr_table
                mov     si,ax                   ;Transfer it to SI
                mov     di,offset errtxt19+24   ;Point DI to error message
                mov     ax,cs                   ;Point ES to this segment
                mov     es,ax
                call    copyz                   ;Append the COM port address
                mov     si,offset errtxt19      ;Point SI to error message
                mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                call    msg_window              ;Display the error message
                ret
;
; Make sure there's a modem attached and ready.
;
dial2:          add     dx,3                    ;Point DX to modem status
                in      al,dx                   ;Read modem status
                test    al,10h                  ;See if CTS is asserted
                jnz     dial3                   ;Branch if it is
                mov     si,offset errtxt15      ;Point SI to error message
                mov     di,offset screenbuffer2 ;Point DI to buffer area
                call    strlen                  ;Get message length in CX
                add     cx,3                    ;Add 3 to it
                mov     al,comport              ;Get COM port number in AL
                add     al,31h                  ;Convert binary to ASCII
                mov     errtxt15[61],al         ;Write it to ERRTXT15
                call    msg_window              ;Display the error message
                ret                             ;  and exit
;
; Initialize the modem.
;
dial3:          mov     si,offset initcmd       ;Copy the command to
                mov     di,offset inbuffer      ;  initialize the modem
                mov     ax,cs
                mov     es,ax
                call    copyz
                mov     si,offset inbuffer      ;Append a carriage return
                mov     di,offset endcmd        ;  to the end of it
                call    strcat
                mov     dx,uart_addr            ;Load DX with UART address
                mov     si,offset inbuffer      ;Transmit the command
                call    transmit_string

                mov     counter,18              ;Initialize counter
dial4:          cmp     counter,0               ;Pause for about 1 second
                jne     dial4                   ;  before proceeding
;
; Dial the phone number.
;
                mov     si,offset dialcmd       ;Build the command to dial
                mov     di,offset inbuffer      ;  the phone, starting with
                call    copyz                   ;  the "ATDT" string
                mov     si,offset inbuffer      ;Append the dialing prefix
                mov     di,offset prefix
                call    strcat
                mov     si,offset inbuffer      ;Append the number itself
                mov     di,bufferptr
                mov     cx,255
                sub     al,al
                repne   scasb
                repne   scasb
                call    strcat
                mov     si,offset inbuffer      ;Append the dial command
                mov     di,offset suffix        ;  suffix
                call    strcat
                mov     si,offset inbuffer      ;Append a carriage return to
                mov     di,offset endcmd        ;  terminate the command
                call    strcat

                mov     dx,uart_addr            ;Load DX with UART address
                mov     si,offset inbuffer      ;Transmit the command
                call    transmit_string
;
; Display the message window saying what to do next.
;
                mov     bx,color_ptr            ;Place attributes in AH,
                mov     ah,border_color         ;  AL, and BH
                mov     al,window_color_2
                mov     bh,shadow_color
                mov     ch,window_start         ;Load CX with the window's
                add     ch,3                    ;  starting coordinates
                mov     cl,13
                mov     dh,ch                   ;Load DX with the window's
                add     dh,4                    ;  ending coordinates
                mov     dl,66
                push    cx                      ;Save window coordinates
                push    dx
                push    cs                      ;Point ES:DI to the buffer
                pop     es                      ;  for screen data
                mov     di,offset screenbuffer2
                call    openwindow              ;Display the window

                mov     dh,window_start         ;Display the message itself
                add     dh,4
                mov     dl,15
                push    dx
                mov     si,offset msg1a
                call    show_string
                pop     dx
                inc     dh
                push    dx
                mov     si,offset msg1b
                call    show_string
                pop     dx
                inc     dh
                mov     si,offset msg1c
                call    show_string

                call    readkey                 ;Wait for a keypress

                pop     dx                      ;Close the message window
                pop     cx                      ;  by restoring what was
                add     dx,0101h                ;  under it
                mov     si,offset screenbuffer2
                call    restregion
;
; Disconnect the modem from the line and exit.
;
                mov     si,offset hangcmd       ;Build the command to
                mov     di,offset inbuffer      ;  disconnect the modem
                mov     ax,cs                   ;  from the line
                mov     es,ax
                call    copyz
                mov     si,offset inbuffer
                mov     di,offset endcmd
                call    strcat
                mov     dx,uart_addr            ;Load DX withe UART address
                mov     si,offset inbuffer      ;Transmit the command
                call    transmit_string
dial_exit:      ret
dial            endp

;****************************************************************************
; TRANSMIT_STRING transmits an ASCIIZ string to a COM port.  On entry,
; DS:SI points to the string and DX contains the COM port's I/O address.
;****************************************************************************

transmit_string proc    near
                lodsb                           ;Get a character
                or      al,al                   ;Exit if it's zero
                jz      trans_exit
                call    transmit_char           ;Transmit it to the COM port
                jmp     transmit_string         ;Loop back for more
trans_exit:     ret
transmit_string endp

;****************************************************************************
; TRANSMIT_CHAR transmits a character to a COM port.  On entry, AL holds
; the character and DX contains the COM port's I/O address.
;****************************************************************************

transmit_char   proc    near
                push    ax                      ;Save the character
                add     dx,5                    ;Point DX to line status
trans_loop:     in      al,dx                   ;Read line status from UART
                test    al,20h                  ;Loop until transmit holding
                jz      trans_loop              ;  register comes empty
                pop     ax                      ;Retrieve the character
                sub     dx,5                    ;Point DX to output port
                out     dx,al                   ;Transmit the character
                ret
transmit_char   endp

;****************************************************************************
; OPENWINDOW opens a window for input.  On entry CH and CL hold the
; coordinates of the window's upper left corner; DH and DL hold the
; coordinates of the lower right corner; AH holds the window's border
; color; AL holds the color of its interior; BH holds the drop shadow
; color; and ES:DI points to the buffer where video data will be stored.
;****************************************************************************

ul_col          db      ?                       ;Upper left column
ul_row          db      ?                       ;Upper left row
lr_col          db      ?                       ;Lower right column
lr_row          db      ?                       ;Lower right row
border_attr     db      ?                       ;Border attribute
window_attr     db      ?                       ;Window attribute
shadow_attr     db      ?                       ;Shadow attribute

openwindow      proc    near
                mov     word ptr ul_col,cx      ;Save window coordinates
                mov     word ptr lr_col,dx
                mov     border_attr,ah          ;Store video attributes
                mov     window_attr,al
                mov     shadow_attr,bh

                add     dx,0101h                ;Include drop shadow
                call    saveregion              ;Save the screen

                mov     ah,border_attr          ;Retrieve attribute
                mov     cx,word ptr ul_col      ;Retrieve window coordinates
                mov     dx,word ptr lr_col
                call    drawbox                 ;Draw the window border

                mov     bh,window_attr          ;Retrieve attribute
                mov     cx,word ptr ul_col      ;Retrieve window coordinates
                add     cx,0101h
                mov     dx,word ptr lr_col
                sub     dx,0101h
                mov     ax,0600h                ;Blank the interior of the
                int     10h                     ;  window

                mov     al,shadow_attr          ;Retrieve attribute
                mov     cl,lr_col               ;Compute length of the
                sub     cl,ul_col               ;  horizontal portion
                inc     cl                      ;  of the drop shadow
                sub     ch,ch                   ;Byte to word in CX
                mov     dh,lr_row               ;Load DX with the starting
                mov     dl,ul_col               ;  row and column address
                add     dx,0101h
                call    write_attr              ;Write the attribute

                mov     cl,lr_row               ;Compute length of the
                sub     cl,ul_row               ;  vertical portion of
                inc     cl                      ;  the drop shadow
                sub     ch,ch                   ;Byte to word in CX
                push    cx                      ;Save it
                mov     dh,ul_row               ;Load DX with the starting
                mov     dl,lr_col               ;  row and column address
                add     dx,0101h
                call    compute_address         ;Compute the address
                mov     di,ax                   ;Transfer offset to DI
                inc     di                      ;Point DI to attribute
                mov     es,video_segment        ;Point ES to video segment
                pop     cx                      ;Retrieve length
                mov     al,shadow_attr          ;Retrieve attribute
open_loop:      stosb                           ;Write one attribute
                add     di,line_length          ;Point DI to next attribute
                dec     di                      ;  byte
                loop    open_loop               ;Loop until done
                ret
openwindow      endp

;****************************************************************************
; PROMPT_WINDOW opens a window for string input.  On entry, CL holds the
; length of the window horizontally, CH holds the maximum number of char-
; acters that will be accepted, DS:SI points to the prompt string, and DS:DI
; points to the default input string.  On exit, AX holds the return code
; from EDIT_STRING.
;****************************************************************************

straddr1        dw      ?                       ;String address 1
straddr2        dw      ?                       ;String address 2
charlimit       db      ?                       ;Maximum character count

prompt_window   proc    near
                mov     straddr1,si             ;Save string addresses
                mov     straddr2,di
                mov     charlimit,ch            ;Save maximum character count
                mov     al,cl                   ;Copy window length to AL
                mov     ah,cl                   ;Copy window length to AH
                inc     al                      ;Increment length by 1
                shr     al,1                    ;Then divide it by 2
                
                mov     ch,window_start         ;Load CX and DX with the
                add     ch,4                    ;  window coordinates
                mov     cl,40
                sub     cl,al
                mov     dh,ch
                add     dh,2
                mov     dl,cl
                add     dl,ah
                mov     bx,color_ptr            ;Place video attributes in
                mov     ah,border_color         ;  AH, AL, and BH
                mov     al,window_color_2
                mov     bh,shadow_color
                push    cs                      ;Point ES:DI to the buffer
                pop     es                      ;  where screen data will
                mov     di,offset screenbuffer2 ;  be saved
                push    cx                      ;Save window coordinates
                push    dx
                call    openwindow              ;Display the prompt window
                pop     dx                      ;Retrieve window coordinates
                pop     cx

                push    cx                      ;Save the window coordinates
                push    dx                      ;  again
                mov     dx,cx                   ;Compute location to display
                add     dx,0102h                ;  prompt
                mov     si,straddr1             ;Point SI to the prompt
                call    show_string             ;Display the prompt
                pop     dx                      ;Retrieve window coordinates
                pop     cx

                push    cx                      ;Save the window coordinates
                push    dx                      ;  again
                mov     dx,cx                   ;Compute location to display
                add     dx,0103h                ;  the input string
                push    cx
                mov     si,straddr1
                call    strlen
                add     dl,cl
                pop     cx
                push    dx
                mov     si,straddr2             ;Point SI to the input string
                call    show_string             ;Display the input string
                pop     dx

                mov     al,01h                  ;Prepare registers for call
                mov     bx,00h                  ;  to EDIT_STRING
                mov     cl,charlimit
                sub     ch,ch
                mov     si,straddr2
                call    edit_string             ;Input a new string

                pop     dx                      ;Retrieve window coordinates
                pop     cx
                add     dx,0101h
                push    ax                      ;Save the return code
                mov     si,offset screenbuffer2 ;Close the window
                call    restregion
                pop     ax                      ;Retrieve return code and
                ret                             ;  return to caller
prompt_window   endp

;****************************************************************************
; MSG_WINDOW opens a window and displays a message.  On entry, CL holds
; the length of the window horizontally, DS:SI points to the ASCIIZ message
; string, and DS:DI points to the buffer where video data will be stored.
;****************************************************************************

msg_window      proc    near
                push    di                      ;Save buffer address
                push    si                      ;Save string address
                mov     al,cl                   ;Copy window length to AL
                mov     ah,cl                   ;Copy window length to AH
                inc     al                      ;Increment length by 1
                shr     al,1                    ;Then divide it by 2

                mov     bx,ds                   ;Point ES to this segment
                mov     es,bx
                mov     ch,window_start         ;Load CX and DX with the
                add     ch,4                    ;  window coordinates
                mov     cl,40
                sub     cl,al
                mov     dh,ch
                add     dh,2
                mov     dl,cl
                add     dl,ah
                mov     bx,color_ptr            ;Place video attributes in
                mov     ah,error_color          ;  AH, AL, and BH
                mov     al,ah
                mov     bh,shadow_color
                push    cx                      ;Save window coordinates
                push    dx
                call    openwindow              ;Display the error window
                pop     dx                      ;Retrieve window coordinates
                pop     cx

                pop     si                      ;Retrieve message address
                push    cx                      ;Save the window coordinates
                push    dx                      ;  again
                mov     dx,cx                   ;Compute the row and column
                add     dh,1                    ;  to display the error
                call    strlen                  ;  message
                shr     cl,1
                mov     dl,40
                sub     dl,cl
                call    show_string             ;Display the error message

                call    readkey                 ;Wait until a key is pressed

                pop     dx                      ;Retrieve window coordinates
                pop     cx                      ;  and close the window
                add     dx,0101h
                pop     si
                call    restregion
                ret
msg_window      endp

;****************************************************************************
; READFILE loads a data file.  Carry set on return means an error occurred.
; If carry is set, AL holds one of the following error codes:
;
;       01H     File was not found or could not be opened
;       02H     File is too large to fit in the internal buffer
;       03H     File was not created by PC-DIAL (header is not valid)
;       04H     An error occurred while reading the file
;
; On entry, both DS and ES must point to the segment that holds the file
; specification and the data buffer.
;****************************************************************************

bytestoread     dw      ?

readfile        proc    near
                mov     ax,3D02h                ;Open the file with read/
                mov     dx,offset filename      ;  write access privilege
                int     21h
                jc      read_error1             ;Error if call failed

                mov     bx,ax                   ;Transfer file handle to BX
                mov     ax,4202h                ;Determine the file's size
                sub     cx,cx                   ;  by positioning the file
                mov     dx,cx                   ;  pointer at the end of
                int     21h                     ;  the file
                or      dx,dx                   ;Error if file is larger
                jnz     read_error2             ;  than 64k
                cmp     ax,15                   ;Error if the file is less
                jb      read_error3             ;  than 15 bytes long
                sub     ax,15                   ;Adjust file size
                mov     bytestoread,ax          ;Save the result
                cmp     ax,buffersize           ;Error if file is larger
                ja      read_error2             ;  than the buffer size

                push    ax                      ;Save the file size
                mov     ax,4200h                ;Reset the file pointer to
                sub     cx,cx                   ;  the beginning of the
                mov     dx,cx                   ;  file
                int     21h

                mov     ah,3Fh                  ;Read the 13-byte header
                mov     cx,13                   ;  from the file
                mov     dx,offset databuffer
                int     21h
                jc      read_error4             ;Branch on error
                mov     si,offset signature     ;Compare strings to see if
                mov     di,offset databuffer    ;  this is a valid PC-DIAL
                mov     cx,13                   ;  data file
                repe    cmpsb
                pop     cx                      ;Retrieve file size
                jne     read_error3             ;Error if strings don't match

                mov     ah,3Fh                  ;Read the number of records
                mov     cx,2                    ;  from the data file
                mov     dx,offset records
                int     21h
                jc      read_error4             ;Branch on error

                mov     ah,3Fh                  ;Read the file into the
                mov     cx,bytestoread          ;  data buffer
                mov     dx,offset databuffer
                int     21h
                jc      read_error4             ;Branch on error

                mov     ah,3Eh                  ;Close the file
                int     21h
                mov     bufferptr,offset databuffer     ;Reset pointers
                mov     ax,offset databuffer
                add     ax,bytestoread
                mov     buffertop,ax
                clc                             ;Clear carry and exit
                ret

read_error1:    mov     al,01h                  ;Set AL to 01H and exit
                jmp     short read_error
read_error2:    mov     ah,3Eh                  ;Close the file, set AL to
                int     21h                     ;  02H, and exit
                mov     al,02h
                jmp     short read_error
read_error3:    mov     ah,3Eh                  ;Close the file, set AL to
                int     21h                     ;  03H, and exit
                mov     al,03h
                jmp     short read_error
read_error4:    mov     ah,3Eh                  ;Close the file, set AL to
                int     21h                     ;  04H, and exit
                mov     al,04h
read_error:     stc
                ret
readfile        endp

;****************************************************************************
; SAVEFILE writes the contents of the data buffer to disk.  Carry set on
; return means an error occurred.  If carry is set, AL holds one of the
; following error codes:
;
;       01H     The file could not be created
;       02H     An error occurred during a write
;       03H     The destination disk is full
;****************************************************************************

bytestowrite    dw      ?

savefile        proc    near
                mov     ah,3Ch                  ;Create the file or truncate
                sub     cx,cx                   ;  it to zero length if it
                mov     dx,offset filename      ;  already exists
                int     21h
                jc      save_error1             ;Exit if the call failed

                mov     bx,ax                   ;Transfer file handle to BX
                mov     ah,40h                  ;Write the 13-byte header to
                mov     cx,13                   ;  the file to identify it
                mov     dx,offset signature     ;  as a PC-DIAL file
                int     21h
                jc      save_error2             ;Exit if the call failed
                cmp     ax,13                   ;Also exit if the disk is
                jb      save_error3             ;  full

                mov     ah,40h                  ;Write the record count to
                mov     cx,2                    ;  the file
                mov     dx,offset records
                int     21h
                jc      save_error2             ;Exit if the call failed
                cmp     ax,2                    ;Also exit if the disk is
                jb      save_error3             ;  full

                mov     ah,40h                  ;Write the contents of the
                mov     cx,buffertop            ;  data buffer to the file
                sub     cx,offset databuffer
                mov     dx,offset databuffer
                mov     bytestowrite,cx
                int     21h
                jc      save_error2             ;Exit if the call failed
                cmp     ax,bytestowrite         ;Also exit if the disk is
                jb      save_error3             ;  full

                mov     ah,3Eh                  ;Close the file and exit
                int     21h                     ;  with carry clear
                clc
                ret

save_error1:    mov     al,01h                  ;Set AL to 01H and exit
                jmp     short save_error
save_error2:    mov     ah,3Eh                  ;Close the file, set AL to
                int     21h                     ;  02H, and exit
                mov     al,02h
                jmp     short save_error        ;Close the file, set AL to
save_error3:    mov     ah,3Eh                  ;  03H, and exit
                int     21h
                mov     al,03h
save_error:     stc
                ret
savefile        endp

;****************************************************************************
; SAVESETUP writes setup information to disk.  Carry set on return means an
; error occurred.  If carry is set, AL holds one of the following error
; codes:
;
;       01H     PC-DIAL.COM was not found or could not be opened
;       02H     An unknown error occurred during the write operation
;****************************************************************************

savesetup       proc    near
                mov     ax,3D01h                ;Open the program file for
                mov     dx,offset progfile      ;  writing
                int     21h
                jc      setup_error1            ;Branch if call failed

                mov     bx,ax                   ;Transfer file handle to BX
                mov     ax,4200h                ;Move the file pointer to the
                sub     cx,cx                   ;  beginning of the setup
                mov     dx,offset addr_table    ;  information
                sub     dx,0100h
                int     21h

                mov     ah,40h                  ;Write the setup information
                mov     cx,offset commptr+1     ;  to disk
                sub     cx,offset addr_table
                mov     dx,offset addr_table
                int     21h
                jc      setup_error2

                mov     ah,3Eh                  ;Close the file and exit
                int     21h                     ;  with carry clear
                clc
                ret

setup_error1:   mov     al,01h                  ;Set AL to 01H and exit
                jmp     short setup_error
setup_error2:   mov     ah,3Eh                  ;Close the file, set AL to
                int     21h                     ;  01H, and exit
                mov     al,02h
setup_error:    stc
                ret
savesetup       endp

;****************************************************************************
; HEX2BIN converts a hex number entered in ASCIIZ form into a binary
; value in AX.  On entry, DS:SI points to the string.  Carry set on return
; indicates that an error occurred in the conversion.
;****************************************************************************

sixteen         dw      16

hex2bin         proc    near
                sub     ax,ax                   ;Initialize registers
                sub     bh,bh

h2b_loop:       mov     bl,[si]                 ;Get a character
                inc     si
                or      bl,bl                   ;Exit if it's 0
                jz      h2b_exit

                cmp     bl,"0"                  ;Error if character is less
                jb      h2b_error               ;  than "0"
                cmp     bl,"9"                  ;Branch if it's between "0"
                jbe     h2b_shift               ;  and "9"
                and     bl,0DFh                 ;Capitalize the character
                cmp     bl,"A"                  ;Error if it's less than "A"
                jb      h2b_error
                cmp     bl,"F"                  ;Error if it's greater than
                ja      h2b_error               ;  "B"
                sub     bl,7                    ;Convert hex digit to number

h2b_shift:      mul     sixteen                 ;Multiply the value in AX by
                jc      h2b_error               ;  16 and exit on overflow
                sub     bl,30h                  ;ASCII => binary
                add     ax,bx                   ;Add latest value to AX and
                adc     dx,0                    ;  DX
                or      dx,dx                   ;Error if DX is not 0
                jz      h2b_loop                ;Loop back for more

h2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

h2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
hex2bin         endp

;****************************************************************************
; BIN2HEX converts a binary value in AX to an ASCIIZ number in hex format.
; On entry, ES:DI points to the location where the string will be written.
;****************************************************************************

bin2hex         proc    near
                mov     bx,16                   ;Initialize divisor word and
                xor     cx,cx                   ;  digit counter
b2h1:           inc     cx                      ;Increment digit count
                sub     dx,dx                   ;Divide by 16
                div     bx
                push    dx                      ;Save remainder on stack
                or      ax,ax                   ;Loop until quotient is zero
                jnz     b2h1
b2h2:           pop     ax                      ;Retrieve last digit
                cmp     al,10                   ;Branch if it's less than 10
                jb      b2h3
                add     al,7                    ;Otherwise add 7
b2h3:           add     al,30h                  ;binary => ASCII
                stosb
                loop    b2h2                    ;Loop until done
                sub     al,al                   ;Append the terminating
                stosb                           ;  zero to the string
                ret
bin2hex         endp

;****************************************************************************
; Buffer areas to be used after installation.
;****************************************************************************

screenbuffer1   =       $                       ;Screen buffer (1920 bytes)
screenbuffer2   =       $ + 1920                ;Screen buffer (1034 bytes)
screenbuffer3   =       $ + 2954                ;Screen buffer (568 bytes)
mystack         =       $ + 4034                ;Internal stack (512 bytes)
databuffer      =       $ + 4038                ;Data buffer

;****************************************************************************
; Data that will be discarded when the program becomes memory-resident.
;****************************************************************************

helpmsg         db      "Installs a popup phone directory and dialer."
                db      13,10,13,10
                db      "PC-DIAL [[d:][path]filename] [/B=size] [/M]"
                db      13,10,13,10
                db      "  filename  Data file to load at start-up."
                db      13,10
                db      "  /B=size   Buffer size in kilobytes (default=16)."
                db      13,10
                db      "  /M        Use monochrome video attributes."
                db      13,10,13,10
                db      "Once PC-DIAL is installed, pressing Alt and the "
                db      "right Shift key pops ",13,10
                db      "up the phone directory window.",13,10,"$"

errmsg1         db      "Syntax: PC-DIAL [[d:][path]filename] [/B=size] "
                db      " [/M]",13,10,"$"
errmsg2         db      "Requires DOS 3.0 or higher",13,10,"$"
errmsg3         db      "Program could not be installed",13,10,"$"
errmsg4         db      "Program is already installed",13,10,"$"
errmsg5         db      "Invalid buffer size (minimum=1, maximum=48)"
                db      13,10,"$"
errmsg6         db      "Invalid path or file name",13,10,"$"
errmsg7         db      "File not found or could not be opened",13,10,"$"
errmsg8         db      "File is too large for the data buffer",13,10,"$"
errmsg9         db      "File was not created by PC-DIAL",13,10,"$"

msg1            db      "PC-DIAL 1.0 installed",13,10
                db      "Hotkey is Alt-Right Shift",13,10,"$"

eol_flag        db      0                       ;1=End of line reached

;****************************************************************************
; INIT makes the program resident in memory.
;****************************************************************************

                assume  cs:code,ds:code

init            proc    near
                cld                             ;Clear direction flag
                mov     si,81h                  ;Point SI to command line
                call    scanhelp                ;Scan for "/?" switch
                jnc     checkver                ;Branch if not found
                mov     ah,09h                  ;Display help text and exit
                mov     dx,offset helpmsg       ;  with ERRORLEVEL=0
                int     21h
                mov     ax,4C00h
                int     21h
;
; Check the DOS version and terminate if the program is already installed.
;
checkver:       mov     dx,offset errmsg2       ;Get the DOS version number
                mov     ah,30h
                int     21h
                mov     byte ptr dosversion,ah  ;Save it
                mov     byte ptr dosversion[1],al
                cmp     al,3                    ;Exit if version number is
                jae     checkins                ;  less than 3.0

error:          mov     ah,09h                  ;Display error message and
                int     21h                     ;  exit with ERRORLEVEL=1
                mov     ax,4C01h
                int     21h

checkins:       push    es                      ;Save ES
                call    check_install           ;Get installed status
                pop     es                      ;Restore ES
                mov     dx,offset errmsg4       ;Error if the program is
                jc      error                   ;  already installed
                mov     si,81h                  ;Reset SI
;
; Parse the command line.
;
        
parse:          call    findchar                ;Find first parameter
                jc      endparse                ;Branch if none found
                cmp     byte ptr [si],"/"       ;Branch if it's a "/"
                je      slashb                  ;  character

                mov     bx,si                   ;Save SI temporarily
                call    finddelim               ;Find the end of the name
                jnc     parse1                  ;Branch if not end of line
                mov     eol_flag,1              ;Set EOL flag to 1
parse1:         mov     byte ptr [si],0         ;Convert string to ASCIIZ
                inc     si                      ;Advance SI past the zero
                push    si                      ;Save SI
                mov     ah,60h                  ;Convert the file name into
                mov     si,bx                   ;  a fully qualified file
                mov     di,offset filename      ;  name
                int     21h
                pop     si                      ;Retrieve SI
                mov     dx,offset errmsg6       ;Initialize error pointer
                jc      error                   ;Error if call failed
                cmp     eol_flag,0              ;Branch if end of line was
                jne     endparse                ;  reached
                jmp     parse                   ;Otherwise loop back
;
; Process a /B switch.
;
slashb:         mov     dx,offset errmsg1       ;Initialize error pointer
                inc     si                      ;Skip the "/" character
                lodsb                           ;Get a character
                and     al,0DFh                 ;Capitalize it
                cmp     al,"B"                  ;Branch if it's not a "B"
                jne     slashm

                lodsb                           ;Get the next character
                cmp     al,"="                  ;Error if it's not a "="
                jne     error
                cmp     byte ptr [si],"0"       ;Error if the next character
                jb      error                   ;  is not a number
                cmp     byte ptr [si],"9"
                ja      error
                call    asc2bin                 ;Convert the number to binary
                mov     dx,offset errmsg5       ;Error if it's less than 1 or
                cmp     al,1                    ;  greater than 48
                jb      error
                cmp     al,48
                ja      error
                cbw                             ;Convert kilobytes to bytes
                mov     cl,10                   ;  by shifting left 10
                shl     ax,cl                   ;  places
                mov     buffersize,ax           ;Store the result
                sub     bufferlimit,defaultsize ;Adjust address of top of
                add     bufferlimit,ax          ;  data buffer
                jmp     parse                   ;Reenter the parse loop
;
; Process a /M switch.
;
slashm:         cmp     al,"M"                  ;Error if the character is
                jne     error1                  ;  not an "M"
                push    si                      ;Save SI
                mov     si,offset mono_table    ;Point SI to mono attributes
                mov     di,offset color_table   ;Point DI to color attributes
                mov     cx,10                   ;Initalize counter
                rep     movsb                   ;Copy mono to color
                pop     si                      ;Restore SI
                jmp     parse                   ;Reenter the parsing loop
;
; Read the data file if a name was specified.
;
endparse:       cmp     filename,0              ;Go install if no file name
                je      install                 ;  was entered
                call    readfile                ;Read the data file
                jnc     install                 ;Go install if no error
                mov     dx,offset errmsg7       ;Initialize DX with address
                cmp     al,1                    ;  of error message and exit
                je      error1
                mov     dx,offset errmsg8
                cmp     al,2
                je      error1
                mov     dx,offset errmsg9
error1:         jmp     error
;
; Install the program.
;
install:        call    mplex_id                ;Find a multiplex ID number
                mov     dx,offset errmsg3       ;Error if none available
                jc      error1
                mov     prog_id,ah              ;Save the ID number

                mov     ah,34h                  ;Get and save the address of
                int     21h                     ;  the InDOS flag
                mov     word ptr indos,bx
                mov     word ptr indos[2],es

                push    ds                      ;Save DS
                mov     ax,5D06h                ;Get and save the address of
                int     21h                     ;  the critical error flag
                mov     word ptr cs:[errorflag],si
                mov     word ptr cs:[errorflag+2],ds
                pop     ds                      ;Restore DS
                mov     dx,offset errmsg3       ;Error if function returned
                jc      error1                  ;  with carry set

                mov     si,offset databuffer-1  ;Zero the byte just before
                mov     byte ptr [si],0         ;  the data buffer

                mov     ah,12h                  ;Test for EGA/VGA
                mov     bl,10h
                int     10h
                cmp     bl,10h                  ;EGA/VGA installed if BL
                jne     set_vectors             ;  returned changed
                mov     egaflag,0               ;Zero video adapter flag

set_vectors:    mov     ax,3508h                ;Hook interrupt 08H
                int     21h
                mov     word ptr int08h,bx
                mov     word ptr int08h[2],es
                mov     ax,2508h
                mov     dx,offset timer_int
                int     21h

                mov     ax,3509h                ;Hook interrupt 09H
                int     21h
                mov     word ptr int09h,bx
                mov     word ptr int09h[2],es
                mov     ax,2509h
                mov     dx,offset kb_int
                int     21h

                mov     ax,3510h                ;Hook interrupt 10H
                int     21h
                mov     word ptr int10h,bx
                mov     word ptr int10h[2],es
                mov     ax,2510h
                mov     dx,offset video_int
                int     21h

                mov     ax,3513h                ;Hook interrupt 13H
                int     21h
                mov     word ptr int13h,bx
                mov     word ptr int13h[2],es
                mov     ax,2513h
                mov     dx,offset disk_int
                int     21h

                mov     ax,3525h                ;Hook interrupt 25H
                int     21h
                mov     word ptr int25h,bx
                mov     word ptr int25h[2],es
                mov     ax,2525h
                mov     dx,offset abs_read_int
                int     21h

                mov     ax,3526h                ;Hook interrupt 26H
                int     21h
                mov     word ptr int26h,bx
                mov     word ptr int26h[2],es
                mov     ax,2526h
                mov     dx,offset abs_write_int
                int     21h

                mov     ax,3528h                ;Hook interrupt 28H
                int     21h
                mov     word ptr int28h,bx
                mov     word ptr int28h[2],es
                mov     ax,2528h
                mov     dx,offset dosidle_int
                int     21h

                mov     ax,352Fh                ;Hook interrupt 2FH
                int     21h
                mov     word ptr int2Fh,bx
                mov     word ptr int2Fh[2],es
                mov     ax,252Fh
                mov     dx,offset mplex_int
                int     21h

                mov     ah,49h                  ;Get the segment address of
                mov     es,ds:[2Ch]             ;  the environment block
                int     21h                     ;  and free the segment

                mov     ah,09h                  ;Display message verifying
                mov     dx,offset msg1          ;  the installation
                int     21h

                mov     ax,3100h                ;Terminate with function 31H
                mov     dx,offset databuffer+15 ;Compute amount of memory to
                add     dx,buffersize           ;  reserve in DX first
                mov     cl,4
                shr     dx,cl
                int     21h
init            endp

;****************************************************************************
; FINDCHAR advances SI to the next non-space or non-comma character.
; On return, carry set indicates EOL was encountered.
;****************************************************************************

findchar        proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Loop if space
                je      findchar
                cmp     al,2Ch                  ;Loop if comma
                je      findchar
                dec     si                      ;Point SI to the character
                cmp     al,0Dh                  ;Exit with carry set if end
                je      eol                     ;  of line is reached

                clc                             ;Clear carry and exit
                ret

eol:            stc                             ;Set carry and exit
                ret
findchar        endp

;****************************************************************************
; FINDDELIM advances SI to the next space or comma character.  On return,
; carry set indicates EOL was encountered.
;****************************************************************************

finddelim       proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Exit if space
                je      fd_exit
                cmp     al,2Ch                  ;Exit if comma
                je      fd_exit
                cmp     al,0Dh                  ;Loop back for more if end
                jne     finddelim               ;  of line isn't reached

                dec     si                      ;Set carry and exit
                stc
                ret

fd_exit:        dec     si                      ;Clear carry and exit
                clc
                ret
finddelim       endp

;****************************************************************************
; SCANHELP scans the command line for a /? switch.  If found, carry returns
; set and SI contains its offset.  If not found, carry returns clear.
;****************************************************************************

scanhelp        proc    near
                push    si                      ;Save SI
scanloop:       lodsb                           ;Get a character
                cmp     al,0Dh                  ;Exit if end of line
                je      scan_exit
                cmp     al,"?"                  ;Loop if not "?"
                jne     scanloop
                cmp     byte ptr [si-2],"/"     ;Loop if not "/"
                jne     scanloop

                add     sp,2                    ;Clear the stack
                sub     si,2                    ;Adjust SI
                stc                             ;Set carry and exit
                ret

scan_exit:      pop     si                      ;Restore SI
                clc                             ;Clear carry and exit
                ret
scanhelp        endp

;****************************************************************************
; CHECK_INSTALL returns carry set if the program is already installed,
; carry clear if it's not.  If carry returns set, AH holds the program's
; multiplex ID number.
;****************************************************************************

check_install   proc    near
                mov     ax,8000h                ;Initialize AH and AL
                mov     cx,80h                  ;Initialize count

chinst1:        push    ax                      ;Save AX and CX
                push    cx
                sub     di,di                   ;Set ES and DI to 0
                mov     es,di
                int     2Fh                     ;Interrupt 2Fh
                cmp     al,0FFh                 ;Nothing here if AL isn't
                jne     chinst2                 ;  equal to FFH

                mov     si,offset signature     ;See if program signature
                mov     cx,13                   ;  appears at the address
                repe    cmpsb                   ;  returned in ES:DI
                jne     chinst2                 ;Branch if it does not

                pop     cx                      ;Clear the stack and exit
                pop     ax                      ;  with carry set
                stc
                ret

chinst2:        pop     cx                      ;Retrieve AX and CX
                pop     ax
                inc     ah                      ;Next multiplex ID
                loop    chinst1                 ;Loop until done

                clc                             ;Exit with carry clear
                ret
check_install   endp

;****************************************************************************
; MPLEX_ID searches for an unused multiplex ID number.  If one is found,
; it is returned in AH with carry clear.  Carry set means no multiplex
; ID numbers are currently available.
;****************************************************************************

mplex_id        proc    near
                mov     ax,8000h                ;Initialize AH and AL
                mov     cx,80h                  ;Initialize count

mxid1:          push    ax                      ;Save AX and CX
                push    cx
                int     2Fh                     ;Interrupt 2Fh
                or      al,al                   ;Branch if AL=0
                jz      mxid2
                pop     cx                      ;Retrieve AX and CX
                pop     ax
                inc     ah                      ;Increment ID number
                loop    mxid1                   ;Loop until done

                stc                             ;Exit with carry set
                ret

mxid2:          pop     cx                      ;Clear the stack and exit
                pop     ax                      ;  with carry clear
                clc
                ret
mplex_id        endp

;****************************************************************************
; ASC2BIN converts a decimal number entered in ASCII form into a binary
; value in AL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

asc2bin         proc    near
                sub     ax,ax                   ;Initialize registers
                sub     bh,bh
                mov     dl,10

a2b_loop:       mov     bl,[si]                 ;Get a character
                inc     si
                cmp     bl,20h                  ;Exit if space
                je      a2b_exit
                cmp     bl,2Ch                  ;Exit if comma
                je      a2b_exit
                cmp     bl,0Dh                  ;Exit if carriage return
                je      a2b_exit

                cmp     bl,"0"                  ;Error if character is not
                jb      a2b_error               ;  a number
                cmp     bl,"9"
                ja      a2b_error

                mul     dl                      ;Multiply the value in AL by
                jc      a2b_error               ;  10 and exit on overflow
                sub     bl,30h                  ;ASCII => binary
                add     ax,bx                   ;Add latest value to AX
                cmp     ax,255                  ;Error if sum > 255
                jna     a2b_loop                ;Loop back for more

a2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

a2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
asc2bin         endp

code            ends
                end     begin
