;****************************************************************************
; PC-LOG maintains a log of all the programs run on your PC in an ASCII
; text file.  Its syntax is
;
;       PC-LOG [[d:][path]filename]
;
; where "filename" is the name of the ASCII file programs will be logged
; to.  If "filename" is not specified, the program creates a file called
; USAGE.LOG in the root directory of the current drive.
;****************************************************************************

code            segment
                assume  cs:code
                org     100h
begin:          jmp     init

signature       db      0,1,2,"PC-LOG",2,1,0    ;Program signature
prog_id         db      ?                       ;Multiplex ID number

int21h          dd      ?                       ;Interrupt 21H vector
int2Fh          dd      ?                       ;Interrupt 2FH vector

dosversion      dw      ?                       ;DOS version number
handle          dw      ?                       ;File handle for log file
level           dw      0                       ;EXEC level number (0 thru 9)
execflag        db      0                       ;0=Entering EXEC function

xerror_ax       dw      ?                       ;Storage array for DOS
xerror_bx       dw      ?                       ;  extended error
xerror_cx       dw      ?                       ;  information
xerror_dx       dw      ?
xerror_si       dw      ?
xerror_di       dw      ?
xerror_ds       dw      ?
xerror_es       dw      ?
                dw      3 dup (0)

months          db      "JanFebMarAprMayJunJulAugSepOctNovDec"

header          db      "Log file created by PC-LOG version 1.0",13,10
                db      "Copyright (c) 1991 ",254," Jeff Prosise ",254
                db      " Ziff-Davis Press",13,10

titles          db      "START",5 DUP (32),"END",5 DUP (32)
                db      "ELAPSED",4 DUP (32),"LEVEL",4 DUP (32)
                db      "PROGRAM",10 DUP (32),"PARAMETERS",13,10

logfile         db      "x:\USAGE.LOG", 116 dup (0)

;****************************************************************************
; DOS_INT handles interrupt 21H.
;****************************************************************************

dos_int         proc    far
                pushf                           ;Save FLAGS
                cmp     ax,4B00h                ;Exit immediately if this
                jne     exec0                   ;  isn't a call to EXEC
                cmp     cs:[level],9            ;Exit if we've exceeded
                jb      exec1                   ;  reentrancy limits
exec0:          popf                            ;Restore FLAGS
                jmp     cs:[int21h]             ;Exit to DOS EXEC handler
;
; Save registers passed to EXEC.
;
exec1:          sti                             ;Interrupts on
                push    ax                      ;Save registers
                push    bx
                push    cx
                push    dx
                push    si
                push    di
                push    bp
                push    ds
                push    es
;
; Record the program start/end time and write an entry to the log file.
;
                push    bx                      ;Save registers set for
                push    dx                      ;  EXEC call
                push    ds
                push    es
                push    cs                      ;Point DS to the code
                pop     ds                      ;  segment
                assume  ds:code

                inc     level                   ;Increment reentrancy count
                call    record_time             ;Record the current time
                cmp     level,1
                je      skipwrite1
                call    open_file               ;Open the log file
                mov     si,level                ;Point SI to start time and
                dec     si                      ;  DI to end time
                shl     si,1
                shl     si,1
                add     si,offset times
                mov     di,si
                add     di,4
                mov     al,byte ptr level       ;Store previous level number
                dec     al                      ;  in AL
                call    write_entry             ;Write log entry
                call    close_file              ;Close the log file
skipwrite1:     mov     execflag,1              ;Set EXEC flag

                assume  ds:nothing
                pop     es                      ;Restore EXEC registers
                pop     ds
                pop     dx
                pop     bx
;
; Record the name of the program just EXECed.
;
                push    es                      ;Save ES and BX
                push    bx
                push    ds                      ;Set ES equal to DS
                pop     es
                mov     di,dx                   ;Scan for terminating zero
                sub     al,al                   ;  in ASCIIZ file name
                mov     cx,128
                cld                             ;Clear the direction flag
                repne   scasb
                mov     bx,127                  ;Transfer string length
                sub     bx,cx                   ;  to CX
                mov     cx,bx
                mov     si,di                   ;Get ending address in SI
                sub     si,2                    ;Point SI to last character
                std                             ;Set the direction flag
exec2:          lodsb                           ;Scan backward for first
                cmp     al,"\"                  ;  character in file name
                je      exec3
                cmp     al,":"
                je      exec3
                loop    exec2
                dec     si
exec3:          add     si,2                    ;Point SI to first character
                sub     bx,cx                   ;Compute length of file name
                mov     cx,bx                   ;Transfer length to CX
                cld                             ;Clear direction flag again
                push    cs                      ;Point ES to code segment
                pop     es
                mov     al,13                   ;Compute offset into table
                mov     dl,byte ptr cs:[level]  ;  of program names
                mul     dl
                mov     di,ax
                add     di,offset names
                mov     al,cl                   ;Write file name character
                stosb                           ;  count into table
exec5:          lodsb                           ;Convert lowercase characters
                cmp     al,"a"                  ;  to upper and copy file
                jb      exec6                   ;  name into table
                cmp     al,"z"
                ja      exec6
                and     al,0DFh
exec6:          stosb
                loop    exec5
                pop     bx                      ;Restore ES and BX
                pop     es
;
; Record the command line passed to the program just EXECed.
;
                lds     si,es:[bx+2]            ;Load string address
                mov     cl,[si]                 ;Get string length in CX
                sub     ch,ch
                inc     cx
                push    cs                      ;Point ES:DI to command line
                pop     es                      ;  buffer in code segment
                mov     di,offset cmdline
                rep     movsb                   ;Copy the command line
;
; Restore the register values passed to EXEC.
;
                pop     es
                pop     ds
                pop     bp
                pop     di
                pop     si
                pop     dx
                pop     cx
                pop     bx
                pop     ax
                popf
;
; Save registers again since the EXEC function destroys them.
;
                push    bx
                push    cx
                push    dx
                push    si
                push    di
                push    bp
                push    ds
                push    es
;
; Save the values of SS and SP, call EXEC, and restore the stack.
;
                pushf                           ;Save FLAGS
                mov     cs:[handle],bx          ;Save BX temporarily
                mov     bx,cs:[level]           ;Compute an index into the
                dec     bx                      ;  STACKSEG and STACKPTR
                shl     bx,1                    ;  tables
                popf                            ;Restore FLAGS
                mov     word ptr cs:[stackseg+bx],ss    ;Save the SS and SP
                mov     word ptr cs:[stackptr+bx],sp    ;  registers
                mov     bx,cs:[handle]          ;Restore BX

                pushf                           ;Push FLAGS
                cli                             ;Disable interrupts
                call    int21h                  ;Call the DOS EXEC function

                mov     cs:[handle],ax          ;Save the return code
                lahf                            ;Store FLAGS in AH for now
                mov     bx,cs:[level]           ;Recompute the table index
                dec     bx
                shl     bx,1
                cli                             ;Disable interrupts
                mov     ss,word ptr cs:[stackseg+bx]    ;Restore SS and SP
                mov     sp,word ptr cs:[stackptr+bx]
                sti                             ;Enable interrupts
                sahf                            ;Restore the FLAGS register
                mov     ax,cs:[handle]          ;Restore return code in AX
;
; Restore the registers to their conditions before EXEC was called.
;
                pop     es
                pop     ds
                pop     bp
                pop     di
                pop     si
                pop     dx
                pop     cx
                pop     bx
;
; Save registers again while post-processing is performed.
;
                pushf
                push    ax
                push    bx
                push    cx
                push    dx
                push    si
                push    di
                push    bp
                push    ds
                push    es
;
; Save extended error information if DOS version is 3.10 or later.
;
                push    cs                      ;Point DS to code segment
                pop     ds
                assume  ds:code
                cmp     dosversion,030Ah        ;Skip if not 3.10 or later
                jb      exec7
                push    ds                      ;Save DS
                mov     ah,59h                  ;Get extended error
                sub     bx,bx                   ;  information
                int     21h
                mov     cs:[xerror_ds],ds       ;Save return value of DS
                pop     ds                      ;Set DS to code segment again
                mov     xerror_ax,ax            ;Save return values in
                mov     xerror_bx,bx            ;  XERROR array
                mov     xerror_cx,cx
                mov     xerror_dx,dx
                mov     xerror_si,si
                mov     xerror_di,di
                mov     xerror_es,es
;
; Record the program start/end time and write an entry to the log file.
;
exec7:          dec     level                   ;Decrement reentrancy count
                call    record_time             ;Record the current time
                call    open_file               ;Open the log file
                mov     si,level                ;Point SI to start time and
                inc     si                      ;  DI to end time
                shl     si,1
                shl     si,1
                add     si,offset times
                mov     di,si
                sub     di,4
                mov     al,byte ptr level       ;Store previous level number
                inc     al                      ;  in AL
                call    write_entry             ;Write log entry
                call    close_file              ;Close the log file
                mov     execflag,0              ;Clear EXEC flag
;
; Restore extended error information if DOS version is 3.10 or later.
;
                cmp     dosversion,030Ah        ;Skip if not 3.10 or later
                jb      exec8
                mov     ax,5D0Ah                ;Restore information with
                mov     dx,offset xerror_ax     ;  DOS function 5Dh
                int     21h
;
; Restore registers a final time and exit to the parent program.
;
                assume  ds:nothing
exec8:          pop     es
                pop     ds
                pop     bp
                pop     di
                pop     si
                pop     dx
                pop     cx
                pop     bx
                pop     ax
                popf
                ret 2                           ;Return with FLAGS intact
dos_int         endp

;****************************************************************************
; MPLEX_INT handles interrupt 2FH.  If, on entry, AH is set to PC-LOG's
; multiplex ID number, MPLEX_INT uses the value in AL as a function code.
; The functions supported are:
;
;    00H    Returns FFH in AL to indicate the program is installed
;           and returns the address of its signature in ES:DI.
;****************************************************************************

mplex_int       proc    far
                pushf                           ;Save FLAGS register
                cmp     ah,cs:[prog_id]         ;Branch if AH holds the
                je      mplex2                  ;  multiplex ID
                popf                            ;Restore FLAGS
                jmp     cs:[int2Fh]             ;Pass the interrupt on
;
; Function 00H verifies that the program is installed.
;               
mplex2:         popf                            ;Restore FLAGS
                or      al,al                   ;Branch if function code
                jnz     mplex3                  ;  is other than 00H
                mov     al,0FFh                 ;Set AL to FFH
                push    cs                      ;Point ES:DI to the program
                pop     es                      ;  signature
                mov     di,offset signature
mplex3:         iret                            ;Return from interrupt
mplex_int       endp

;****************************************************************************
; OPEN_FILE attempts to open or create the file output is redirected to.
; On return, carry is clear if the attempt was successful and the file
; handle is in AX.  Carry set means the attempt failed.  On entry, DS
; must point to the code segment.
;****************************************************************************

callflag        db      0

open_file       proc    near
                assume  ds:code
                mov     ax,3D01h                ;Attempt to open the file
                mov     dx,offset logfile       ;  for writing
                int     21h
                jnc     opened                  ;Branch if it worked

                mov     ax,4301h                ;Attempt to strip all the
                sub     cx,cx                   ;  attributes off the file
                mov     dx,offset logfile
                int     21h

                mov     ax,3D01h                ;Then attempt to open it
                mov     dx,offset logfile       ;  for writing again
                int     21h
                jc      create_file             ;Branch if it failed

opened:         mov     handle,ax               ;Store the file handle
                mov     bx,ax                   ;Move the file pointer
                mov     ax,4202h                ;  to the end of the
                sub     cx,cx                   ;  file
                mov     dx,cx
                int     21h
                cmp     callflag,0              ;Branch if this routine has
                jne     open_exit               ;  been called before
set_flag:       mov     callflag,1              ;Set flag indicating it has
                call    write_date              ;Write date to the log file
                call    Write_header            ;Write header to the log file
open_exit:      clc                             ;Clear the carry flag and
                ret                             ;  exit

create_file:    mov     ah,3Ch                  ;Attempt to create the
                sub     cx,cx                   ;  file from scratch
                mov     dx,offset logfile
                int     21h
                jc      not_created             ;Branch if it failed

                mov     handle,ax               ;Store the file handle
                mov     bx,ax                   ;Transfer handle to BX
                mov     ah,40h                  ;Write the header to the
                mov     cx,94                   ;  file
                mov     dx,offset header
                int     21h
                jmp     set_flag                ;Exit

not_created:    stc                             ;Set the carry flag and
                ret                             ;  exit
open_file       endp

;****************************************************************************
; CLOSE_FILE closes the log file.
;****************************************************************************

close_file      proc    near
                assume  ds:code
                mov     ah,3Eh
                mov     bx,handle
                int     21h
                ret
close_file      endp

;****************************************************************************
; WRITE_ENTRY writes a 1-line entry to the log file.  On entry, DS:SI
; points to the start time, DS:DI points to the end time, and AL holds the
; latest reentrancy level.
;****************************************************************************

lastlevel       db      ?

write_entry     proc    near
                assume  ds:code
                mov     lastlevel,al            ;Store last level number
                call    write_time              ;Write out the starting
                mov     cx,4                    ;  time
                call    write_spaces            ;Pad it with spaces

                push    si                      ;Save starting time
                mov     si,di                   ;Write out the ending
                call    write_time              ;  time
                mov     cx,3
                call    write_spaces            ;Pad it with spaces
                pop     si                      ;Retrieve start time

                call    write_diff              ;Write elapsed time
                mov     cx,5
                call    write_spaces            ;Pad it with spaces

                mov     al,lastlevel            ;Write out the reentrancy
                mov     bl,1                    ;  level
                call    write_num
                mov     cx,6
                call    write_spaces            ;Pad it with spaces

                mov     al,13                   ;Compute offset into table
                mul     lastlevel               ;  of program names
                mov     dx,ax
                add     dx,offset names
                mov     ah,40h                  ;Write the program name to
                mov     bx,dx                   ;  the log file
                mov     cl,[bx]
                sub     ch,ch
                push    cx
                mov     bx,handle
                inc     dx
                int     21h
                pop     bx

                cmp     execflag,0              ;Exit now if this is called
                je      write_exit              ;  on entry to EXEC

                mov     cx,12                   ;Compute number of spaces
                sub     cx,bx                   ;  to skip
                add     cx,5
                call    write_spaces            ;Output spaces
                mov     bx,offset cmdline       ;Write command line to the
                mov     cl,[bx]                 ;  log file
                sub     ch,ch
                jcxz    write_exit
                mov     ah,40h
                mov     bx,handle
                mov     dx,offset cmdline+1
                int     21h

write_exit:     call    write_crlf              ;End line with CR/LF
                ret
write_entry     endp

;****************************************************************************
; WRITE_TIME writes the time to the log file.  On entry, DS:SI points to
; the time.  The byte at [SI] holds the number of minutes, while the byte
; at [SI+1] holds the number of hours.
;****************************************************************************

colon           db      ":"

write_time      proc    near
                assume  ds:code
                mov     al,[si+1]               ;Write hours to log file
                mov     bl,1
                call    write_num
                mov     ah,40h                  ;Write ":" to separate
                mov     bx,handle               ;  hours and minutes
                mov     cx,1
                mov     dx,offset colon
                int     21h
                mov     al,[si]                 ;Write minutes to log file
                sub     bl,bl
                call    write_num
                ret
write_time      endp

;****************************************************************************
; WRITE_DIFF writes the elapsed time to the log file.  On entry, DS:SI
; points to the start time and DS:SI points to the end time.
;****************************************************************************

delta           db      3 dup (?)

write_diff      proc    near
                assume  ds:code
                mov     al,[di]                 ;Retrieve starting and ending
                mov     bl,[si]                 ;  seconds, minutes, and
                mov     cl,[di+1]               ;  hours
                mov     dl,[si+1]
                mov     ah,[di+2]
                mov     bh,[si+2]
;
; Prepare to compute differences between hours, minutes, and seconds.
;
                cmp     ah,bh                   ;If ending seconds is less
                jae     wrdiff1                 ;  than starting, add 60 to
                add     ah,60                   ;  ending seconds and
                dec     al                      ;  decrement ending minutes

wrdiff1:        cmp     al,bl                   ;If ending minutes is less
                jge     wrdiff2                 ;  than starting, add 60 to
                add     al,60                   ;  ending minutes and
                dec     cl                      ;  decrement ending hours

wrdiff2:        cmp     cl,dl                   ;If ending hours is less
                jge     wrdiff3                 ;  than starting, add 24
                add     cl,24                   ;  to ending hours
;
; Compute the difference and write it to the log file.
;
wrdiff3:        sub     ah,bh                   ;Calculate seconds difference
                mov     delta+2,ah              ;Store it
                sub     al,bl                   ;Calculate minutes difference
                mov     delta,al                ;Store it
                sub     cl,dl                   ;Calculate hours difference
                mov     delta+1,cl              ;Store it

                mov     si,offset delta         ;Write hours and minutes to
                call    write_time              ;  the log file
                mov     ah,40h                  ;Write ":" to log file
                mov     bx,handle
                mov     cx,1
                mov     dx,offset colon
                int     21h
                mov     al,[si+2]               ;Write seconds to log file
                sub     bl,bl
                call    write_num
                ret
write_diff      endp

;****************************************************************************
; WRITE_NUM converts a binary byte value in AL to ASCII and writes it to
; the log file.  Value must be between 0 and 99, inclusive.  On entry, BL=0
; means include leading zeroes; BL=1 means don't include leading zeroes.
;****************************************************************************

fileword        dw      ?

write_num       proc    near
                assume  ds:code
                aam                             ;Convert to BCD in AX
                add     ax,3030h                ;Convert to ASCII
                or      bl,bl                   ;Convert leading zero to
                jz      wrnum1                  ;  space if BL=1
                cmp     ah,30h
                jne     wrnum1
                mov     ah,20h
wrnum1:         xchg    ah,al                   ;Swap bytes
                mov     fileword,ax             ;Store them
                mov     ah,40h                  ;Then write them to the
                mov     bx,handle               ;  log file
                mov     cx,2
                mov     dx,offset fileword
                int     21h
                ret
write_num       endp

;****************************************************************************
; WRITE_CRLF writes a carriage return/line feed to the log file.
;****************************************************************************

crlf            db      13,10

write_crlf      proc    near
                assume  ds:code
                mov     ah,40h
                mov     bx,handle
                mov     cx,2
                mov     dx,offset crlf
                int     21h
                ret
write_crlf      endp

;****************************************************************************
; WRITE_SPACES writes the designated number of spaces to the log file.  On
; entry, CX holds the number of spaces to write.
;****************************************************************************

space           db      32

write_spaces    proc    near
                assume  ds:code
                push    cx                      ;Save counte
                mov     ah,40h                  ;Write one space
                mov     bx,handle
                mov     cx,1
                mov     dx,offset space
                int     21h
                pop     cx                      ;Retrieve count
                loop    write_spaces            ;Loop until done
                ret
write_spaces    endp

;****************************************************************************
; RECORD_TIME records the current time in the slot designated by LEVEL.
;****************************************************************************

record_time     proc    near
                assume  ds:code
                mov     ah,2Ch                  ;Get the current time
                int     21h                     ;  from DOS
                mov     bx,level                ;Compute offset into table
                shl     bx,1                    ;  of start/end times
                shl     bx,1
                mov     word ptr times[bx],cx   ;Store current time
                mov     byte ptr times[bx+2],dh
                ret
record_time     endp

;****************************************************************************
; WRITE_DATE writes the current date to the log file.
;****************************************************************************

century         db      "19"

write_date      proc    near
                assume  ds:code
                call    write_crlf              ;Skip one line
                mov     ah,2Ah                  ;Get today's date from DOS
                int     21h
                push    cx                      ;Save it
                push    dx

                mov     al,dl                   ;Write day of the month to
                sub     bl,bl                   ;  the log file
                call    write_num
                mov     cx,1                    ;Skip a space
                call    write_spaces

                pop     dx                      ;Retrieve month from stack
                dec     dh                      ;Compute offset into month
                mov     cl,dh                   ;  table of the name of
                sub     ch,ch                   ;  the current month
                mov     dx,offset months
                jcxz    wrdate2
wrdate1:        add     dx,3
                loop    wrdate1
wrdate2:        mov     ah,40h                  ;Write month name to the
                mov     bx,handle               ;  log file
                mov     cx,3
                int     21h
                mov     cx,1                    ;Skip a space
                call    write_spaces

                mov     ah,40h                  ;Write "19" portion of year
                mov     cx,2
                mov     dx,offset century
                int     21h
                pop     cx                      ;Retrieve year from stack
                sub     cx,1900                 ;Subtract century portion
                mov     al,cl                   ;Write year to the log file
                sub     bl,bl
                call    write_num
                call    write_crlf              ;Finish with CR/LF pairs
                call    write_crlf
                ret
write_date      endp

;****************************************************************************
; WRITE_HEADER writes the column titles to the log file.
;****************************************************************************

equalsign       db      "="

write_header    proc    near
                assume  ds:code
                mov     ah,40h                  ;Write column titles
                mov     bx,handle
                mov     cx,67
                mov     dx,offset titles
                int     21h

                mov     cx,79                   ;Write row of "=" symbols
wrhead1:        push    cx
                mov     ah,40h
                mov     cx,1
                mov     dx,offset equalsign
                int     21h
                pop     cx
                loop    wrhead1

                call    write_crlf              ;End it with a CR/LF pair
                ret
write_header    endp

;****************************************************************************
; Buffer areas to be used after installation.
;****************************************************************************

names           =       $                       ;Program names
times           =       $ + 130                 ;Start and end times
stackseg        =       $ + 170                 ;Saved values of SS register
stackptr        =       $ + 190                 ;Saved values of SP register
cmdline         =       $ + 210                 ;Command line buffer
lastbyte        =       $ + 338                 ;Last byte

;****************************************************************************
; Data that will be discarded when the program becomes memory-resident.
;****************************************************************************

helpmsg         db      "Maintains a log of the programs run on your PC."
                db      13,10,13,10
                db      "PC-LOG [[d:][path]filename]",13,10,13,10
                db      "  filename  Name of the ASCII log file.",13,10,13,10
                db      "If no file name is specified, the program creates a "
                db      "log file named USAGE.LOG",13,10
                db      "in the root directory of the current drive."
                db      13,10,"$"

errmsg1         db      "Syntax: PC-LOG [[d:][path]filename]",13,10,"$"
errmsg2         db      "Requires DOS 3.0 or higher",13,10,"$"
errmsg3         db      "Program is already installed",13,10,"$"
errmsg4         db      "Invalid time value (max=59:59)",13,10,"$"
errmsg5         db      "Invalid file specification",13,10,"$"
errmsg6         db      "Invalid file specification or file could not be "
                db      "opened",13,10,"$"
errmsg7         db      "Program could not be installed",13,10,"$"

msg1            db      "PC-LOG 1.0 installed",13,10
                db      "Entries will be written to $"

;****************************************************************************
; INIT makes the program resident in memory.
;****************************************************************************

                assume  cs:code,ds:code

init            proc    near
                cld                             ;Clear direction flag
                mov     si,81h                  ;Point SI to command line
                call    scanhelp                ;Scan for "/?" switch
                jnc     checkver                ;Branch if not found
                mov     ah,09h                  ;Display help text and exit
                mov     dx,offset helpmsg       ;  with ERRORLEVEL=0
                int     21h
                mov     ax,4C00h
                int     21h
;
; Check the DOS version and see if the program is already installed.
;
checkver:       mov     dx,offset errmsg2       ;Get the DOS version number
                mov     ah,30h
                int     21h
                mov     byte ptr dosversion,ah  ;Save it
                mov     byte ptr dosversion[1],al
                cmp     al,3                    ;Exit if the version number
                jae     checkins                ;  is less than 3.0

error:          mov     ah,09h                  ;Display error message and
                int     21h                     ;  exit with ERRORLEVEL=1
                mov     ax,4C01h
                int     21h

checkins:       call    check_install           ;Error if the program is
                mov     dx,offset errmsg3       ;  already installed
                jc      error

                mov     ah,19h                  ;Get default drive code with
                int     21h                     ;  DOS function 19H
                add     al,41h                  ;Convert it to ASCII
                mov     logfile,al              ;Store it
;
; Parse the command line.
;
                mov     si,81h                  ;Reset SI
                call    findchar                ;Find first parameter
                jc      checkfile               ;Branch if none found

                mov     bx,si                   ;Save starting address
                call    finddelim               ;Find the end of the string
                mov     byte ptr [si],0         ;Convert string to ASCIIZ
                mov     ah,60h                  ;Call DOS function 60H to
                mov     si,bx                   ;  convert the string to a
                mov     di,offset logfile       ;  full qualified file
                mov     bx,cs                   ;  name
                mov     es,bx
                int     21h
                mov     dx,offset errmsg5       ;Error if carry returns set
                jc      error

checkfile:      call    open_file               ;Make sure the log file can
                mov     dx,offset errmsg6       ;  be opened
                jc      error                   ;Error if it can't
                call    close_file              ;Close the file
                call    record_time             ;Record the time
;
; Install the program.
;
                call    mplex_id                ;Find a multipex ID number
                mov     dx,offset errmsg7       ;Error if none available
                jc      error
                mov     prog_id,ah              ;Save the ID number

                mov     ax,3521h                ;Hook interrupt 21H
                int     21h
                mov     word ptr int21h,bx
                mov     word ptr int21h[2],es
                mov     ax,2521h
                mov     dx,offset dos_int
                int     21h

                mov     ax,352Fh                ;Hook interrupt 2FH
                int     21h
                mov     word ptr int2Fh,bx
                mov     word ptr int2Fh[2],es
                mov     ax,252Fh
                mov     dx,offset mplex_int
                int     21h

                mov     ah,49h                  ;Get the segment address of
                mov     es,ds:[2Ch]             ;  the environment block
                int     21h                     ;  and free the segment

                mov     ah,09                   ;Display message verifying
                mov     dx,offset msg1          ;  the installation
                int     21h
                mov     si,offset logfile       ;Display the name of the
                call    dos_out                 ;  log file

                mov     ax,3100h                ;Terminate with function 31H
                mov     dx,(offset lastbyte - offset code + 15) shr 4
                int     21h
init            endp

;****************************************************************************
; FINDCHAR advances SI to the next non-space or non-comma character.
; On return, carry set indicates EOL was encountered.
;****************************************************************************

findchar        proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Loop if space
                je      findchar
                cmp     al,2Ch                  ;Loop if comma
                je      findchar
                dec     si                      ;Point SI to the character
                cmp     al,0Dh                  ;Exit with carry set if end
                je      eol                     ;  of line is reached

                clc                             ;Clear carry and exit
                ret

eol:            stc                             ;Set carry and exit
                ret
findchar        endp

;****************************************************************************
; FINDDELIM advances SI to the next space or comma character.  On return,
; carry set indicates EOL was encountered.
;****************************************************************************

finddelim       proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Exit if space
                je      fd_exit
                cmp     al,2Ch                  ;Exit if comma
                je      fd_exit
                cmp     al,0Dh                  ;Loop back for more if end
                jne     finddelim               ;  of line isn't reached

                dec     si                      ;Set carry and exit
                stc
                ret

fd_exit:        dec     si                      ;Clear carry and exit
                clc
                ret
finddelim       endp

;****************************************************************************
; SCANHELP scans the command line for a /? switch.  If found, carry returns
; set and SI contains its offset.  If not found, carry returns clear.
;****************************************************************************

scanhelp        proc    near
                push    si                      ;Save SI
scanloop:       lodsb                           ;Get a character
                cmp     al,0Dh                  ;Exit if end of line
                je      scan_exit
                cmp     al,"?"                  ;Loop if not "?"
                jne     scanloop
                cmp     byte ptr [si-2],"/"     ;Loop if not "/"
                jne     scanloop

                add     sp,2                    ;Clear the stack
                sub     si,2                    ;Adjust SI
                stc                             ;Set carry and exit
                ret

scan_exit:      pop     si                      ;Restore SI
                clc                             ;Clear carry and exit
                ret
scanhelp        endp

;****************************************************************************
; CHECK_INSTALL returns carry set if the program is already installed,
; carry clear if it's not.  If carry returns set, AH holds the program's
; multiplex ID number.
;****************************************************************************

check_install   proc    near
                mov     ax,8000h                ;Initialize AH and AL
                mov     cx,80h                  ;Initialize count

chinst1:        push    ax                      ;Save AX and CX
                push    cx
                sub     di,di                   ;Set ES and DI to 0
                mov     es,di
                int     2Fh                     ;Interrupt 2Fh
                cmp     al,0FFh                 ;Nothing here if AL isn't
                jne     chinst2                 ;  equal to FFH

                mov     si,offset signature     ;See if program signature
                mov     cx,12                   ;  appears at the address
                repe    cmpsb                   ;  returned in ES:DI
                jne     chinst2                 ;Branch if it does not

                pop     cx                      ;Clear the stack and exit
                pop     ax                      ;  with carry set
                stc
                ret

chinst2:        pop     cx                      ;Retrieve AX and CX
                pop     ax
                inc     ah                      ;Next multiplex ID
                loop    chinst1                 ;Loop until done

                clc                             ;Exit with carry clear
                ret
check_install   endp

;****************************************************************************
; MPLEX_ID searches for an unused multiplex ID number.  If one is found,
; it is returned in AH with carry clear.  Carry set means no multiplex
; ID numbers are currently available.
;****************************************************************************

mplex_id        proc    near
                mov     ax,8000h                ;Initialize AH and AL
                mov     cx,80h                  ;Initialize count

mxid1:          push    ax                      ;Save AX and CX
                push    cx
                int     2Fh                     ;Interrupt 2Fh
                or      al,al                   ;Branch if AL=0
                jz      mxid2
                pop     cx                      ;Retrieve AX and CX
                pop     ax
                inc     ah                      ;Increment ID number
                loop    mxid1                   ;Loop until done

                stc                             ;Exit with carry set
                ret

mxid2:          pop     cx                      ;Clear the stack and exit
                pop     ax                      ;  with carry clear
                clc
                ret
mplex_id        endp

;****************************************************************************
; ASC2BIN converts a decimal number entered in ASCII form into a binary
; value in AL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

asc2bin         proc    near
                sub     ax,ax                   ;Initialize registers
                sub     bh,bh
                mov     dl,10

a2b_loop:       mov     bl,[si]                 ;Get a character
                inc     si
                cmp     bl,20h                  ;Exit if space
                je      a2b_exit
                cmp     bl,3Ah                  ;Exit if colon
                je      a2b_exit
                cmp     bl,0Dh                  ;Exit if carriage return
                je      a2b_exit

                cmp     bl,"0"                  ;Error if character is not
                jb      a2b_error               ;  a number
                cmp     bl,"9"
                ja      a2b_error

                mul     dl                      ;Multiply the value in AL by
                jc      a2b_error               ;  10 and exit on overflow
                sub     bl,30h                  ;ASCII => binary
                add     ax,bx                   ;Add latest value to AX
                cmp     ax,255                  ;Error if sum > 255
                jna     a2b_loop                ;Loop back for more

a2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

a2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
asc2bin         endp

;****************************************************************************
; DOS_OUT displays the ASCIIZ string pointed to by DS:SI.
;****************************************************************************

dos_out         proc    near
                lodsb                           ;Get a character
                or      al,al                   ;Exit if it's zero
                jz      out_exit
                mov     ah,02h                  ;Output it using DOS
                mov     dl,al                   ;  function 02H
                int     21h
                jmp     dos_out                 ;Loop until done
out_exit:       ret
dos_out         endp

code            ends
                end     begin
