;****************************************************************************
; SAVER blanks the screen after a specified amount of time has elapsed
; with no mouse or keyboard activity.  Its syntax is
;
;       SAVER [mm]
;
; where "mm" is the length of time that must pass before the screen is
; blanked (in minutes).  Valid values range from 1 to 60.  Typing SAVER 0
; disables SAVER until a non-zero value is entered.  You may also blank the
; screen on demand by pressing both Shift keys.
;
; SAVER must be installed after the mouse driver if it's to monitor
; the actions of the mouse.  Once the screen is blanked, pressing any key,
; moving the mouse, or pressing a mouse button will restore video output.
;****************************************************************************

code            segment
                assume  cs:code
                org     100h
begin:          jmp     init

signature       db      0,1,2,"SAVER",2,1,0     ;Program signature
mouseflag       db      0                       ;0=No mouse installed
scancode        db      ?                       ;Keyboard scan code
blanked         db      0                       ;1=Screen blanked
prog_id         db      ?                       ;Multiplex ID number
adapter         db      ?                       ;Video adapter ID
ac_addr         dw      ?                       ;Attribute controller address
mcount          dw      0                       ;Nested INT 33h calls
time            dw      5460                    ;Countdown timer reset value
count           dw      5460                    ;Countdown timer
mset            db      2Ch,28h,2Dh,29h         ;CGA/MDA mode select values
                db      2Ah,2Eh,1Eh,29h

int08h          dd      ?                       ;Interrupt 08H vector
int09h          dd      ?                       ;Interrupt 09H vector
int2Fh          dd      ?                       ;Interrupt 2FH vector
int33h          dd      ?                       ;Interrupt 33H vector

buttons         dw      ?                       ;Mouse button status
vertical        dw      ?                       ;Mouse horizontal position
horizontal      dw      ?                       ;Mouse vertical position

;****************************************************************************
; TIMER_INT handles interrupt 08H.
;****************************************************************************

timer_int       proc    far
                pushf                           ;Push FLAGS register
                call    cs:[int08h]             ;Call previous handler
                sti                             ;Enable interrupts
                cmp     cs:[time],0             ;Exit if countdown timer is
                je      timer_exit              ;  disabled
                cmp     cs:[blanked],1          ;Branch if the screen is
                je      timer3                  ;  already blanked
;
; Reset the countdown timer if mouse has moved, decrement it if it has not.
;
                cmp     cs:[mouseflag],0        ;Branch if no mouse installed
                je      timer2
                call    check_mouse             ;See if the mouse has moved
                jc      timer4                  ;Branch if it has

timer2:         dec     cs:[count]              ;Decrement countdown timer
                cmp     cs:[count],0            ;Has the countdown reached 0?
                jne     timer_exit              ;Branch if not
                call    disable_video           ;Blank the screen
                iret                            ;Return from interrupt
;
; Unblank the screen if the mouse has moved.
;
timer3:         cmp     cs:[mouseflag],0        ;Exit if no mouse installed
                je      timer_exit
                call    check_mouse             ;See if the mouse has moved
                jnc     timer_exit              ;Unblank the screen if it
                call    enable_video            ;  has
timer4:         push    cs:[time]               ;Reset countdown timer
                pop     cs:[count]
timer_exit:     iret                            ;Return from interrupt
timer_int       endp

;****************************************************************************
; KB_INT handles interrupt 09H.
;****************************************************************************

kb_int          proc    far
                pushf                           ;Save FLAGS
                push    ax                      ;Save AX
                in      al,60h                  ;Read the scan code
                mov     cs:[scancode],al        ;Save it
                pop     ax                      ;Restore AX
                popf                            ;Restore FLAGS
                pushf                           ;Push FLAGS
                call    cs:[int09h]             ;Call previous handler
                sti                             ;Enable interrupts

                test    cs:[scancode],80h       ;Exit if high bit of scan
                jnz     kb_exit                 ;  code is set

                push    ax                      ;Save AX and ES
                push    es
                mov     ax,40h                  ;Point ES to the BIOS
                mov     es,ax                   ;  Data Area
                mov     al,es:[17h]             ;Get keyboard flags
                and     al,03h                  ;Zero the upper 6 bits
                cmp     al,03h                  ;Are the Shift keys pressed?
                pop     es                      ;Restore AX and ES
                pop     ax
                jne     kb2                     ;Branch if they're not
                call    disable_video           ;Blank the screen
                iret                            ;Return from interrupt

kb2:            push    cs:[time]               ;Reset countdown timer
                pop     cs:[count]
                cmp     cs:[blanked],0          ;Is the screen blanked?
                je      kb_exit                 ;If not, then exit
                call    enable_video            ;Unblank the screen
kb_exit:        iret                            ;Return from interrupt
kb_int          endp

;****************************************************************************
; MPLEX_INT handles interrupt 2FH.  If, on entry, AH is set to SAVER's
; multiplex ID number, MPLEX_INT uses the value in AL as a function code.
; The functions supported are:
;
;    00H    Returns FFH in AL to indicate the program is installed
;           and returns the address of its signature in ES:DI.
;
;    01H    Sets the countdown timer to the value passed in BX.
;           A value of 0 temporarily disables the timer.
;
;    02H    Returns the timer reset value in AX.
;****************************************************************************

mplex_int       proc    far
                pushf                           ;Save FLAGS register
                cmp     ah,cs:[prog_id]         ;Branch if AH holds the
                je      mplex2                  ;  multiplex ID
                popf                            ;Restore FLAGS
                jmp     cs:[int2Fh]             ;Pass the interrupt on
;
; Function 00H verifies that the program is installed.
;               
mplex2:         popf                            ;Restore FLAGS
                or      al,al                   ;Branch if function code
                jnz     mplex3                  ;  is other than 00H
                mov     al,0FFh                 ;Set AL to FFH
                push    cs                      ;Point ES:DI to the program
                pop     es                      ;  signature
                mov     di,offset signature
                iret                            ;Return from interrupt
;
; Function 01H sets the countdown timer.
;
mplex3:         cmp     al,01h                  ;Branch if function code
                jne     mplex4                  ;  is other than 01H
                mov     cs:[time],bx            ;Load new timer value
                mov     cs:[count],bx
                iret                            ;Return from interrupt
;
; Function 02H returns the timer reset value.
;
mplex4:         mov     ax,cs:[time]            ;Get value in AX
                iret                            ;Return from interrupt
mplex_int       endp

;****************************************************************************
; MOUSE_INT handles interrupt 33H.
;****************************************************************************

mouse_int       proc    far
                pushf                           ;Save FLAGS
                inc     cs:[mcount]             ;Increment count
                popf                            ;Restore FLAGS
                pushf                           ;Push FLAGS
                call    cs:[int33h]             ;Call mouse driver
                pushf                           ;Save FLAGS
                dec     cs:[mcount]             ;Decrement count
                popf                            ;Restore FLAGS
                ret     2                       ;Return from interrupt
mouse_int       endp

;****************************************************************************
; DISABLE_VIDEO disables the video output signal.
;****************************************************************************

disable_video   proc    near
                push    ax                      ;Save registers
                push    bx
                push    dx
                push    es
                cli                             ;Disable interrupts
                cmp     cs:[adapter],2          ;Branch if EGA or VGA
                jae     dv2
;
; Blank a CGA or MDA screen.
;
                mov     ax,40h                  ;Point ES to the BIOS
                mov     es,ax                   ;  Data Area
                mov     al,es:[49h]             ;Get mode number in AL
                sub     ah,ah                   ;Convert it to a word and
                mov     bx,ax                   ;  transfer to BX
                mov     al,cs:[bx+offset mset]  ;Get mode select value
                and     al,0F7h                 ;Clear bit 3
                mov     dx,es:[63h]             ;Get CRTC base address
                add     dx,4                    ;Compute mode select address
                out     dx,al                   ;OUT the value
                jmp     short dv_exit           ;Exit
;
; Blank an EGA or VGA screen.
;
dv2:            mov     dx,cs:[ac_addr]         ;Get register address
                in      al,dx                   ;Reset addr/data flip-flop
                sub     al,al                   ;Clear bit 5 of the address
                mov     dx,3C0h                 ;  register
                out     dx,al
                mov     dx,cs:[ac_addr]         ;Reset the flip-flop again
                in      al,dx

dv_exit:        mov     cs:[blanked],1          ;Set screen flag
                sti                             ;Enable interrupts
                pop     es                      ;Restore registers and exit
                pop     dx
                pop     bx
                pop     ax
                ret
disable_video   endp

;****************************************************************************
; ENABLE_VIDEO enables the video output signal.
;****************************************************************************

enable_video    proc    near
                push    ax                      ;Save registers
                push    bx
                push    dx
                push    es
                cli                             ;Disable interrupts
                cmp     cs:[adapter],2          ;Branch if EGA or VGA
                jae     ev2
;
; Unblank a CGA or MDA screen.
;
                mov     ax,40h                  ;Point ES to the BIOS
                mov     es,ax                   ;  Data Area
                mov     al,es:[49h]             ;Get mode number in AL
                sub     ah,ah                   ;Convert it to a word and
                mov     bx,ax                   ;  transfer to BX
                mov     al,cs:[bx+offset mset]  ;Get mode select value
                mov     dx,es:[63h]             ;Get CRTC base address
                add     dx,4                    ;Compute mode select address
                out     dx,al                   ;OUT the value
                jmp     short ev_exit           ;Exit
;
; Unblank an EGA or VGA screen.
;
ev2:            mov     dx,cs:[ac_addr]         ;Get register address
                in      al,dx                   ;Reset addr/data flip-flop
                mov     al,20h                  ;Set bit 5 of the address
                mov     dx,3C0h                 ;  register
                out     dx,al
                mov     dx,cs:[ac_addr]         ;Reset the flip-flop again
                in      al,dx

ev_exit:        mov     cs:[blanked],0          ;Clear screen flag
                sti                             ;Enable interrupts
                pop     es                      ;Restore registers and exit
                pop     dx
                pop     bx
                pop     ax
                ret
enable_video    endp

;****************************************************************************
; CHECK_MOUSE returns with the carry flag clear if the mouse status has not
; changed since the last call, set if it has.
;****************************************************************************

check_mouse     proc    near
                cmp     cs:[mcount],0           ;Exit now if another INT
                je      cm0                     ;  33H is in progress
                clc
                ret

cm0:            push    ax                      ;Save registers
                push    bx
                push    cx
                push    dx

                mov     ax,03h                  ;Get information with
                int     33h                     ;  mouse function 03H
                cmp     bx,cs:[buttons]         ;Branch and set the carry
                jne     cm2                     ;  flag if anything has
                cmp     cx,cs:[horizontal]      ;  changed since the
                jne     cm2                     ;  last call
                cmp     dx,cs:[vertical]
                jne     cm2

cm1:            clc                             ;Clear the carry flag
                jmp     short cm3               ;Jump to exit

cm2:            mov     cs:[buttons],bx         ;Save return values
                mov     cs:[horizontal],cx
                mov     cs:[vertical],dx
                stc                             ;Set the carry flag
cm3:            pop     dx                      ;Restore registers and
                pop     cx                      ;  exit
                pop     bx
                pop     ax
                ret
check_mouse     endp

;****************************************************************************
; Data that will be discarded when the program becomes memory-resident.
;****************************************************************************

helpmsg         db      "Blanks the screen after a period of mouse or "
                db      "keyboard inactivity.",13,10,13,10
                db      "SAVER [mm]",13,10,13,10
                db      "  mm  Number of minutes before blanking occurs "
                db      "(1 to 60, default=5).",13,10,13,10
                db      "Typing SAVER 0 disables SAVER until a non-zero "
                db      "value is entered.",13,10,"$"

errmsg1         db      "Syntax: SAVER [mm]",13,10,"$"
errmsg2         db      "Requires DOS 3.0 or higher",13,10,"$"
errmsg3         db      "Minutes must be 60 or less",13,10,"$"
errmsg4         db      "Program could not be installed",13,10,"$"

msg1            db      "SAVER 1.0 installed",13,10,"$"
msg2            db      "Countdown timer set to $"
msg3            db      " minute(s)",13,10,"$"
msg4            db      "Countdown timer disabled",13,10,"$"

parmflag        db      0                       ;0=No command line parameter
minutes         db      5                       ;Minutes in countdown timer

;****************************************************************************
; INIT makes the program resident in memory.
;****************************************************************************

                assume  cs:code,ds:code

init            proc    near
                cld                             ;Clear direction flag
                mov     si,81h                  ;Point SI to command line
                call    scanhelp                ;Scan for "/?" switch
                jnc     checkver                ;Branch if not found
                mov     ah,09h                  ;Display help text and exit
                mov     dx,offset helpmsg       ;  with ERRORLEVEL=0
                int     21h
                mov     ax,4C00h
                int     21h
;
; Check the DOS version and parse the command line.
;
checkver:       mov     dx,offset errmsg2       ;Exit if DOS version
                mov     ah,30h                  ;  is less than 3.0
                int     21h
                cmp     al,3
                jae     parse

error:          mov     ah,09h                  ;Display error message and
                int     21h                     ;  exit with ERRORLEVEL=1
                mov     ax,4C01h
                int     21h

parse:          call    findchar                ;Find parameter (if any)
                jc      checkprog               ;Branch if none found
                mov     parmflag,1              ;Set PARMFLAG to 1
                call    asc2bin                 ;Convert ASCII to binary
                mov     dx,offset errmsg3       ;Exit on error
                jc      error
                cmp     al,60                   ;Error if value entered is
                ja      error                   ;  greater than 60

                mov     minutes,al              ;Save number of minutes
                sub     ah,ah                   ;Multiply value entered by
                mov     bx,1092                 ;  1,092 and save it away
                mul     bx
                mov     time,ax
                mov     count,ax

checkprog:      call    check_install           ;See if program is installed
                jnc     install                 ;Branch if it's not
;
; Communicate with an installed copy of the code.
;
                cmp     parmflag,0              ;Branch if no time was
                jne     cp2                     ;  entered
                mov     al,02h                  ;Get counter reset value
                int     2Fh                     ;  from installed code
                sub     dx,dx                   ;Divide it by 1,092
                mov     bx,1092
                div     bx
                mov     minutes,al              ;Save it
                jmp     short cp3               ;Branch and display it

cp2:            mov     al,01h                  ;AL = Function code
                mov     bx,time                 ;BX = Reset value
                int     2Fh                     ;Set the value

cp3:            call    showtime                ;Display minutes
                mov     ax,4C00h                ;Exit with ERRORLEVEL=0
                int     21h
;
; Install the program.
;
install:        call    mplex_id                ;Find a multiplex ID number
                mov     dx,offset errmsg4       ;Error if none available
                jc      error
                mov     prog_id,ah              ;Save the ID number

                call    video_id                ;ID the video adapter
                mov     adapter,al              ;Save ID value
                mov     ax,40h                  ;Get and save the address of
                mov     es,ax                   ;  the attribute controller
                mov     ax,es:[63h]             ;  for EGAs and VGAs
                add     ax,6
                mov     ac_addr,ax

                mov     ah,00h                  ;Check for mouse driver
                int     33h
                or      ax,ax                   ;Branch if no mouse
                jz      nomouse1
                mov     mouseflag,1             ;Set flag for mouse

nomouse1:       mov     ax,3508h                ;Hook interrupt 08H
                int     21h
                mov     word ptr int08h,bx
                mov     word ptr int08h[2],es
                mov     ax,2508h
                mov     dx,offset timer_int
                int     21h

                mov     ax,3509h                ;Hook interrupt 09H
                int     21h
                mov     word ptr int09h,bx
                mov     word ptr int09h[2],es
                mov     ax,2509h
                mov     dx,offset kb_int
                int     21h

                mov     ax,352Fh                ;Hook interrupt 2FH
                int     21h
                mov     word ptr int2Fh,bx
                mov     word ptr int2Fh[2],es
                mov     ax,252Fh
                mov     dx,offset mplex_int
                int     21h

                cmp     mouseflag,0             ;Branch if no mouse
                je      nomouse2                ;  installed
                mov     ax,3533h                ;Hook interrupt 33H
                int     21h
                mov     word ptr int33h,bx
                mov     word ptr int33h[2],es
                mov     ax,2533h
                mov     dx,offset mouse_int
                int     21h

nomouse2:       mov     ah,49h                  ;Get the segment address of
                mov     es,ds:[2Ch]             ;  the environment block
                int     21h                     ;  and free the segment

                mov     ah,09h                  ;Display message verifying
                mov     dx,offset msg1          ;  the installation
                int     21h
                call    showtime                ;Display minutes

                mov     ax,3100h                ;Terminate with function 31H
                mov     dx,(offset helpmsg - offset code + 15) shr 4
                int     21h
init            endp

;****************************************************************************
; FINDCHAR advances SI to the next non-space or non-comma character.
; On return, carry set indicates EOL was encountered.
;****************************************************************************

findchar        proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Loop if space
                je      findchar
                cmp     al,2Ch                  ;Loop if comma
                je      findchar
                dec     si                      ;Point SI to the character
                cmp     al,0Dh                  ;Exit with carry set if end
                je      eol                     ;  of line is reached

                clc                             ;Clear carry and exit
                ret

eol:            stc                             ;Set carry and exit
                ret
findchar        endp

;****************************************************************************
; SCANHELP scans the command line for a /? switch.  If found, carry returns
; set and SI contains its offset.  If not found, carry returns clear.
;****************************************************************************

scanhelp        proc    near
                push    si                      ;Save SI
scanloop:       lodsb                           ;Get a character
                cmp     al,0Dh                  ;Exit if end of line
                je      scan_exit
                cmp     al,"?"                  ;Loop if not "?"
                jne     scanloop
                cmp     byte ptr [si-2],"/"     ;Loop if not "/"
                jne     scanloop

                add     sp,2                    ;Clear the stack
                sub     si,2                    ;Adjust SI
                stc                             ;Set carry and exit
                ret

scan_exit:      pop     si                      ;Restore SI
                clc                             ;Clear carry and exit
                ret
scanhelp        endp

;****************************************************************************
; CHECK_INSTALL returns carry set if the program is already installed,
; carry clear if it's not.  If carry returns set, AH holds the program's
; multiplex ID number.
;****************************************************************************

check_install   proc    near
                mov     ax,8000h                ;Initialize AH and AL
                mov     cx,80h                  ;Initialize count

chinst1:        push    ax                      ;Save AX and CX
                push    cx
                sub     di,di                   ;Set ES and DI to 0
                mov     es,di
                int     2Fh                     ;Interrupt 2Fh
                cmp     al,0FFh                 ;Nothing here if AL isn't
                jne     chinst2                 ;  equal to FFH

                mov     si,offset signature     ;See if program signature
                mov     cx,11                   ;  appears at the address
                repe    cmpsb                   ;  returned in ES:DI
                jne     chinst2                 ;Branch if it does not

                pop     cx                      ;Clear the stack and exit
                pop     ax                      ;  with carry set
                stc
                ret

chinst2:        pop     cx                      ;Retrieve AX and CX
                pop     ax
                inc     ah                      ;Next multiplex ID
                loop    chinst1                 ;Loop until done

                clc                             ;Exit with carry clear
                ret
check_install   endp

;****************************************************************************
; MPLEX_ID searches for an unused multiplex ID number.  If one is found,
; it is returned in AH with carry clear.  Carry set means no multiplex
; ID numbers are currently available.
;****************************************************************************

mplex_id        proc    near
                mov     ax,8000h                ;Initialize AH and AL
                mov     cx,80h                  ;Initialize count

mxid1:          push    ax                      ;Save AX and CX
                push    cx
                int     2Fh                     ;Interrupt 2Fh
                or      al,al                   ;Branch if AL=0
                jz      mxid2
                pop     cx                      ;Retrieve AX and CX
                pop     ax
                inc     ah                      ;Increment ID number
                loop    mxid1                   ;Loop until done

                stc                             ;Exit with carry set
                ret

mxid2:          pop     cx                      ;Clear the stack and exit
                pop     ax                      ;  with carry clear
                clc
                ret
mplex_id        endp

;****************************************************************************
; VIDEO_ID returns 2 in AL if the video adapter is an EGA or VGA, 1 if it's
; a CGA, or 0 if it's an MDA or other monochrome adapter.
;****************************************************************************

video_id        proc    near
                mov     ah,12h                  ;Test for EGA/VGA
                mov     bl,10h
                int     10h
                mov     al,2                    ;Initialize AL
                cmp     bl,10h                  ;EGA/VGA installed if BL
                jne     video_exit              ;  returned changed
                dec     al                      ;Set AL to 1
                mov     bx,40h                  ;Point ES to BIOS Data Area
                mov     es,bx
                test    byte ptr es:[63h],40h   ;Test 6845 address word
                jnz     video_exit              ;CGA installed if bit 6 set
                dec     al                      ;Set AL to 0 and exit
video_exit:     ret
video_id        endp

;****************************************************************************
; SHOWTIME displays the length of the countdown in minutes.
;****************************************************************************

showtime        proc    near
                cmp     minutes,0               ;Branch if minutes is
                jne     st2                     ;  not 0
                mov     ah,09h                  ;Display "Countdown timer
                mov     dx,offset msg4          ;  disabled" message
                int     21h
                ret

st2:            mov     ah,09h                  ;Display "Countdown timer"
                mov     dx,offset msg2          ;  part of message
                int     21h
                mov     al,minutes              ;Display the number of
                sub     ah,ah                   ;  minutes
                call    bin2asc
                mov     ah,09h                  ;Display "minute(s)"
                mov     dx,offset msg3
                int     21h
                ret
showtime        endp

;****************************************************************************
; ASC2BIN converts a decimal number entered in ASCII form into a binary
; value in AL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

asc2bin         proc    near
                sub     ax,ax                   ;Initialize registers
                sub     bh,bh
                mov     dl,10

a2b_loop:       mov     bl,[si]                 ;Get a character
                inc     si
                cmp     bl,20h                  ;Exit if space
                je      a2b_exit
                cmp     bl,2Ch                  ;Exit if comma
                je      a2b_exit
                cmp     bl,0Dh                  ;Exit if carriage return
                je      a2b_exit

                cmp     bl,"0"                  ;Error if character is not
                jb      a2b_error               ;  a number
                cmp     bl,"9"
                ja      a2b_error

                mul     dl                      ;Multiply the value in AL by
                jc      a2b_error               ;  10 and exit on overflow
                sub     bl,30h                  ;ASCII => binary
                add     ax,bx                   ;Add latest value to AX
                cmp     ax,255                  ;Error if sum > 255
                jna     a2b_loop                ;Loop back for more

a2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

a2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
asc2bin         endp

;****************************************************************************
; BIN2ASC converts a binary value in AX to ASCII form and displays it.
;****************************************************************************

bin2asc         proc    near
                mov     bx,10                   ;Initialize divisor word and
                xor     cx,cx                   ;  digit counter
b2a1:           inc     cx                      ;Increment digit count
                xor     dx,dx                   ;Divide by 10
                div     bx
                push    dx                      ;Save remainder on stack
                or      ax,ax                   ;Loop until quotient is zero
                jnz     b2a1
b2a2:           pop     dx                      ;Retrieve a digit from stack
                add     dl,30h                  ;Convert it to ASCII
                mov     ah,2                    ;Display it
                int     21h
                loop    b2a2                    ;Loop until done
                ret
bin2asc         endp

code            ends
                end     begin
