;****************************************************************************
; TEXTOUT echoes a string to the screen using the attribute specified.
; Its syntax is:
;
;       TEXTOUT attr "string"
;
; where "attr" is a hex number indicating what color attribute is to
; be used, and "string" is the text to be output.  For example, the
; command
;
;       TEXTOUT 1E "Hello, world"
;
; displays "Hello, world" in yellow on blue at the current cursor position.
; The cursor can be positioned with the separate SETPOS utility.
;****************************************************************************

code            segment
                assume  cs:code,ds:code
                org     100h
begin:          jmp     short main

msg1            db      13,10,"Syntax: TEXTOUT nn ",34,"string",34,13,10,"$"
msg2            db      13,10,"Invalid parameter",13,10,"$"
start           dw      ?
attr            db      ?

;****************************************************************************
; Procedure MAIN
;****************************************************************************

main            proc    near
                cld                             ;Clear direction flag
                mov     si,81h                  ;Point SI to command line
                call    findchar                ;Advance to first character
                jnc     getattr                 ;Branch if found

error1:         mov     dx,offset msg1          ;Display error message
error2:         mov     ah,09h
                int     21h
                mov     ax,4C01h                ;Exit with ERRORLEVEL=1
                int     21h
;
; Parse the strings entered on the command line.
;
getattr:        call    hex2bin                 ;Convert the attribute
                mov     dx,offset msg2          ;  value to binary and
                jc      error2                  ;  store it for later
                mov     attr,bl

                call    findchar                ;Advance to next character
                jc      error1                  ;Error if EOL encountered
                cmp     al,22h                  ;Error if character is not
                jne     error1                  ;  a quotation mark
                inc     si                      ;Advance SI to string
                mov     start,si                ;Save string address

mloop:          lodsb                           ;Get a character
                cmp     al,0Dh                  ;Error if EOL encountered
                je      error1
                cmp     al,22h                  ;Loop back if it's not a
                jne     mloop                   ;  quotation mark

                mov     byte ptr [si-1],0       ;Mark end of string
;
; Output the string using BIOS interrupt 10H video I/O functions.
;
                mov     ah,0Fh                  ;Get active page number
                int     10h                     ;  in BH
                mov     ah,03h                  ;Get current cursor position
                int     10h                     ;  in DH and DL
                mov     bl,attr                 ;Get attribute in BL
                mov     si,start                ;Point SI to string

outloop:        lodsb                           ;Get a character
                or      al,al                   ;Exit if zero
                jz      exit
                mov     ah,09h                  ;Output the character
                mov     cx,1
                int     10h
                mov     ah,02h                  ;Advance the cursor for
                inc     dl                      ;  next write
                int     10h
                jmp     outloop                 ;Loop back for more

exit:           mov     ax,4C00h                ;Exit with ERRORLEVEL=0
                int     21h
main            endp

;****************************************************************************
; FINDCHAR advances SI to the next non-space or non-comma character.
; On return, carry set indicates EOL was encountered.
;****************************************************************************

findchar        proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Loop if space
                je      findchar
                cmp     al,2Ch                  ;Loop if comma
                je      findchar
                dec     si                      ;Point SI to the character
                cmp     al,0Dh                  ;Exit with carry set if end
                je      eol                     ;  of line is reached

                clc                             ;Clear carry and exit
                ret

eol:            stc                             ;Set carry and exit
                ret
findchar        endp

;****************************************************************************
; HEX2BIN converts a hex number entered in ASCII form into a binary
; value in BL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

hex2bin         proc    near
                sub     ah,ah                   ;Initialize registers
                sub     bx,bx

h2b_loop:       lodsb                           ;Get a character
                cmp     al,20h                  ;Exit if space
                je      h2b_exit
                cmp     al,2Ch                  ;Exit if comma
                je      h2b_exit
                cmp     al,0Dh                  ;Exit if carriage return
                je      h2b_exit

                cmp     al,"0"                  ;Check for digits "0"
                jb      h2b_error               ;  through "9"
                cmp     al,"9"
                ja      h2b2

                sub     al,30h                  ;ASCII => binary
h2b1:           mov     cl,4                    ;Multiply BX by 16 and
                shl     bx,cl                   ;  add the latest
                add     bx,ax                   ;  digit
                cmp     bx,255                  ;Error if sum > 255
                ja      h2b_error
                jmp     h2b_loop                ;Loop back for more

h2b2:           and     al,0DFh                 ;Capitalize the letter
                cmp     al,"A"                  ;Check range and exit if
                jb      h2b_error               ;  not "A" through "F"
                cmp     al,"F"
                ja      h2b_error               
                sub     al,37h                  ;ASCII => binary
                jmp     h2b1                    ;Finish and loop back

h2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

h2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
hex2bin         endp

;****************************************************************************
; ASC2BIN converts a decimal number entered in ASCII form into a binary
; value in AL.  Carry set on return indicates that an error occurred in
; the conversion.
;****************************************************************************

asc2bin         proc    near
                sub     ax,ax                   ;Initialize registers
                sub     bh,bh
                mov     dl,10

a2b_loop:       mov     bl,[si]                 ;Get a character
                inc     si
                cmp     bl,20h                  ;Exit if space
                je      a2b_exit
                cmp     bl,2Ch                  ;Exit if comma
                je      a2b_exit
                cmp     bl,0Dh                  ;Exit if carriage return
                je      a2b_exit

                cmp     bl,"0"                  ;Error if character is not
                jb      a2b_error               ;  a number
                cmp     bl,"9"
                ja      a2b_error

                mul     dl                      ;Multiply the value in AL by
                jc      a2b_error               ;  10 and exit on overflow
                sub     bl,30h                  ;ASCII => binary
                add     ax,bx                   ;Add latest value to AX
                cmp     ax,255                  ;Error if sum > 255
                jna     a2b_loop                ;Loop back for more

a2b_error:      dec     si                      ;Set carry and exit
                stc
                ret

a2b_exit:       dec     si                      ;Clear carry and exit
                clc
                ret
asc2bin         endp

code            ends
                end     begin
