;****************************************************************************
; TOGGLE turns Caps Lock, Num Lock, and Scroll Lock on or off.  Its syntax
; is
;
;       TOGGLE [+C|-C] [+N|-N] [+S|-S] [/L]
;
; where +C or -C turns Caps Lock on or off, +N or -N turns Num Lock on or
; off, and +S or -S turns Scroll Lock on or off.  /L turns the keyboard
; indicator lights on or off also (on most PCs, the lights will toggle
; without the /L switch).  Running TOGGLE with no switches displays the
; current settings of Caps Lock, Num Lock, and Scroll Lock.
;****************************************************************************

code            segment
                assume  cs:code,ds:code
                org     100h
begin:          jmp     main

helpmsg         db      "Turns Caps Lock, Num Lock, and Scroll Lock on "
                db      "or off.",13,10,13,10
                db      "TOGGLE [+C|-C] [+N|-N] [+S|-S] [/L]",13,10,13,10
                db      "  +C  Turns Caps Lock on.",13,10
                db      "  -C  Turns Caps Lock off.",13,10
                db      "  +N  Turns Num Lock on.",13,10
                db      "  -N  Turns Num Lock off.",13,10
                db      "  +S  Turns Scroll Lock on.",13,10
                db      "  -S  Turns Scroll Lock off.",13,10
                db      "  /L  Toggles keyboard indicator lights.",13,10
                db      13,10
                db      "On most PCs, the keyboard indicator lights will "
                db      "toggle without the /L switch.",13,10
                db      "/L is included for those that do not.",13,10,"$"

errmsg1         db      "Syntax: Toggle [+C|-C] [+N|-N] [+S|-S] [/L]"
                db      13,10,"$"

msg1            db      13,10,"     Caps Lock   : $"
msg2            db      "     Num Lock    : $"
msg3            db      "     Scroll Lock : $"
msg4            db      "Off",13,10,"$"
msg5            db      "On",13,10,"$"

syncflag        db      0                       ;0=No /L switch

;****************************************************************************
; Procedure MAIN
;****************************************************************************

main            proc    near
                cld                             ;Clear direction flag
                mov     si,81h                  ;Point SI to command line
                call    scanhelp                ;Scan for "/?" switch
                jnc     parse                   ;Branch if not found
                mov     ah,09h                  ;Display help text and exit
                mov     dx,offset helpmsg       ;  with ERRORLEVEL=0
                int     21h
                mov     ax,4C01h
                int     21h
;
; Parse the command line and toggle keyboard states.
;
parse:          mov     ax,40h                  ;Point ES to the BIOS
                mov     es,ax                   ;  Data Area
nextparm:       call    findchar                ;Advance SI to next entry
                jc      checklites              ;Branch if end of line
                lodsb                           ;Get the character
                cmp     al,"+"                  ;Branch if "+"
                je      toggle
                cmp     al,"-"                  ;Branch if "-"
                je      toggle
                cmp     al,"/"                  ;Branch if "/"
                je      slash

error:          mov     ah,09h                  ;Display error message and
                mov     dx,offset errmsg1       ;  exit with ERRORLEVEL=1
                int     21h
                mov     ax,4C01h
                int     21h

slash:          lodsb                           ;Get next character
                and     al,0DFh                 ;Capitalize it
                cmp     al,"L"                  ;Error if not equal to "L"
                jne     error
                mov     syncflag,1              ;Set SYNCFLAG
                jmp     nextparm                ;Loop back for more

toggle:         lodsb                           ;Get next character
                and     al,0DFh                 ;Capitalize it
                mov     ah,40h                  ;Set AH for Caps Lock
                cmp     al,"C"                  ;Branch if "C" was entered
                je      on_off
                shr     ah,1                    ;Set AH for Num Lock
                cmp     al,"N"                  ;Branch if "N" was entered
                je      on_off
                shr     ah,1                    ;Set AH for Scroll Lock
                cmp     al,"S"                  ;Branch if "S" was entered
                jne     error

on_off:         cmp     byte ptr [si-2],"-"     ;Branch if leader was a "-"
                je      off
                or      byte ptr es:[17h],ah    ;Toggle a state bit on
                jmp     nextparm                ;Loop back for more
off:            not     ah                      ;Reverse all the bits
                and     byte ptr es:[17h],ah    ;Toggle a state bit off
                jmp     nextparm                ;Loop back for more
;
; Make sure the BIOS and the indicator lights are in sync if /L was entered.
;
checklites:     cmp     syncflag,0              ;Exit now if SYNCFLAG
                je      echo                    ;  isn't set
                mov     bl,es:[17h]             ;Get keyboard flags
                and     bl,70h                  ;Zero unused bits
                mov     cl,4                    ;Shift four bits right
                shr     bl,cl
                cli                             ;Disable interrupts
                mov     al,0EDh                 ;Output command to keyboard
                out     60h,al                  ;  to set the indicator
                jmp     short $+2               ;  lights
                mov     al,bl                   ;Then output the new status
                out     60h,al                  ;  byte
                sti                             ;Enable interrupts
;
; Display the current keyboard status before exiting.
;
echo:           mov     ah,09h                  ;Echo state of Caps Lock
                mov     dx,offset msg1
                int     21h
                mov     ah,09h
                mov     dx,offset msg4
                test    byte ptr es:[17h],40h
                jz      nocaps
                mov     dx,offset msg5
nocaps:         int     21h

                mov     ah,09h                  ;Echo state of Num Lock
                mov     dx,offset msg2
                int     21h
                mov     ah,09h
                mov     dx,offset msg4
                test    byte ptr es:[17h],20h
                jz      nonum
                mov     dx,offset msg5
nonum:          int     21h

                mov     ah,09h                  ;Echo state of Scroll Lock
                mov     dx,offset msg3
                int     21h
                mov     ah,09h
                mov     dx,offset msg4
                test    byte ptr es:[17h],10h
                jz      noscroll
                mov     dx,offset msg5
noscroll:       int     21h

                mov     ax,4C00h                ;Exit with ERRORLEVEL=0
                int     21h
main            endp

;****************************************************************************
; FINDCHAR advances SI to the next non-space or non-comma character.
; On return, carry set indicates EOL was encountered.
;****************************************************************************

findchar        proc    near
                lodsb                           ;Get the next character
                cmp     al,20h                  ;Loop if space
                je      findchar
                cmp     al,2Ch                  ;Loop if comma
                je      findchar
                dec     si                      ;Point SI to the character
                cmp     al,0Dh                  ;Exit with carry set if end
                je      eol                     ;  of line is reached

                clc                             ;Clear carry and exit
                ret

eol:            stc                             ;Set carry and exit
                ret
findchar        endp

;****************************************************************************
; SCANHELP scans the command line for a /? switch.  If found, carry returns
; set and SI contains its offset.  If not found, carry returns clear.
;****************************************************************************

scanhelp        proc    near
                push    si                      ;Save SI
scanloop:       lodsb                           ;Get a character
                cmp     al,0Dh                  ;Exit if end of line
                je      scan_exit
                cmp     al,"?"                  ;Loop if not "?"
                jne     scanloop
                cmp     byte ptr [si-2],"/"     ;Loop if not "/"
                jne     scanloop

                add     sp,2                    ;Clear the stack
                sub     si,2                    ;Adjust SI
                stc                             ;Set carry and exit
                ret

scan_exit:      pop     si                      ;Restore SI
                clc                             ;Clear carry and exit
                ret
scanhelp        endp

code            ends
                end     begin
