/***********************************************************************
 * Yachtzee.c
 * Copyright 1993 by Jeff Glatt
 *
 * You may freely distribute the executable, source code, and documentation,
 * but may not sell or license it.
 ***********************************************************************/

#define INCL_DOSPROCESS
#define INCL_WINBUTTONS
#define INCL_WININPUT
#define INCL_WINSWITCHLIST
#define INCL_WINSYS
#define INCL_WINFRAMEMGR
#define INCL_WINWINDOWMGR
#define INCL_WINDIALOGS
#define INCL_WINSTDSLIDER
#define INCL_GPIDEFAULTS
#define INCL_GPIPRIMITIVES
#define INCL_WINSTATICS
#define INCL_WINHELP
#define INCL_WINMENUS
#define INCL_WINTIMER

#include <os2.h>
#include <string.h>
#include <time.h>
#include "yachtzee.h"
#include "yahhelp.h"
#include "yahdlg.h"

MRESULT EXPENTRY wndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);

/* ********************* Global variables ********************* */
HWND hwndYahFrame=0;	   /* Yahtzee Frame window */
HWND hwndYah;		     /* Yahtzee Client window */
HAB  hab;			/* anchor block for the process */
HMQ  hmq;		       /* handle to the process' message queue */

/* Define a structure used to relate mouse XY position to catagory */
typedef struct _LOCATION
{
  SHORT Y;
  USHORT LeftX, RightX;
  USHORT Row, IdealScore;
} LOCATION;

/* LOCATION structures for 13 catagories */
LOCATION locations[CATAGORIES+2] = {
     {65535, 0, 0, 0, 0},  /* Dummy struct to mark head */
     {3, 70, 129, 12, 20}, /* Chance */
     {4, 70, 129, 11, 50}, /* Yahtzee */
     {5, 70, 129, 10, 40}, /* Large Straight */
     {6, 70, 129, 9, 30},  /* Small Straight */
     {7, 70, 129, 8, 25},  /* Full House */
     {8, 70, 129, 7, 20},  /* 3 of a Kind */
     {9, 70, 129, 6, 18},  /* 4 of a Kind */
     /* --------- */
     {15, 70, 129, 5, 18}, /* Sixes */
     {16, 70, 129, 4, 15}, /* Fives */
     {17, 70, 129, 3, 12}, /* Fours */
     {18, 70, 129, 2, 9},  /* Threes */
     {19, 70, 129, 1, 6},  /* Twos */
     {20, 70, 129, 0, 3},  /* Ones */
     {65535, 0, 0, 0, 0},  /* Dummy struct to mark end */
};

/* LOCATION structures for 5 dice */
LOCATION dice_locs[DICE+2] = {
     {65535, 0, 0, 0, 0},  /* Dummy struct to mark head */
     {2, 12, 44, 4, 0},
     {3, 12, 44, 3, 0},
     {4, 12, 44, 2, 0},
     {5, 12, 44, 1, 0},
     {6, 12, 44, 0, 0},
     {65535, 0, 0, 0, 0},  /* Dummy struct to mark end */
};

/* An array to hold the UCHAR values of 5 Dice, plus a UCHAR count of the total number
    of rolls the current player has made */
UCHAR dice_vals[DICE+1];

/* An array to hold the UCHAR selected state of 5 Dice, plus a pad byte. The selected state is
    0 is the die isn't selected to be rolled, non-zero if selected. */
UCHAR selected[DICE+1];

/* An array to hold the pointers to 6 Die bitmaps, plus a blank die */
HBITMAP DieBmps[DICE+1];

/* An array to hold the counts for the 6 possible values of a Die (ie, each die has 6 faces), plus
    a SHORT count of the total of all 5 dice values. */
SHORT values[7];

/* An array to store each player's scores for the 13 catagories on a Yachtzee scorecard. We
   initialize all scores to -1 to indicate that all catagories are unscored. */
SHORT scores[MAXPLAYERS][CATAGORIES];

/* ========== Audio Stuff ========== */
/* Assumes a driver whose DosWrite() handling expects MIDI messages for a GM Sound Module */

/* Handle to driver */
HFILE  GMHandle=0;

/* Define a structure used for sound effects */
typedef struct _SOUND
{
  UCHAR   CurrTime;
  UCHAR   Duration;
  UCHAR   MidiMsg[3];
  UCHAR   NtnCnt;
} SOUND;

/* LOCATION structures for 13 catagories */
SOUND sounds[SOUNDEFFS] = {
     {0, 4, 0x99, 79, 127, 0},
     {0, 8, 0x99, 49, 127, 0},
     {0, 4, 0x99, 81, 127, 0},
     {0, 4, 0x99, 29, 127, 0},
     {0, 4, 0x99, 58, 127, 0},
     {0, 8, 0x90, 64, 127, 0},
};

/* Turn GM sound module on */
UCHAR GM_On[]={ 0xF0,0x7F,0x7F,0x04,0x01,0x7F,0x7F,0xF7 };

/* Set Part 1 to APPLAUSE Patch */
UCHAR Applause[]={ 0xC0, 126 };

/* The GM Sound Module driver name. Default to that MPU401 driver that I use */
/* CHAR Module[] = { "MPU401" }; */
CHAR Module[] = { "RAP10_1$" };
CHAR * drv_name = &Module[0];

UCHAR timer=0;
UCHAR volume=127;
/* =============================== */

/* Keeps track of how many more catagories need to be scored before the game is over */
UCHAR total_cats;

/* Keeps track of the current player # */
UCHAR cur_player=0;

/* To blank out a catagory's score */
UCHAR Blanks[] = "    ";

/* The keyboard shortcuts for the catagories */
UCHAR keys[] = "nwhoixkafslyc";

/* Help Global variables */
HWND hwndHelpInstance=0;
CHAR szLibName[]="YACHTZEE.HLP";
CHAR szWindowTitle[HELPTITLELEN];
UCHAR helpFlg=0;


/*************************** messageBox() ********************************
 * Called whenever a message box is to be displayed. Displays the message box with the string
 * given by the passed idMsg (retrieved from the message table in the executable) and using the
 * style flags passed in fsStyle. Returns the value of WinMessageBox().
 ***********************************************************************/
ULONG messageBox(HWND hwndOwner, ULONG idMsg)
{
   CHAR szText[80];

   /* Load the string from the executable's message table */
   if(!WinLoadMessage(hab, 0, idMsg, 80, (PSZ)szText))
   {
      WinAlarm(HWND_DESKTOP, WA_ERROR);
      return(MBID_ERROR);
   }

   /* Beep sound */
   WinAlarm(HWND_DESKTOP, WA_ERROR);

   /* Display the msg in a dialog box */
   return(WinMessageBox(HWND_DESKTOP, hwndOwner, szText, 0, MSGBOXID, MB_OK|MB_ERROR));
}



/******************************** setSysMenu() ****************************
 * Sets only the Move and Close items of the system menu, removing all others from the system
 * menu for that dialog. Any dialog box may call this routine, to edit which menu items appear on
 * its System Menu pulldown.
 *************************************************************************/

VOID setSysMenu(HWND hDlg)
{
    HWND     hSysMenu;
    MENUITEM Mi;
    ULONG    Pos;
    MRESULT  Id;
    SHORT    cItems;

    /* We only want Move and Close in the system menu. Remove all others. */

    hSysMenu = WinWindowFromID(hDlg, FID_SYSMENU);
    WinSendMsg(hSysMenu, MM_QUERYITEM, MPFROM2SHORT(SC_SYSMENU, FALSE), MPFROMP((PCH) & Mi));
    Pos = 0L;
    cItems = (SHORT)WinSendMsg( Mi.hwndSubMenu, MM_QUERYITEMCOUNT,
				(MPARAM)NULL, (MPARAM)NULL);
    while (cItems--)
    {
	Id=WinSendMsg(Mi.hwndSubMenu, MM_ITEMIDFROMPOSITION, MPFROMLONG(Pos), (MPARAM)NULL);
	switch (SHORT1FROMMR(Id))
	{
	    case SC_MOVE:
	    case SC_CLOSE:
	       Pos++;  /* Don't delete that one. */
	       break;
	    default:
	       WinSendMsg(Mi.hwndSubMenu, MM_DELETEITEM, MPFROM2SHORT((USHORT)Id, TRUE), (MPARAM)NULL);
	}
    }
}



/******************************* prodInfoDlgProc() ***********************
 * Processes all msgs sent to the Product Info dialog. This dialog has only a button control, so
 * this routine processes only WM_COMMAND messages. Any WM_COMMAND posted must have
 * come from the OK button, so we dismiss the dialog upon receiving it.
 *************************************************************************/

MRESULT EXPENTRY prodInfoDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   switch(msg)
   {
      case WM_INITDLG:
	 /* Modify system menu for this dialog	*/
	 setSysMenu(hwnd);
	 return MRFROMSHORT(FALSE);

      default:
	 return(WinDefDlgProc(hwnd, msg, mp1, mp2));

      case WM_COMMAND:
	 /* Must be OK button since that's the only control. Close the dialog */
	 WinDismissDlg(hwnd, TRUE);
	 /* fall through */
   }
   return(0);
}



/**************************** destroyHelpInstance() **************************
 * Called before program termination. Calls WinDestroyHelpInstance() to destroy the
 * help instance for the application.
 *************************************************************************/
VOID destroyHelpInstance(VOID)
{
   if(hwndHelpInstance) WinDestroyHelpInstance(hwndHelpInstance);
   hwndHelpInstance=0;
}



VOID badHelp(HWND hwnd)
{
    messageBox(hwnd, IDMSG_HELPDISPLAYERROR);
}



/********************************* initHelp() ****************************
 * Initializes the IPF help facility. Called once during initialization of the program. Initializes the
 * HELPINIT structure and creates the help instance. If successful, the help instance is
 * associated with the main (Yachtzee) Frame window.
 **********************************************************************/
VOID initHelp(VOID)
{
   HELPINIT hini;

   /* Initialize help init structure */
   memset(&hini, 0, sizeof(HELPINIT));
   hini.cb = sizeof(HELPINIT);
   hini.phtHelpTable = (PHELPTABLE)MAKELONG(YAH_HELP_TABLE, 0xFFFF);

   /* Load the .hlp window title */
   hini.pszHelpWindowTitle = (PSZ)szWindowTitle;
   if(!WinLoadString(hab, (HMODULE)0, IDS_HELPWINDOWTITLE, HELPTITLELEN, (PSZ)szWindowTitle))
   {
      hini.pszHelpWindowTitle = (PSZ)szLibName;
   }

   /* The .hlp file name */
   hini.pszHelpLibraryName = (PSZ)szLibName;

   /* Create help instance */
   hwndHelpInstance = WinCreateHelpInstance(hab, &hini);
   if((!hwndHelpInstance) || hini.ulReturnCode)  return;

   /* Associate help instance with main frame window (Parts) */
   if(!WinAssociateHelpInstance(hwndHelpInstance, hwndYahFrame))
   {
      destroyHelpInstance();
      messageBox(hwndYahFrame, IDMSG_HELPLOADERROR);
   }
}



/********************************** helpGeneral() ***************************
 * Processes the WM_COMMAND message posted by the "Yachtzee Help" item of the Help menu.
 * Called from wndProc() when the "Yachtzee Help" menu item is selected. Sends an
 * HM_EXT_HELP msg to the help instance so that the default "Extended Help" is displayed.
 *************************************************************************/
VOID helpGeneral(HWND hwnd)
{
   /* Display the system extended help panel (ie, Yachtzee's Online .hlp book) */
   if(hwndHelpInstance)
   {
      if(!WinSendMsg(hwndHelpInstance, HM_EXT_HELP, 0, 0)) return;
   }
   badHelp(hwnd);
}



/****************************** helpUsingHelp() *************************
 * Processes the WM_COMMAND message posted by the "Using Help" item of the Help menu.
 * Called from wndProc() when the "Using Help" menu item is selected. Sends an
 * HM_DISPLAY_HELP msg to the help instance so that the default "Using Help" is displayed.
 *************************************************************************/
VOID helpUsingHelp(HWND hwnd)
{
   /* Display the system help for help panel. (ie, how to use OS/2's Help Manager) */
   if(hwndHelpInstance)
   {
      if(!WinSendMsg(hwndHelpInstance, HM_DISPLAY_HELP, 0, 0)) return;
   }
   badHelp(hwnd);
}



/******************************** helpIndex() *****************************
 * Processes the WM_COMMAND message posted by the "Index" item of the Help menu. Called
 * from wndProc() when the "Index" menu item is selected. Sends an HM_INDEX_HELP message
 * to the help instance so that the default "Help Index" is displayed.
 *************************************************************************/
VOID  helpIndex(HWND hwnd)
{
   /* Display the system help index panel (ie, Yachtzee Index) */
   if(hwndHelpInstance)
   {
      if(!WinSendMsg(hwndHelpInstance, HM_HELP_INDEX, 0, 0)) return;
   }
   badHelp(hwnd);
}



/****************************** helpProductInfo() ****************************
 * Processes the WM_COMMAND message posted by the "Product information" item of the Help
 * Menu. Called from wndProc() when the Product information menu item is selected. Calls
 * WinDlgBox() to display the Product information dialog.
 *************************************************************************/
VOID helpProductInfo(HWND hwnd)
{
   /* Display the Product Information dialog. */
   WinDlgBox(HWND_DESKTOP, hwnd, (PFNWP)prodInfoDlgProc, 0, IDD_PRODUCTINFO, 0);
}



/*************************** displayHelpPanel() ***********************
 * Displays the help panel whose ID is passed. Called whenever a help panel is desired to be
 * displayed, perhaps from the WM_HELP processing of the dialog boxes. Sends
 * HM_DISPLAY_HELP msg to the help instance.
 *
 * Parameters :  idPanel = panel i.d.
 *************************************************************************/
VOID displayHelpPanel(HWND hwnd, ULONG idPanel)
{
   if(hwndHelpInstance)
   {
      if(!WinSendMsg(hwndHelpInstance, HM_DISPLAY_HELP, MPFROMLONG(idPanel),
			    MPFROMSHORT(HM_RESOURCEID))) return;
   }
   badHelp(hwnd);
}



/* **************************** sendGmStr() ******************************
 *  Sends the passed str to the Sound Module (via the driver's Write vector).
 ******************************************************************** */

VOID sendGmStr(UCHAR * str, ULONG count)
{
    ULONG  bytes;

    if (GMHandle)
    {
       DosWrite(GMHandle, str, count, &bytes);
    }
}



/* **************************** send16() ******************************
 *  Sends the passed controller number to the GM Sound Module on all 16 channels.
 ********************************************************************** */

VOID send16(UCHAR ctlNum)
{
   register USHORT i;
   UCHAR  Msg[3];

   /* Format the 2 data bytes for MIDI msg */
   Msg[2]=0x00;
   Msg[1]=ctlNum;

   /* Send the controller msg on all 16 channels */
   for(i=0;i<16;i++)
   {
      Msg[0]=0xB0|i;
      sendGmStr(&Msg[0], 3);
   }
}



/* **************************** allNotesOff() ******************************
 *  Turns "all notes off" and "all sounds off" on the 16 MIDI channels
 ********************************************************************** */

VOID allNotesOff(VOID)
{
   send16(0x78);
   send16(0x7B);
}



/* **************************** resetGM() ******************************
 *  Turns all sounds and note off, and resets MIDI controllers on all 16 channels
 ********************************************************************** */

VOID resetGM(VOID)
{
   /* Turn off all notes */
   allNotesOff();

   /* Reset controllers */
   send16(0x79);
}



/* ****************************** do_sound() *******************************
 * Sends a MIDI note-on based upon the passed sound #. Then it starts an asyncronous timer
 * that will send a WM_TIMER message to my wndProc(). At that point, I send a MIDI note-off
 * for any sounds timed out.
 *************************************************************************/
VOID do_sound(USHORT sndnum)
{
   register SOUND * sndptr;

   /* Get SOUND struct for this sndnum */
   sndptr=&sounds[sndnum];

   /* Set the time-out value (ie, Duration) */
   sndptr->CurrTime=sndptr->Duration;

   /* Increase NOTE-ON count (one more of this note # is playing) */
   sndptr->NtnCnt++;

   /* Send the MIDI msg associated with this sound event */
   sndptr->MidiMsg[2]=volume;
   sendGmStr(&sndptr->MidiMsg[0], 3);

   /* If Timer isn't already started, start it */
   if (!timer)
   {
      /* Start the timer to send WM_TIMER msgs at 1/2 sec intervals */
      timer=1;
      WinStartTimer(hab, hwndYah, 0, 500);
   }
}



/* ***************************** closeGM() *********************************
 * Closes any currently opened GM driver (handle in GMHandle, which if 0 is used to indicate no
 * driver open). Clears GMHandle.
 *************************************************************************/

VOID closeGM(VOID)
{
   /* Close GM driver and clear handle */
   if(GMHandle)
      DosClose(GMHandle);
   GMHandle=0;
}



/* ***************************** openGM() *********************************
 * Closes any previously opened GM driver, and opens the driver whose name drv_name points
 * to. Stores the handle in GMHandle. Turns on the GM module which a GM Standard SYSEX
 * message. Sets MIDI chan 1 (0) to the APPLAUSE Patch. Assumes GM drums are on MIDI chan
 * 10 (9).
 *************************************************************************/

VOID openGM(VOID)
{
   ULONG Action;

   /* Close any currently opened driver */
   closeGM();

   /* Open Sound Module's driver */
   if(DosOpen(drv_name, &GMHandle, &Action, 0, FILE_NORMAL, FILE_OPEN,
	   OPEN_ACCESS_READWRITE|OPEN_SHARE_DENYNONE, 0))
   {
      messageBox(hwndYah, IDMSG_CANNOTOPENGM);
   }

   /* Turn GM System On */
   sendGmStr(&GM_On[0], 8);

   /* Set MIDI chan 1 to Applause patch */
   sendGmStr(&Applause[0], 2);
}



/* ***************************** audioDlgProc() ****************************
 * Processes all msgs sent to the "Yahtzee Audio" dialog.
 *************************************************************************/

MRESULT EXPENTRY audioDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   register USHORT code;

   switch(msg)
   {
      case WM_INITDLG:
	 /* All notes and controllers OFF */
	 resetGM();

	 /* Modify system menu for this dialog */
	 setSysMenu(hwnd);

	 /* Set volume slider */
	 WinSendDlgItemMsg(hwnd, ID_AUDIO_VOL, SLM_SETSLIDERINFO,
		MPFROM2SHORT(SMA_SLIDERARMPOSITION,SMA_INCREMENTVALUE), (VOID *)volume);
	 /* Add FULL Detent for volume slider */
	 WinSendDlgItemMsg(hwnd, ID_AUDIO_VOL, SLM_ADDDETENT, MPFROMSHORT(127), 0);

	 /* Set Sound ON/OFF checkmark */
	 code=0;
	 if (GMHandle) code++;
	 WinSendDlgItemMsg(hwnd, ID_AUDIO_ON, BM_SETCHECK, MPFROMSHORT(code), 0);
	 break;

      case WM_CLOSE:
	 goto do_enter;

      case WM_CHAR:
	 /* Ignore key releases */
	 code=SHORT1FROMMP(mp1);
	 if (code & KC_KEYUP) goto dodef;

	 /* If ENTER or NEWLINE, dismiss the dlg */
	 if (code & KC_VIRTUALKEY)
	 {
	      code=SHORT2FROMMP(mp2);
	      switch(code)
	      {
		  case VK_NEWLINE:
		  case VK_ENTER:
		     /* If ENTER, set the new values (ie, ON/OFF, volume). */
do_enter:	    volume = (UCHAR)WinSendDlgItemMsg(hwnd, ID_AUDIO_VOL, SLM_QUERYSLIDERINFO,
			    MPFROM2SHORT(SMA_SLIDERARMPOSITION,SMA_INCREMENTVALUE), 0);
		     if (!volume) volume++;

		     if (WinSendDlgItemMsg(hwnd, ID_AUDIO_ON, BM_QUERYCHECK, 0, 0))
		     {
			openGM();
		     }
		     else
		     {
			closeGM();
		     }

		     /* fall through to ESC */

		 case VK_ESC:
		     /* If ESC, dismiss the dialog without storing any value */
		     WinDismissDlg(hwnd, TRUE);
	      }
	 }

      /* If we don't handle the msg, let the OS do some default action */
      default:
dodef:
	 return(WinDefDlgProc(hwnd, msg, mp1, mp2));
   }
   return(0);
}



/************************** exitProc(usTermCode) **************************
 * Cleans up app's resources and terminates application. This is called by OS/2's DosExitList
 * when the app exits. Global resources, such as the main window and message queue, are
 * destroyed and any system resources are freed. It is passed a termination code number
 * (arbitrary) and returns EXLST_EXIT to the DosExitList handler.
 *************************************************************************/
VOID exitProc(USHORT usTermCode)
{
   /* Destroy the help instance */
   destroyHelpInstance();

   /* Destroy the Frame window if it exists (and all of its children) */
   if(WinIsWindow(hab, hwndYahFrame)) WinDestroyWindow(hwndYahFrame);

   /* Close GM driver */
   resetGM();
   closeGM();

   /* Free PM queue */
   WinDestroyMsgQueue(hmq);
   WinTerminate(hab);

   /* Exit */
   DosExitList(EXLST_EXIT, (PFNEXITLIST)NULL);

   /* Referenced here to prevent an 'Unreferenced Parameter' warning at compile time */
   usTermCode;
}



/* ******************************* evaluate() **********************************
 * Evaluates the current dice values (in the dice_vals[] array) in terms of the score they produce
 * for the specified category (ie, row), and returns that score.
 ************************************************************************* */

SHORT evaluate(USHORT row)
{
   USHORT count, total;

   switch(row)
   {
      /* =========== 3 of a kind ========= */
      case 6:
	 /* Check if there are more than 2 dice of any one value. */
	 total=2;
	 goto chk_me;

      /* ========== 4 of a kind ========== */
      case 7:
	 /* Check if there are more than 3 dice of any one value. */
	 total=3;
chk_me: for(count=0; count<6; count++)
	 {
	    if(values[count] > total)
	       return(values[6]);
	 }
	 return(0);

      /* ============= Full house ============== */
      case 8:
	 total=0;
	 for(count=0; count<6; count++)
	 {
	    if(values[count] > 1)
	       total+=values[count];
	 }

	 return( ((total == 5) ? 25 : 0) );

      /* ============ Small Straight ============ */
      case 9:
	 if ( (values[0] && values[1] && values[2] && values[3])
	 || (values[1] && values[2] && values[3] && values[4])
	 || (values[2] && values[3] && values[4] && values[5]) )
	    return(30);

	 return(0);

      /* ======== Large Straight ======== */
      case 10:
	 if ( (values[0] && values[1] && values[2] && values[3] && values[4])
	 || (values[1] && values[2] && values[3] && values[4] && values[5]) )
	    return(40);

	 return(0);

      /* ============= Yachtzee ============== */
      case 11:
	 /* Check if there are 5 dice of any one value (ie, all 5 are the same) */
	 for(count=0; count<6; count++)
	 {
	    if(values[count] == 5)
	    {
	       do_sound(YAHSOUND);
	       return(50);
	    }
	 }
	 return(0);

      /* ============== Chance ============== */
      case 12:
	 return(values[6]);

      /* ============= Single # (in the upper half of scorecard) =========== */
      default:
	 /* Return total of desired spots */
	 return((row+1)*values[row]);
   }
}



/**************************** stuff_values() ******************************
 * Initializes the values[] array based upon current dice values.
 *************************************************************************/

VOID stuff_values(VOID)
{
   UCHAR row;

   /* Clear dice sort array */
   for(row=0; row<7; row++)
      values[row]=0;

   /* Count how many dice there are of each value (ie, a die has 6 possible values). Also,
       total the value of all 5 dice. */
   for(row=0; row<5; row++)
   {
      values[dice_vals[row]]++;
      values[6]+=(SHORT)(1+dice_vals[row]);
   }
}



/* ***************************** calc_totals() ******************************
 * For the cur_player, displays the score totals (ie, upper, lower, bonus, etc).
 *************************************************************************/

VOID calc_totals(VOID)
{
   register USHORT i, upper, lower, bonus_pt;

   /* Recalculate totals */
   upper=lower=0;
   for (i=0; i<6; i++)
   {
      if (scores[cur_player][i]!=-1)
	 upper+=scores[cur_player][i];
   }
   for (i=6; i<CATAGORIES; i++)
   {
      if (scores[cur_player][i]!=-1)
	 lower+=scores[cur_player][i];
   }

   /* Refresh upper sub-total, total, and bonus */
   WinSetDlgItemShort(hwndYah, ID_UPPER_T+(cur_player*CATAGORIES), upper, 0);
   bonus_pt=0;
   if (upper>=BONUS)
   {
      bonus_pt=35;
   }
   WinSetDlgItemShort(hwndYah, ID_BONUS_T+(cur_player*CATAGORIES), bonus_pt, 0);
   WinSetDlgItemShort(hwndYah, ID_UTOTAL_T+(cur_player*CATAGORIES), upper+bonus_pt, 0);

   /* Refresh lower sub-total */
   WinSetDlgItemShort(hwndYah, ID_LOWER_T+(cur_player*CATAGORIES), lower, 0);

   /* Refresh grand total */
   WinSetDlgItemShort(hwndYah, ID_GRAND_T+(cur_player*CATAGORIES), lower+upper, 0);
}



/**************************** show_unscored() ******************************
 * For the cur_player, displays the possible scores to choose for the current state of the dice.
 *************************************************************************/

VOID show_unscored(VOID)
{
   register USHORT i;

   /* Stuff new values[] based upon dice */
   stuff_values();

   /* Check all catagories */
   for (i=0; i<CATAGORIES; i++)
   {
      /* Is this catagory unscored (ie, value is -1)? */
      if (scores[cur_player][i] == -1)
      {
	 /* Set the Text Control for this catagory to the evaluated dice score */
	 WinSetDlgItemShort(hwndYah, (i+ID_CAT1)+(cur_player*CATAGORIES), evaluate(i), 0);
      }
   }

   /* Display totals so far */
   calc_totals();
}



/**************************** blank_unscored() ******************************
 * For the cur_player, blanks all unscored catagories.
 *************************************************************************/

VOID blank_unscored(VOID)
{
   register USHORT i;

   /* Check all catagories */
   for (i=0; i<CATAGORIES; i++)
   {
      /* Is this catagory unscored (ie, value is -1)? */
      if (scores[cur_player][i] == -1)
      {
	 /* Set the Text Control for this catagory to blanks */
	 WinSetWindowText(WinWindowFromID(hwndYah, (i+ID_CAT1)+(cur_player*CATAGORIES)), &Blanks[0]);
      }
   }

   /* Display totals so far */
   calc_totals();
}



/* ******************************* roll_one() *********************************
 * Rolls the selected die number, altering the image for the die to the new rolled value. Returns
 * the new rolled value.
 ************************************************************************* */

UCHAR roll_one(UCHAR die_num)
{
   UCHAR i, val;

   /* Do 9 quick tumbles of the rolling die. (ie, Pretend that a rolled die tumbles through
       10 sides before finally coming to rest) */
   for (i=9; i; i--)
   {
      val = rand()%(DICE+1);
      WinSendMsg(WinWindowFromID(hwndYah, die_num+ID_DIE_1), SM_SETHANDLE, (MPARAM)DieBmps[val], 0);
   }

   /* Return the final value of the die */
   return(val);
}



/* ***************************** shake_dice() *********************************
 * Rolls the dice that the player selected to be rolled (ie, updates the display with the new
 * dice values, and updates the dice_vals[] array).
 ************************************************************************ */

VOID shake_dice(VOID)
{
   register UCHAR die_num;
   register UCHAR dflg=0;

   for(die_num=0; die_num<DICE; die_num++)
   {
      /* If the Die was selected to be rolled by the player, roll it */
      if(selected[die_num])
      {
	 /* Display new die value, and store in dice array */
	 dice_vals[die_num] = roll_one(die_num);

	 /* Indicate that die is no longer selected by player */
	 selected[die_num]=0;

	 dflg=1;
      }
   }

   /* Increment the # of times that player has rolled the dice */
   if (dflg) dice_vals[TURNS]++;
}



/* ***************************** initial_roll() ***************************** */

VOID initial_roll(VOID)
{
   UCHAR die_num;

   /* Make the sound of dice rolling */
   do_sound(DICEROLL);

   /* Do the first roll of all 5 dice, and set all dice to be initially unselected */
   for(die_num=0; die_num<DICE; die_num++)
   {
      selected[die_num] = 0;
      dice_vals[die_num] = roll_one(die_num);
   }

   /* Initialize the # of rolls to 1 */
   dice_vals[TURNS] = 1;

   /* Show initial unscored catagories */
   show_unscored();

   /* Enable the ROLL control and give it the focus (so that SPACE BAR can roll dice) */
   WinEnableWindow(WinWindowFromID(hwndYah, ID_ROLL), 1);
   WinSetFocus(HWND_DESKTOP,WinWindowFromID(hwndYah, ID_ROLL));
}



/* ******************************* initGame() ******************************
 * Prepares structures to start a new game.
 *********************************************************************** */

VOID initGame(VOID)
{
   double time;
   register UCHAR cats;
   ULONG color=CLR_PINK;
   register HWND t_hwnd;

   /* Reset the GM Sound Module */
   resetGM();

   /* Use time to seed random number generator */
   time = clock();
   srand((ULONG)(time/CLOCKS_PER_SEC));

   /* Initialize the scores array of each player */
   for(cur_player=0; cur_player<MAXPLAYERS; cur_player++)
   {
      /* Do all CATAGORIES for this player */
      for (cats=0; cats<CATAGORIES; cats++)
      {
	 /* Set this catagory's score to -1 (to indicate an unscored catagory) */
	 scores[cur_player][cats] = -1;

	 /* Get Text (ie, Static) control for this catagory's score */
	 t_hwnd = WinWindowFromID(hwndYah, (cats+ID_CAT1)+(cur_player*CATAGORIES));

	 /* Reset text foreground color to RED */
	 WinSetPresParam(t_hwnd, PP_FOREGROUNDCOLORINDEX, sizeof(ULONG), (PVOID)&color);

	 /* Blank this catagory's score */
	 WinSetWindowText(t_hwnd, &Blanks[0]);
      }
   }

   /* Reset the # of catagories to play (per game) */
   total_cats=CATAGORIES;

   /* Start with first player */
   cur_player=0;

   /* Highlight player's name */
   /* !!! */

   /* Do the first dice roll */
   initial_roll();
}



/**************************** set_die_state() ******************************
 * Sets the die's BITMAP CONTROL to the die image that correlates the value for that die.
 *************************************************************************/


VOID set_die_state(USHORT die_num)
{
   /* Assume blank bitmap (ie, die is selected for rolling) */
   HBITMAP bmp=DieBmps[6];

   /* If die is not selected, set the bitmap that correlates to the current value for this die */
   if (!selected[die_num])
   {
      bmp=DieBmps[dice_vals[die_num]];
   }

   WinSendMsg(WinWindowFromID(hwndYah, die_num+ID_DIE_1), SM_SETHANDLE, (MPARAM)bmp, 0);
}



/**************************** wndProc() ******************************
 * Processes a msg sent to the Yahtzee window. This is called by the OS (usually via our own
 * call to WinDispatchMsg) for each msg placed in our PM msg queue (usually by PM). A switch
 * statement branches to the routines to be performed for each msg processed. Any msg not
 * specifically processed here is passed to the OS/2 function WinDefWindowProc() for OS default
 * action for that msg.
 *************************************************************************/

MRESULT EXPENTRY wndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   RECTL rclUpdate;
   register USHORT i, y, cat;
   register LOCATION * loc;
   register SHORT cur_score;
   double time;
   ULONG color;
   HWND hwndMenu;
   UCHAR val;

   switch(msg)
   {

     /* ============= Timer timed out (ie, for sequencing events) ========= */
     case WM_TIMER:
	/* Assume that timer will be turned off */
	color=0;
	/* Check all sound effects to see if any are timing out (ie, playing back right now) */
	for (i=0; i<SOUNDEFFS; i++)
	{
	   if (sounds[i].CurrTime)
	   {
	      /* Decrement the CurrTime (ie duration). If 0, turn off the sound */
	      if (!(--sounds[i].CurrTime))
	      {
		 sounds[i].MidiMsg[2]=0;
		 while (sounds[i].NtnCnt--)
		 {
		    sendGmStr(&sounds[i].MidiMsg[0], 3);
		 }
	      }
	      /* Must leave timer on because an event is still timing out */
	      else
	      {
		 color=1;
	      }
	   }
	}
	if (!color)
	{
	   timer=0;
	   WinStopTimer(hab, hwndYah, 0);
	}
	break;

     /* ============== A keyboard press ============ */
     case WM_CHAR:
	/* Get the code */
	i=SHORT2FROMMP(mp2);

	/* Ignore keyup */
	if (!(SHORT1FROMMP(mp1) & KC_KEYUP))
	{
	   /* Is this a VIRTUAL (ie, no ascii equivalent) key? */
	   if (!(SHORT1FROMMP(mp1) & KC_VIRTUALKEY))
	   {
	      /* Get the ascii char */
	      i=(UCHAR)SHORT1FROMMP(mp2);

	      /* If one of the digits 1, 2, 3, 4, or 5, these correspond to toggling the respective
		  die for selection */
	      if (i>'0' && i<'6')
	      {
		 /* Toggle this die's control for selection/deselection */
		 i-='1';
		 if (dice_vals[TURNS]<3) goto tog_me;
		 goto do_err;
	      }
	      /* Check if player is choosing a catagory to score */
	      else
	      {
		 /* If SPACE bar, player wants to roll selected dice */
		 if (i==' ') goto roll_me;
		 if ( i<'Z' && i>'A') i+=32;
		 for (cat=0; cat<CATAGORIES; cat++)
		 {
		    if (keys[cat]==i)
		    {
		       loc = &locations[CATAGORIES-cat];
		       goto do_sel;
		    }
		 }
	      }
	   }
	   /* If ENTER key, then player wants to roll selected dice */
	   else if (i == VK_NEWLINE || i == VK_ENTER)
	   {
	      goto roll_me;
	   }
	 }
	goto hdef;

     /* ====== BUTTON1DOWN is sent when user presses and releases the mouse 1. ======
		 SINGLESELECT is sent if user presses/releases within a time limit. */
     case WM_BUTTON1DOWN:
	/* If a game is still in play, check if player clicked on a score or die */
	if (total_cats)
	{
	   /* Get MouseX coord */
	   i = SHORT1FROMMP(mp1);

	   /* Clicked on a die */
	   if (i<BOARDLEFT)
	   {
	      /* If he already rolled 3 times, then he can't change a die */
	      if (dice_vals[TURNS]>2) goto do_err;

	      /* Get MouseY coord, and divide by 39 to convert to a die area number. */
	      y = (SHORT2FROMMP(mp1))/39;

	      /* Get ptr to first DICE LOCATION struct */
	      loc=&dice_locs[1];

	      /* Go through the list until we find the die within this area, or go past this line */
	      for(;;)
	      {
		 if(loc->Y == y)
		 {
		    do
		    {
		       if((loc->LeftX<= i) && (loc->RightX>= i))
		       {
			  /* A die has been selected/deselected. Flip its state */
			  i=loc->Row;

			  /* Toggle selected state */
tog_me: 		 selected[i]^=1;

			  /* Set the die control's new bitmap */
			  set_die_state(i);

			  goto hdef;
		       }
		       loc++;
		    } while (loc->Y == y);

		    /* The X coord wasn't inside of any die hit box, so ignore this mouse click */
		    goto hdef;
		 }

		 /* Did we go past this line? If so, then no catagory on this line, so return 0 */
		 else if(loc->Y > y) goto hdef;

		 /* Next LOCATION structure */
		 loc++;
	      }
	   }
	   /* Check for a click on the scoreboard */
	   else
	   {
	      /* Get MouseY coord, and divide by 18 to convert to a line number. Get x coord. */
	      y = (SHORT2FROMMP(mp1))/18;

	      /* Get ptr to first LOCATION struct */
	      loc=&locations[1];

	      /* Go through the list until we find the catagory on this line, or go past this line */
	      for(;;)
	      {
		 if(loc->Y == y)
		 {
		    do
		    {
		       if((((loc->LeftX)+WIDTH*cur_player)<= i) && (((loc->RightX)+WIDTH*cur_player)>= i))
		       {
			  cat=loc->Row;

do_sel: 		  /* Wants help instead of select it? */
			  if (helpFlg)
			  {
			     helpFlg=0;
			     displayHelpPanel(hwnd, cat+2500);
			     goto hdef;
			  }

			  /* A catagory has been chosen. Make sure that this catagory is unscored */
			  if(scores[cur_player][cat] != -1) goto hdef;

			  /* Calculate the score, and place into score struct */
			  cur_score = evaluate(cat);
			  scores[cur_player][cat] = cur_score;

			  /* Set Text (ie, Static) control for this catagory to BLUE text foreground
			      because it has now been scored */
			  color=CLR_NEUTRAL;
			  i=(cat+ID_CAT1)+(cur_player*CATAGORIES);
			  WinSetPresParam(WinWindowFromID(hwnd, i), PP_FOREGROUNDCOLORINDEX, sizeof(ULONG), (PVOID)&color);

			  /* Set the Text control for this catagory to display the score */
			  WinSetDlgItemShort(hwnd, i, cur_score, 0);

			  /* Make sound based upon score */
			  if (cur_score < loc->IdealScore) do_sound(BOMBSOUND);
			  else if (cur_score >= loc->IdealScore) do_sound(HAPPYSOUND);

			  /* Blank out all unscored catagories */
			  blank_unscored();

			  /* Unhighlight the player's name */
			  /* !!! */


			  /* Increment cur_player to next player. If all players have just completed
			      filling in 1 more catagory, then decrement total_cats. */
			  cur_player++;
			  if (cur_player>=MAXPLAYERS)
			  {
			     cur_player=0;
			     if(!(--total_cats))
			     {
				 /* Make ending sound */
				 do_sound(ENDSOUND);

				 /* If more than 1 player, figure out who won and display name */
				 /* !!! */

				 goto hdef;
			     }
			  }

			  /* Highlight next player's name */
			  /* !!! */


			  /* Now, begin next player's turn. Roll dice, and display possible scores */
			  initial_roll();
			  goto hdef;
		       }
		       loc++;
		    } while (loc->Y == y);

		    /* The X coord wasn't inside of any catagory hit box, so ignore this mouse click */
		    goto hdef;
		 }

		 /* Did we go past this line? If so, then no catagory on this line, so return 0 */
		 else if(loc->Y > y) goto hdef;

		 /* Next LOCATION structure */
		 loc++;
	      }
	   }
	}
	goto hdef;    /* always let OS do default processing because it uses BUTTON1DOWN to
			  select active window */

     /* ========== Some controls send WM_COMMAND msgs ========== */
     case WM_COMMAND:
	/* get code */
	i=SHORT1FROMMP(mp1);

	switch (i)
	{
	   /* ------ ROLL PushButton ------ */
	   case ID_ROLL:
	      /* If player has rolled the dice 3 times already, ignore ROLL control */
roll_me:      if (dice_vals[TURNS]<3)
	      {
		 /* Use time to seed random number generator */
		 time = clock();
		 srand((ULONG)(time/CLOCKS_PER_SEC));

		 /* Make the sound of dice rolling */
		 do_sound(DICEROLL);

		 /* Roll the selected dice */
		 shake_dice();

		 /* Show all possible scores */
		 show_unscored();

		 /* If this is the 3rd roll, disable the DICE and ROLL controls (ie, Player can't select
		    them anymore. He needs to choose his score now */
		 if (dice_vals[TURNS]>2)
		 {
		    WinEnableWindow(WinWindowFromID(hwnd, ID_ROLL), 0);
		 }
	      }
	      /* Beep to let player know that he MUST choose a score */
	      else
	      {
do_err: 	 if (GMHandle)
		    do_sound(AUDIO_ERROR);
		 else
		    WinAlarm(HWND_DESKTOP, WA_ERROR);
	      }
	      goto hdef;

	   /* ---------------- FILE MENU --------------- */
	   case IDM_AUDIO:
	      WinDlgBox(hwnd, 0, (PFNWP)audioDlgProc, 0, ID_AUDIO_WIN, 0);
	      break;

	   case IDM_FILENEW:
	      initGame();
	      break;

	   /* ---------------- HELP MENU --------------- */
	   case IDM_HELPINDEX:
	      helpIndex(hwnd);
	      break;

	   case IDM_HELPGENERAL:
	      helpGeneral(hwnd);
	      break;

	   case IDM_HELPUSINGHELP:
	      helpUsingHelp(hwnd);
	      break;

	   case IDM_HELPPRODUCTINFO:
	      helpProductInfo(hwnd);
	}
	break;

     /* ---- Called when the window background needs repainting ---- */
     case WM_ERASEBACKGROUND:
	/* Note: return(MRESULT)(TRUE); to request PM to paint the window background in
	    SYSCLR_WINDOW color. */
	return(MRESULT)(TRUE);

     /* ==== BUTTON2DOWN is sent when user presses and releases mouse button 2. ==== */
     case WM_BUTTON2DOWN:
	/* Select all dice for rolling */
	val=1;
do_all: if (dice_vals[TURNS]>2) goto do_err;
	for (i=0; i<DICE; i++)
	{
	   /* Set state */
	   selected[i]=val;

	   /* Set the die control's new bitmap */
	   set_die_state(i);
	}
	break;

     /* ==== BUTTON3DOWN is sent when user presses and releases mouse button 3. ==== */
     case WM_BUTTON3DOWN:
	/* Deselect all dice */
	val=0;
	goto do_all;

     /* ==== Called when player presses F1 (and OS/2 send WM_HELP instead of WM_CHAR) ==== */
     case WM_HELP:
	helpFlg=1;
	break;

hdef:
     /* Return msg to OS for default action */
     default:
	return(WinDefWindowProc(hwnd, msg, mp1, mp2));
   }

hdone:
   /* All window procedures return 0 to the OS to indicate that it was handled. */
   return(0);
}



/********************************** main() ********************************
 * Initializes the PM environment, opens the GM driver, creates the main window, and polls the
 * message queue.
 *
 *		 - obtains anchor block handle and creates msg queue
 *		 - Installs the routine exitProc() into the DosExitList chain
 *		 - performs cmd line processing (ie, driver name)
 *		 - Opens GM driver, and initializes GM Sound Module
 *		 - registers my window class
 *		 - creates the main Frame window and loads the main client window
 *		 - polls the msg queue via Get/Dispatch Msg loop
 *		 - upon exiting the loop, ends program
 *
 * Returns:  0 - if successful execution completed
 *	     1 - if error
 *************************************************************************/

int main(int argc, char *argv[], char *envp[])
{
   QMSG   qmsg;       /* msg structure */
   FRAMECDATA fcd;  /* frame control data */
   SWP	  swp;
   HPS	  hps;
   ULONG  Action;

   /* PM initialize stuff */
   if(!(hab = WinInitialize(0)))
   {
err1: DosBeep(60, 100);
      return(1);
   }

   if(!(hmq = WinCreateMsgQueue(hab, 0)))
   {
      WinTerminate(hab);
      goto err1;
   }

   /* Add exitProc to the exit list to handle the exit processing. If there's an error, then
       terminate the process since there haven't been any resources allocated yet. */
   if(DosExitList(EXLST_ADD, (PFNEXITLIST)exitProc))
   {
      messageBox(HWND_DESKTOP, IDMSG_CANNOTLOADEXITLIST);
      DosExit(EXIT_PROCESS, 1);
   }
   /* NOTE: Clean up from here on is handled by the DosExitList processing */

   /* Store name of GM driver supplied by user on command line or from PM Settings. Use default
       if no supplied name */
   if (argc > 1)
   {
       drv_name=argv[1];
   }

   /* Register the Yahtzee window class: QWERTY keys. */
   if(!WinRegisterClass(hab, "Yahtzee", (PFNWP)wndProc, CS_SIZEREDRAW, 0))
   {
      messageBox(HWND_DESKTOP, IDMSG_INITFAILED);
      return(1);
   }

   /* Create the main (FRAME) window */
   fcd.cb = sizeof(FRAMECDATA);
   fcd.flCreateFlags = FCF_SYSMENU|FCF_TITLEBAR|FCF_MINBUTTON|FCF_MENU|FCF_ACCELTABLE|FCF_TASKLIST|FCF_BORDER;
   fcd.hmodResources = 0;
   fcd.idResources = ID_FRAME;
   if (!(hwndYahFrame = WinCreateWindow(HWND_DESKTOP, WC_FRAME,
	 "Yachtzee 1.0  (C) 1993 Jeff Glatt",
	  0,		/* no window style (ie, not visible yet. We do that later with SWP_SHOW) */
	  0, 0, 	/* position (x,y). Done later with SWP_MOVE */
	  0, 0, 	/* size. Done later with SWP_SIZE */
	  0,		/* owner window */
	  HWND_TOP, /* sibling window */
	  ID_YAH_FRAME, /* window id */
	  &fcd, 	/* control data -- give me titlebar and other controls */
	  0)))		/* no presentation parms */
   {
err2: messageBox(HWND_DESKTOP, IDMSG_MAINWINCREATEFAILED);
      return(1);
   }

   /* Set text background color to CYAN */
   Action=CLR_CYAN;
   WinSetPresParam(hwndYahFrame, PP_BACKGROUNDCOLORINDEX, sizeof(ULONG), (PVOID)&Action);

    /* Size, Move, and Show this FRAME window */
    WinSetWindowPos(hwndYahFrame, 0, 10, 10, 270, 425, SWP_ACTIVATE|SWP_MOVE|SWP_SIZE|SWP_SHOW);

    if(!(hwndYah = WinLoadDlg(hwndYahFrame, 0, (PFNWP)wndProc, 0, ID_YAH_WIN, 0)))
       goto err2;
   /* Move client window so it better fits within the client area of FRAME. Then, show it */
   WinQueryWindowPos(hwndYahFrame, &swp);
   WinSetWindowPos(hwndYah, HWND_TOP, 1, 1,
	swp.cx, swp.cy - (SV_CYMENU+1), SWP_MOVE|SWP_SIZE|SWP_SHOW);

   /* Load die bitmaps */
   hps = WinGetPS(hwndYah);
   for (Action=0; Action<7; Action++)
   {
      DieBmps[Action] = GpiLoadBitmap(hps, 0, Action+ID_BMP1, 32, 32);
   }
   WinReleasePS(hps);

   /* Open GM driver */
   openGM();

   /* Initialize help stuff */
   initHelp();

   /* Prepare for new game */
   initGame();

   /* Get/Dispatch msg loop. We get the next msg in our queue, and dispatch it to an OS
       function which routes that msg to the appropriate Window process function (ie, there are
       OS functions for Frame windows and controls, as well as our own Window classes and
       their functions. */
   while(WinGetMsg(hab, &qmsg, 0, 0, 0)) WinDispatchMsg(hab, &qmsg);

   return(0);
}
