#ifndef DPH_PORT_H
#define DPH_PORT_H

#include <dos.h>        // BORLAND STRINGS CLASS HEADER
#include <cstring.h>    // BORLAND STRINGS CLASS HEADER
#include "dphtime.h"    // TIME CLASS HEADER
#include "dphlist.h"    // LIST TEMPLATE CLASS HEADER

// #DEFINE String & typedef String list template
#define String string
typedef ListOf<String> StringList;

// TYPEDEF THE UNSIGNED INT FOR PORT USAGE
typedef unsigned short WORD;

// MASKS TO DETECT NUMBER OF EXISITING PORTS
const BIOS_4PORTS_MASK = 2048;
const BIOS_3PORTS_MASK = 1536;
const BIOS_2PORTS_MASK = 1024;
const BIOS_1PORTS_MASK = 512;
const BIOS_0PORTS_MASK = 0;

// UART REGISTER MASKS
const unsigned BAUD300 = 384;       // DIVISOR FOR 300 BAUD OPERATION
const unsigned BAUD1200 = 96;       // DIVISOR FOR 1200 BAUD OPERATION
const unsigned BAUD2400 = 48;       // DIVISOR FOR 2400 BAUD OPERATION
const unsigned BAUD4800 = 24;       // DIVISOR FOR 4800 BAUD OPERATION
const unsigned BAUD9600 = 12;       // DIVISOR FOR 9600 BAUD OPERATION
const unsigned _DATABITS8 = 3;      // MASK FOR  8 BITS OF DATA
const unsigned _DATABITS7 = 2;      // MASK FOR  7 BITS OF DATA
const STOPBITS1 = 1;
const STOPBITS2 = 2;
const _DLAB = 0x0080;      // MASK FOR DIVISOR LATCH ACCESS BIT
const _THRE = 0x0020;      // MASK FOR TRANSMIT HOLDING REGISTER EMPTY BIT
const _DTR = 0x0001;       // MASK FOR DATA TERMINAL READY
const _RTS = 0x0002;       // MASK FOR REQUEST TO SEND
const BREAK_ON = 64;       // MASK TO TURN ON BREAK BIT IN LCR
const BREAK_OFF = 191;     // MASK TO TURN OFF BREAK BIT IN LCR


//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//   class PolledSerialPort
//   Implements the basics of a PC serial communications port, including
//   setting port address, baud, parity, stop, data, and the ability
//   to condition registers in the UART.  It does not make any use of IRQs.
//   copyright David Perelman-Hall 1995
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------

class PolledSerialPort {
   friend istream& operator >> (istream& is, PolledSerialPort& sp);
   friend ostream& operator << (ostream& os, const PolledSerialPort& sp);

   // CAN'T COPY
   PolledSerialPort(const PolledSerialPort& p);
   PolledSerialPort& operator=(const PolledSerialPort& p);

   public:
      // CTOR, DEFAULT SETS UP PORT W/OUT ADDRESS OR BAUD RATE
      PolledSerialPort(WORD port=0, unsigned baud=0);

		// DTOR
		virtual ~PolledSerialPort() {}

      // GET PORT SETTINGS
      WORD Address() const { return baseAddress; }
      WORD Data() const { return data; }
      WORD Parity() const { return parity; }
      WORD Stop() const { return stop; }
      WORD Baud() const { return baud; }
      const String& BaudText() const { return baudAsText; }
      const String& PortText() const { return portAsText; }

      // SET PORT SETTINGS
      void SetBaud( const String& baudStr );
      void SetData( const String& dataStr );
      void SetStop( const String& stopStr );
      void SetParity(const String& parityStr );
      void SetPort( const String& portStr );
      void SetAddress( const WORD address ) { baseAddress=address; }
      void SetBaudText( const String& baud ) { baudAsText=baud; }
      void SetPortText( const String& port ) { portAsText=port; }

      // REGISTER VALUES
      WORD RBR() const { return baseAddress; }    // Receive Buffer
      WORD THR() const { return baseAddress; }    // Transmit Hold
      WORD DLAB() const { return baseAddress; }   // Divisor Latch Bit
      WORD DLBM() const { return baseAddress+1; } // Divisor Latch Bit M
      WORD IER() const { return baseAddress+1; }  // Interrupt Enable
      WORD LCR() const { return baseAddress+3; }  // Line Control
      WORD MCR() const { return baseAddress+4; }  // Modem Control
      WORD LSR() const { return baseAddress+5; }  // Line Status
      WORD MSR() const { return baseAddress+6; }  // Modem Status

      // LINE USAGE
      void RaiseDTR() const  {
         unsigned char val=inportb(MCR()); 
         outportb(MCR(), val|1);
      }

      void LowerDTR() const {
         unsigned char val=inportb(MCR()); 
         outportb(MCR(),val&254);
      }

      void RaiseRTS() const {
         unsigned char val=inportb(MCR()); 
         outportb(MCR(),val|2);
      }

      void LowerRTS() const {
         unsigned char val=inportb(MCR()); 
         outportb(MCR(),val&253);
      }

      // PORT USAGE
      virtual void OpenPort(); // does work of talking to UART registers
      virtual void ClearPort() const;
      virtual void SetBaudRate();
      int ReadyToSendChar() const { return (inportb(LSR())& _THRE)==_THRE; }
      int CharWaiting() const { return inportb(LSR()) & 0x0001; }
      void SendChar( const unsigned char& ch ) const { outportb(THR(), ch); }
      unsigned char ReadChar() const { return inportb(RBR()); }
      void SendString( const String& ) const;
      void BreakOn() const;
      void BreakOff() const;

   protected:
      static StringList portsInUse;
      String baudAsText, portAsText;
      WORD baseAddress, data, parity, stop, baud;
      DPH_Time timer;
      int portIsOpen;
};

#endif // DPH_PORT_H
