#include "misc.h"    // CLASS HEADERS AND MISCELLANEOUS INCLUDES
#pragma hdrstop      // STOP PRECOMPILED HEADERS
#include "sdi12.h"

//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//   class SDI12_UART
//   Descends from PolledSerialPort class.  
//   copyright David Perelman-Hall 1995
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
void SDI12_UART::Setup( const String& portStr )
{
   // SET UP THE SDI-12 COMMUNICATIONS PARAMETERS
	SetPort( portStr );  // port is variable between COM1 and COM4
  	SetBaud( "1200" );
	SetData( "7" );
	SetParity( "even" );
	SetStop( "1" );

   // CALL PolledSerialPort::OpenPort TO PHYSICALLY SET UP THE UART REGISTERS
   OpenPort();
}


void SDI12_UART::SendCommand( const String& command ) 
const
{
  timer.Pause18thSeconds(1); // THIS ENSURES THAT BREAK STARTS OK
  BreakOn(); // SEND BREAK AND MARK TO WAKE UP INSTRUMENT
  timer.Pause18thSeconds(1);
  BreakOff(); // STOP BREAK AND MARK
  timer.Pause18thSeconds(1);
  SendString( command ); // SEND COMMAND STRING

  // CHECK FOR 1 18TH OF SECOND THAT CHARACTER IS COMING IN FROM SENSOR
  // IF NO CHARACTER IS PRESENT, THEN RETRY THE COMMAND
  long currentTime = timer.Get18thSeconds();
  while( !timer.Past18thSeconds(currentTime,1) ) {
     if( CharWaiting() )
        return;  // RETURN IMMEDIATELY TO COLLECT THE DATA FROM SENSOR
  }

  // REISSUE THE COMMAND AND RETURN FROM FUNCTION TO TRY TO COLLECT THE DATA
  if( !CharWaiting() )
     SendString( command );
}

int SDI12_UART::GetTerminatedString( String& str, int delayTime, char 
terminator, int length ) 
const
{
   char *buf = new char[length+1];
   if( buf==0 )
      assert( !"char allocation failed" );
   register char c = terminator-1;
   register int i=0;
   int currentTime = timer.GetSeconds();
   while( !timer.PastSeconds(currentTime,delayTime) && i<length && 
c!=terminator ) {
      if( CharWaiting() ) {
         buf[i] = c = ReadChar();
         i++;
      }
   }
   buf[i] = 0;
   str = buf;
   delete [] buf;
   if( c==terminator )
      return 1;
   return 0;
}


//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//   class SDI12_Logger
//   Inherits privately from template class 
//   copyright David Perelman-Hall 1995
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------

String SDI12_Logger::Verify()
const 
{ 
   return TalkToSDI(address+"V!"); 
}

String SDI12_Logger::Acknowledge()
const 
{ 
   return TalkToSDI( address + "!");
}

String SDI12_Logger::Identify()
{ 
   return identity = TalkToSDI( address + "I!"); 
}

String SDI12_Logger::Data( const String& consecutive ) 
{
   return value = TalkToSDI( address + "D" + consecutive + "!" ); 
}

String SDI12_Logger::Measure( const String& consecutive )
const 
{ 
   return TalkToSDI( address + "M" + consecutive + "!" ); 
}

SDI12_Logger::SDI12_REPLY SDI12_Logger::DataReady(const String& str)
{
   // THIS FUNCTION SEEKS A SERVICE REQUEST FROM THE SDI12 OBJECT
   // A SERVICE REQUEST CONSISTS OF "address<CR><LF>"
   // AND IT MUST COME BEFORE THE MAXIMUM TIME THE SENSOR WILL TAKE
   // BEFORE DATA IS READY, AS REPORTED IN REPLY TO A MEAASURE COMMAND.
   // PARSE STRING TO SEE HOW LONG SHOULD WAIT BEFORE TIMING OUT
   if( !ParseReply( str ) )
      return SDI12_Logger::REPLY_FAIL;

   // SEEK THE DATA READY MESSAGE, WAITING timeOut SECONDS FOR IT
   String reply;
   if( method->GetTerminatedString( reply, serviceRequestTime, LF, 
MAX_BUF_SIZE ) ) {
      if( reply.contains(replyTerminal) )
         return SDI12_Logger::REPLY_OK;
      else
         return SDI12_Logger::REPLY_FAIL;
   }
   else
      return SDI12_Logger::REPLY_TIMEOUT;
}

int SDI12_Logger::ParseReply( const String& str ) 
{
   cout << "Parsing reply of " << str << endl;
   String s = str;

   // PARSE OUT THE TIME TILL DATA IS AVAILABLE
   // AND PARSE OUT THE NUMBER OF DATA VALUES
   char time[4] = {0,0,0,0}, numVals[2] = {0,0}; // zero out these arrays
   s.copy( time, 2, s.length() - 4 );
   s.copy( numVals, 1, s.length() - 1 );
   serviceRequestTime = atoi( time );
   numberValues = atoi( numVals );
   cout << "Service request within " << serviceRequestTime << " seconds.\n";
   cout << numberValues << " data value(s) will be available.\n";

   // DO A SANITY CHECK TO ENSURE DATA WAS OK
   if( serviceRequestTime && numberValues )
      return 1;
   return 0;
}
