#ifndef _SDI-12_H_
#define _SDI-12_H_

#include "dphport.h"    // POLLED SERIAL PORT OBJECT CLASS HEADER

// CONSTANTS
const int MAX_BUF_SIZE = 80; // size of one screen's width

//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//   The SDI12_UART class inherits from the PolledSerialPort class in order
//   to implement SDI-12 specific commands.  This leaves the serial port
//   class unsullied by extraneous behavour which it shouldn't have to
//   implement, and therefore it is free to be used by other classes or
//   programs which need a plain serial port class.
//        copyright David Perelman-Hall 1995
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------

class SDI12_UART : public PolledSerialPort
{
   friend ostream& operator << (ostream& os, const SDI12_UART& m) {
      return os << (const PolledSerialPort&)m;
   }

   // CAN'T COPY--MAKES NO SENSE TO COPY A SERIAL PORT
   SDI12_UART( const SDI12_UART& );
   SDI12_UART& operator=( const SDI12_UART& );

   public:

      // CTOR
      SDI12_UART() : PolledSerialPort() {}

      // DTOR
      virtual ~SDI12_UART() {}

      // USAGE WHICH EMPLOYS THE UART--THESE FOLLOWING FOUR FUNCTIONS
      // MUST BE DEFINED IN ANY CLASS WHICH IS TO BE USED TO
      // INSTANTIATE THE SDI-12 COMMUNICATION METHOD CLASS TEMPLATE
      void SendCommand(const String& command) const;
      int GetTerminatedString( String& str, int delayTime, char terminator, 
int length=100 ) const;
      int IsValid() const { return portIsOpen==1; }
      void Setup( const String& );
};

//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//   The template class below must be instantiated when constructing an
//   SDI12_Logger object.  It gets instantiated with the class which
//   actually does the talking to the SDI-12 sensor.  In the case of this
//   application, it is the SDI12_UART class which does the talking, and 
//   which inherits from a very simple polled (not irq-driven) serial port
//   class intended to be used in standard PCs.  Other applications might
//   implement the communication method code in firmware or EPROM, talking
//   to a dedicated set of lines.  In that case, the CommunicationMethod
//   object may not be a UART in a PC, and the template will need to be
//   instantiated with a class which captures the dedicated line's
//   communication method.
//   The four functions which must be implemented in the Method class are:
//
//   The first 2 are used by the template class in TalkToSDI
//   1) void SendCommand( const String& command ) const;
//   2) int GetTerminatedString( String& str, int delayTime, char 
terminator, int replyLength ) const;
//   
//   The last 2 are used by the SDI12_Logger class
//   3) int IsValid() const;
//   4) void Setup( const String& someSetupStuff );
//        copyright David Perelman-Hall 1995
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------

template < class Method >
class CommunicationMethod
{
   friend ostream& operator << (ostream& os, const 
CommunicationMethod<Method>& m) {
      return os << *(m.method);
   }

   // CAN'T COPY
   CommunicationMethod( const CommunicationMethod< Method >& );
   CommunicationMethod& operator=( const CommunicationMethod< Method >& );

   public:
      // CTOR & ASSIGNMENT
      CommunicationMethod(): method( new Method() ) { assert( method!=0 ); }

      // DTOR
      virtual ~CommunicationMethod() { delete method; }

      // USAGE
      String TalkToSDI( const String& command, int time=1 ) const;

   protected:
      // POINTER TO CLASS WHICH ACTUALLY DOES COMMUNICATING
      Method *method; 
};


template < class Method >
String CommunicationMethod< Method >::TalkToSDI( const String& command, int 
time=10 )
const
{
   cout << "TalkToSDI sending out string [" << command << ']' 
   << " and waiting " << time << " seconds for reply" << endl;

   // SEND COMMAND STRING.  
   // CLASS INSTANTIATING THE CommunicationMethod TEMPLATE CLASS 
   // MUST IMPLEMENT SendCommand FUNCTION 
   method->SendCommand( command );

   // GET REPLY.
   // CLASS INSTANTIATING THE CommunicationMethod TEMPLATE CLASS 
   // MUST IMPLEMENT GetTerminatedString FUNCTION
   String reply;
   method->GetTerminatedString( reply, time, LF, MAX_BUF_SIZE );

   // REMOVE BACK OF REPLY BEYOND THE DELIMITING <CR><LF>
   unsigned pos = reply.rfind( "\r\n" );
   if( pos != NPOS )
      reply.resize( pos );
   else {
      reply.resize(0); // null out the value as being invalid
      return reply;
   }

   // PARSE OFF THE COMMAND FROM THE FRONT OF THE DATA
   pos = reply.find( "!" );
   if( pos != NPOS ) {
      reply.remove( 0, pos+1 );
   }

   cout << "TalkToSDI replied [" << reply << ']' << endl;
   return reply;
}

//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
//   The SDI12_Logger class class inherits privately from the
//   CommunicationMethod template class.  Inheritance from a template allows 
//   for a wide family of communication methods.  Private inheritance models
//   the "has-a" relationship instead of the "is-a" relationship, which
//   means that the base class is not accessible to the users of the
//   SDI12_Logger class.  This is good because SDI12_Loggers certainly
//   are not communication methods.  Nevertheless, it simplifies use of 
//   the communication method because of the inheritance.
//        copyright David Perelman-Hall 1995
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------


class SDI12_Logger : private CommunicationMethod< SDI12_UART > 
{
   // READ IN AN SDI12_Logger FROM INPUT STREAM REFERENCE "istream& is"
   friend istream& operator >> (istream& is, SDI12_Logger& sdiO) {
      getline( is, sdiO.model );
      is >> sdiO.address;
      is >> sdiO.header;
      is >> sdiO.measure;
      return is;
   }
   // WRITE OUT AN SDI12_Logger ONTO OUTPUT STREAM REFERENCE "ostream& os"
   friend ostream& operator << (ostream& os, const SDI12_Logger& sdiO) {
      os << "SDI-12 MODEL....: " << sdiO.model << endl
         << "SDI-12 IDENTITY.: " << sdiO.identity << endl
         << "SDI-12 VALUE....: " << sdiO.value << endl
         << "SDI-12 HEADER...: " << sdiO.header
         << "\t\tSDI-12 ADDRESS..: " << sdiO.address << endl
         << *(sdiO.method);
      return os;
   }

   // CAN'T COPY
   SDI12_Logger( const SDI12_Logger& );
   SDI12_Logger& operator=( const SDI12_Logger& );

   public:

     // ENUM DESIGNATING STATES APPLICABLE WHEN WAITING ON SERVICE REQUEST
     enum SDI12_REPLY { REPLY_FAIL, REPLY_OK, REPLY_TIMEOUT };

      // CTOR
      SDI12_Logger()
         : replyTerminal( "\r\n" ),
         address( "0" ),
         model( "unspecified" ),
         value( "none"  ),
         header( "none"  ),
         measure( "M" ),
         identity( "unidentified" ),
         serviceRequestTime(-1),   // construct with known bad value
         numberValues(-1)          // construct with known bad value
         { method->LowerRTS(); }   // RTS low for NR Systems SDI-12 Verifier

      // DTOR
      virtual ~SDI12_Logger() {}

      // SDI-12 USAGE
      String Verify() const;
      String Acknowledge() const;
      String Measure( const String& consecutive = "" ) const;
      String Identify();
      String Data( const String& consecutive="0" );
      SDI12_REPLY DataReady( const String& );
      int ParseReply( const String& );

      // CONST ACCESS TO DATA MEMBERS
      const String& Model() const { return model; }
      const String& Value() const { return value; }
      const String& Header() const { return header; }
      const String& MeasureCommand() const { return measure; }
      const String& Address() const { return address; }
      int ServiceRequestTime() const { return serviceRequestTime; }
      int NumberValues() const { return numberValues; }

      // CLASS INSTANTIATING THE CommunicationMethod TEMPLATE CLASS 
      // MUST IMPLEMENT THE IsValid MEMBER FUNCTION
      int IsValid() const { return method->IsValid(); }

      // CLASS INSTANTIATING THE CommunicationMethod TEMPLATE CLASS 
      // MUST IMPLEMENT THE Setup MEMBER FUNCTION
      void Setup( const String& str ) { method->Setup( str ); }

   private:
      const String replyTerminal;
      String address, model, value, header, measure, identity;
      int serviceRequestTime, numberValues;
};

#endif //_SDI-12_H_
