/*  ALL1BYTE.C  -  allocate 1 byte at a time until memory exhausted.  */


#define  INCL_DOSMEMMGR
#include <os2.h>
#include <stdio.h>


main()
{
  char *cP;
  int  count = 0;

  printf("Allocating until memory exhausted.\n");
  while ( !DosAllocMem((void **)&cP,1,PAG_READ|PAG_WRITE) )
    count++;
  printf("%d one byte blocks allocated.\n",count);

  return 0;
}


/*
 *  Output (your mileage may vary):
 *
 *  3944 one byte blocks allocated.
 */


------------------------------------------------------------------------


/*  TILEDMEM.C  -  tiled memory, you always get at least 64 Kb.  */


#define  INCL_DOSMEMMGR
#include <os2.h>
#include <stdio.h>


main()
{
  char *c1P, *c2P, *c3P;

  /*  Three allocations, 1 byte each  */
  DosAllocMem((void **)&c1P,1,PAG_READ|PAG_WRITE);
  DosAllocMem((void **)&c2P,1,PAG_READ|PAG_WRITE);
  DosAllocMem((void **)&c3P,1,PAG_READ|PAG_WRITE);
  printf("c1P = 0x%08x\nc2P = 0x%08x\nc3P = 0x%08x\n",c1P,c2P,c3P);

  DosFreeMem(c1P);
  DosFreeMem(c2P);
  DosFreeMem(c3P);

  return 0;
}


/*
 *  Output:
 *
 *  c1P = 0x000e0000
 *  c2P = 0x000f0000
 *  c3P = 0x00100000
 *
 */


------------------------------------------------------------------------


/*  ALL1INT.C  -  allocate one int, use 64 Kb.  */


#define  INCL_DOSMEMMGR
#include <os2.h>
#include <stdio.h>


main()
{
  int *iP, i;

  /*  Allocate one int  */
  DosAllocMem((void **)&iP,sizeof(int),PAG_READ|PAG_WRITE);

  /*  Commit and use 64 Kb  */
  DosSetMem(iP,65536,PAG_DEFAULT|PAG_COMMIT);
  for ( i = 0; i < 65536/sizeof(int); i++ )
    iP[i] = i;

  /*  Print int samples  */
  printf("first:  %d\n",iP[0]);
  printf("middle: %d\n",iP[65536/sizeof(int)/2]);
  printf("last:   %d\n",iP[65536/sizeof(int)-1]);

  DosFreeMem(iP);

  return 0;
}


/*
 *  Output:
 *
 *  first:  0
 *  middle: 8192
 *  last:   16383
 *
 */


------------------------------------------------------------------------


/*  MEMMAP.C  -  memory mapping with DosQueryMem().  */


#define INCL_DOSMEMMGR
#include <os2.h>
#include <stdio.h>

#define MAXRGN     (512*1024*1024)
#define TILECHUNK  (64*1024)


main()
{
  char   *base = 0;
  ULONG  flags,
         region = MAXRGN;

  while ( (int)base < MAXRGN ) {
    if ( !DosQueryMem(base,&region,&flags) ) {
      printf("base = 0x%08x   region = %u\tflags = 0x%08x\n",base,region,flags);
      if ( flags & PAG_BASE )    printf("\t\t\t\t\t\t\tPAG_BASE\n");
      if ( flags & PAG_COMMIT )  printf("\t\t\t\t\t\t\tPAG_COMMIT\n");
      if ( flags & PAG_FREE )    printf("\t\t\t\t\t\t\tPAG_FREE\n");
      if ( flags & PAG_SHARED )  printf("\t\t\t\t\t\t\tPAG_SHARED\n");
      if ( flags & PAG_READ )    printf("\t\t\t\t\t\t\tPAG_READ\n");
      if ( flags & PAG_WRITE )   printf("\t\t\t\t\t\t\tPAG_WRITE\n");
      if ( flags & PAG_EXECUTE ) printf("\t\t\t\t\t\t\tPAG_EXECUTE\n");
      if ( flags & PAG_GUARD )   printf("\t\t\t\t\t\t\tPAG_GUARD\n");
      base += region;
    }
    else
      base += TILECHUNK;

    region = MAXRGN - (int)base;
  }

  return 0;
}


------------------------------------------------------------------------


/*
 *  HEAPMGR.H
 *  Header file for 32-bit OS/2 heap management library, 8/95
 *
 *  Jens A. Jensen, Sunway Software ApS
 *  jaj@inet.uni-c.dk
 *
 */
#ifndef _H_HEAPMGR_
#define _H_HEAPMGR_

#include <stdlib.h>    /*  for size_t  */

/*
 *  The incomplete HeapManager control struct
 *
 */
struct _heap;
#define HEAP struct _heap

/*
 *  Prototypes
 *
 */
#ifdef __cplusplus
extern "C" {
#endif

HEAP * createHeap(size_t size);
int    destroyHeap(HEAP *heap);

void * allocMem(HEAP *heap, size_t size);
int    freeMem(HEAP *heap, void *blockP);

size_t queryHeapSize(HEAP *heap);
size_t queryHeapFree(HEAP *heap);
size_t queryHeapBlocks(HEAP *heap);
void * queryHeapBase(HEAP *heap);
size_t queryBlockSize(void *blockP);

#ifdef __cplusplus
}
#endif

#endif   /*  _H_HEAPMGR_  */


------------------------------------------------------------------------


/*
 *  HEAPMGR.C
 *  32-bit OS/2 heap management library, 8/95
 *
 *  Jens A. Jensen, Sunway Software ApS
 *  jaj@inet.uni-c.dk
 *
 */
#define  INCL_DOSMEMMGR
#define  INCL_DOSPROCESS
#include <os2.h>

#include <stdio.h>
#include "heapmgr.h"


/*
 *  Completion of struct _heap
 *
 */

HEAP {
  size_t size,
         free,
         blocks;
  void   *base;
};


/*
 *  Round up to boundary
 *  (does not round negative numbers correctly)
 *
 */

#define ROUNDUP(num,bdry)  ((((num)+(bdry)-1)/(bdry))*(bdry))


/* * * * * * *  create & destroy heap  * * * * * * */


HEAP *createHeap(size_t size)
{
  HEAP heap,
       *retval;

  size = ROUNDUP(size,64*1024);             /*  DosAllocMem() rounds anyway  */

  heap.size   = size;
  heap.free   = size - 64;                /*  size less system control info  */
  heap.blocks = 0;
  heap.base   = 0;

  if ( size == 0 || DosAllocMem(&heap.base,size,PAG_READ | PAG_WRITE) )
    return 0;

  if ( DosSubSetMem(heap.base,
                    DOSSUB_INIT | DOSSUB_SPARSE_OBJ | DOSSUB_SERIALIZE,
                    size) ) {
    DosFreeMem(heap.base);
    return 0;
  }

  if ( !DosSubAllocMem(heap.base,(void **)&retval,sizeof(heap)) ) {
    heap.free -= sizeof(heap) + sizeof(int);
    *retval = heap;                        /*  place control struct in heap  */
  }
  else {            /*  shouldn't occur, DosSubAllocMem() shouldn't fail...  */
    DosSubUnsetMem(heap.base);
    DosFreeMem(heap.base);
  }

  return retval;
}


int destroyHeap(HEAP *heap)
/*  Returns OS/2 error code as a negative number on failure,  */
/*  otherwise returns the number of unfreed blocks.           */
{
  void   *base  = heap->base;          /*  after deallocation, HEAP content  */
  size_t blocks = heap->blocks;        /*  can no longer be trusted          */
  APIRET rc;

                                                    /*  free HEAP structure  */
  if ( (rc = DosSubFreeMem(base,heap,sizeof(HEAP))) == 0 )
    if ( (rc = DosSubUnsetMem(base)) == 0 )   /*  clear system control info  */
      rc = DosFreeMem(base);                           /*  free heap memory  */

  return !rc ? blocks : rc * -1;
}


/* * * * * * * *   alloc & free  * * * * * * * * */


void *allocMem(HEAP *heap, size_t size)
{
  size_t *p,
         totsize = size+sizeof(size_t); /*  user mem plus room to keep size  */

  if ( size == 0 ||
       DosSubAllocMem(heap->base,(void **)&p,totsize) )
    return 0;

  *p = size;                            /*  keep size in front of user area  */

  DosEnterCritSec();
  heap->free -= ROUNDUP(totsize,sizeof(int)) + sizeof(void *);
  heap->blocks++;
  DosExitCritSec();

  return  p + 1;                          /*  return a pointer to user area  */
}


int freeMem(HEAP *heap, void *blockP)
{
  void   *blckbase = (char *)blockP-sizeof(size_t);
  size_t size      = *(size_t *)blckbase,
         totsize   = size + sizeof(size_t);

  if ( DosSubFreeMem(heap->base,blckbase,totsize) )  /*  failure is sign of  */
    return FALSE;                                    /*  heap corruption     */

  DosEnterCritSec();
  heap->free += ROUNDUP(totsize,sizeof(int)) + sizeof(void *);
  heap->blocks--;
  DosExitCritSec();

  return TRUE;
}


/* * * * * * * * *   utilities   * * * * * * * * */


size_t queryHeapSize(HEAP *heap)
{
  return heap->size;
}


size_t queryHeapFree(HEAP *heap)
{
  return heap->free;
}


size_t queryHeapBlocks(HEAP *heap)
{
  return heap->blocks;
}


void *queryHeapBase(HEAP *heap)
{
  return heap->base;
}


size_t queryBlockSize(void *blockP)
{
  return *(size_t *)((char *)blockP-sizeof(size_t));
}


------------------------------------------------------------------------


/*  HEAPUSE.C  -  conceptual use of the HeapManager.  */


#include "heapmgr.h"
/*  link with heapmgr.obj  */

main()
{
  HEAP *heapP;
  char *blockP;

  heapP = createHeap(100000);

  blockP = allocMem(heapP,100);
  /*  do something with blockP  */
  freeMem(heapP,blockP);

  destroyHeap(heapP);

  return 0;
}


------------------------------------------------------------------------


/*
 *  HEAPDEMO.C
 *  Demo of the 32-bit OS/2 heap management library, 8/95
 *  Link with HEAPMGR.OBJ
 *
 *  Jens A. Jensen, Sunway Software ApS
 *  jaj@inet.uni-c.dk
 *
 */
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "heapmgr.h"

#define HEAPNUM      4
#define ALLOCNUM     500
#define ITERATIONS   10000
#define MAXHEAPSIZE  2000000

void printHeaps(void);

HEAP *heapArr[HEAPNUM];               /*  may be local variables in main() if the linker  */
char *allocMtx[HEAPNUM][ALLOCNUM];    /*  is told to provide sufficient stack space       */


main()
{
  size_t size,
         i,
         heapNum,
         allocNum,
         allocSize;
  int    rc;

  srand(time(0));    /*  seed rand()  */

  /*
   *  Create heaps
   *
   */
  printf("\nCreate %d heaps...\n",HEAPNUM);
  for ( i = 0; i < HEAPNUM; i++ ) {
    size = rand() * rand() % MAXHEAPSIZE;
  if ( !(heapArr[i] = createHeap(size)) )
      printf("Cannot create heap %u.\n",i);
  }
  printHeaps();

  /*
   *  Make some allocs and frees
   *
   */
  printf("\nMake some allocs and frees...\n");
  memset(allocMtx,0,sizeof(allocMtx));
  for ( i = 0; i < ITERATIONS; i++ ) {
    heapNum = rand() % HEAPNUM;
    allocNum = rand() * rand() % ALLOCNUM;
    if ( allocMtx[heapNum][allocNum] ) {
      freeMem(heapArr[heapNum],allocMtx[heapNum][allocNum]);
      allocMtx[heapNum][allocNum] = 0;
    }
    else {
      allocSize = rand();
      allocMtx[heapNum][allocNum] = allocMem(heapArr[heapNum],allocSize);
      if ( allocMtx[heapNum][allocNum] )
        memset(allocMtx[heapNum][allocNum],-1,allocSize);
    }
  }
  printHeaps();

  /*
   *  Dealloc everything
   *
   */
  printf("\nDealloc everything...\n");
  for ( heapNum = 0; heapNum < HEAPNUM; heapNum++ )
    for ( allocNum = 0; allocNum < ALLOCNUM; allocNum++ )
      if ( allocMtx[heapNum][allocNum] )
        freeMem(heapArr[heapNum],allocMtx[heapNum][allocNum]);
  printHeaps();

  /*
   *  Destroy heaps
   *
   */
  printf("\nDestroy all heaps...\n");
  for ( i = 0; i < HEAPNUM; i++ )
    if ( heapArr[i] )
      if ( (rc = destroyHeap(heapArr[i])) < 0 )
        printf("Cannot destroy heap %u. rc = %d.\n",i,rc*-1);
      else
        printf("%d block%s still allocated. Heap %u destroyed%s.\n",rc,rc==1?"":"s",i,rc==0?"":" anyway");
  return 0;
}


void printHeaps()
{
  int i;

  printf("Heap        HEAP *        Base       Size       Free     Blocks\n");
  printf("---------------------------------------------------------------\n");
  for ( i = 0; i < HEAPNUM; i++ )
    printf("%u\t0x%08x  0x%08x  %9u  %9u  %9u\n",i,
           heapArr[i],
           queryHeapBase(heapArr[i]),
           queryHeapSize(heapArr[i]),
           queryHeapFree(heapArr[i]),
           queryHeapBlocks(heapArr[i]));
}

