// This may look like C code, but it is really -*- C++ -*-
/*
 ************************************************************************
 *
 *	   Relaying TCP packets from one computer/port to another
 *		      and doing snooping at the same time
 *
 * The title says it all: this code relays TCP packets between two different
 * computers (and/or ports). That is, the program listens on port Listen_port.
 * After it got a connection, it connects to Target_host:Target_port, and,
 * if everything goes well, the program relays packets between the two
 * connections, and also, it keeps a log of all packets received/sent.
 * After the connection is closed (normally or semi-normally) the program
 * listens again.
 *
 * Possible applications of the program: relaying packets from one port
 * to another (say, some netnazi sites, in a stupid attempt to prevent people
 * from playing MUD, firewall all connections to "suspicios" ports,
 * usually ones above 6000). So, one can run this program on some computer
 * (outside the firewall) and have it take packets from some legitimate
 * port, say, 25 or 119, and relay them to the "real" port, 9000 or so, on a
 * MUD machine.
 * Another possible application (which actually I had in mind writing
 * this program) is to snoop on the protocol, say, get a "carbon copy"
 * of the http chat between two hosts.
 *
 * The feature of this program is an asynchronous (by SIGIO) reading
 * of communication channels. So, most of the time the program just
 * sleeps (without taking of any CPU time) and wakes up only when
 * some packet shows up in one of the channels. The program also contains
 * a fairly complete class library for the TCP (stream) sockets.
 * Another feature of the code is using of a Finite State Machine to
 * do relay and handle all possible special situations.
 *
 * $Id: tcp_relay.cc,v 1.2 1995/01/24 15:30:20 oleg Exp oleg $
 *
 ************************************************************************
 */

#include "myenv.h"
#include <std.h>
#include <fstream.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
extern "C"
{
				// Convertion between the host and the network
				// byte orders
unsigned short htons(unsigned short data);	// For a short data item
unsigned short ntohs(unsigned short data);	// For a short data item
unsigned long  htonl(unsigned long data);	// For a long data item
#include <netinet/in.h>
#include <arpa/inet.h>
}

enum BOOL { FALSE=0, TRUE };

static const char * Target_host = "zaphod.csci.unt.edu";
//static const char * Target_host = "hypatia.gsfc.nasa.gov";
//static const char * Target_host = "pubweb.parc.xerox.com";
static const short Target_port  = 80;		// http port
//static const short Target_port  = 7;		// echo port
static const short Listen_port  = 4003;

				// Make sure that a system call went well
				// If a system call (say, fcntl()) fails
				// it returns -1 and sets the errno
#define do_well(FX) (void)( (FX) < 0 ? (perror("System error"), \
	_error("Failed system call " #FX " at line %d of `%s'.\n", \
	       __LINE__, __FILE__),0) : 1)


/*
 *------------------------------------------------------------------------
 *			  Class IPaddress that provides
 *		      addressing and naming in the IP network
 */

class SocketAddr;

				// IP address
class IPaddress
{
  friend class SocketAddr;

  unsigned long address;		// Address: 4 bytes in the network
					// byte order
  IPaddress(const unsigned int netaddr) : address(netaddr) {}

public:
  IPaddress(void)   { address = INADDR_ANY; }	// Wildcard address
  IPaddress(const char * name);		// Involves the name resolution
  ~IPaddress(void) {}
  unsigned long net_addr(void) const { return address; }
  operator const char * (void) const;	// get a STATIC symbolic representation
					// of an IP address
};

				// Obtains the IP address of the specified
				// host. The host name may be specified either
				// in the dot notation, or as a host "name".
				// Name resolution is performed in the
				// latter case.
IPaddress::IPaddress(const char * host_name)
{
  					// First check to see if the host
					// name is specified in the IP address
					// dot notation
  if( (address = inet_addr(host_name)) != (unsigned long)(-1) )
    return;

  struct hostent *host_ptr = ::gethostbyname(host_name);
  if( host_ptr == 0 )
    _error("Host name '%s' cannot be resolved",host_name);
  if( host_ptr->h_addrtype != AF_INET ) 
    _error("'%s' isn't an Internet site, or so the DNS says",host_name);

  address = *(unsigned long *)(host_ptr->h_addr);
}


				// get a STATIC symbolic representation
				// of an IP address
				// We use the fact that the IP address is in
				// the net order, that is, the MSB first
				// We do reverse name resolution if necessary
IPaddress::operator const char * (void) const
{
  struct hostent *host_ptr = ::gethostbyaddr((char *)&address, 
					     sizeof(address), AF_INET);
  if( host_ptr != 0 )
    return host_ptr->h_name;

					// Reverse DNS failed, use print in 
					// the dot notation
  static char buffer[80];
  sprintf(buffer,"%0d.%0d.%0d.%0d", address & 0xff, (address >> 8) & 0xff, 
	  (address >> 16) & 0xff, (address >> 24) & 0xff );
  return buffer;
}

/*
 *------------------------------------------------------------------------
 *			  	Sockets
 */

class StreamSocket;

class SocketAddr : sockaddr_in
{
  friend class StreamSocket;
  SocketAddr(void) {}

public:
  SocketAddr(const IPaddress host, const short port_no);
  ~SocketAddr(void) {}
  operator sockaddr * (void) const	{ return (sockaddr *)this; }
  operator const char * (void) const;	// Give a STATIC string representation
};

				// This can be either listening (passive)
				// socket or connective (active) socket.
				// That's why we don't have a public
				// constructor here
class StreamSocket
{
public:
  enum Status { Good, Nodata, Eof, Failure };

protected:
  const int socket_handle;		// Socket handle = file handle
  Status status;

  StreamSocket(void);
  StreamSocket(const int _socket_handle);	// wrap a class around

public:
  ~StreamSocket(void);
  SocketAddr get_my_name(void) const;		// Name of this socket
  SocketAddr get_peer_name(void) const;		// Name of the socket this
						// socket is connected to
  void print(const char * title) const;		// Print the socket status

  void set_blocking_io(const BOOL onoff);
  void enable_sigio(const BOOL onoff);

					// Write to the socket, return status
  Status write(const void * buffer, const unsigned int len);
					// Read from the socket returning the
					// status and size of the block read
  struct ReadResult { unsigned int len; Status status; };
  ReadResult read(void * buffer, const unsigned int len);
};

class Listen : public StreamSocket
{
public:
  Listen(const SocketAddr bind_address);
  ~Listen(void) {}
  StreamSocket * accept(const int backlog=1);
};

				// This is a socket to initiate an active
				// TCP open, that is, a connection. The
				// constructor attempts to connect to another
				// (hopefully listening) socket on
				// probably another computer/port
class ConnectingSocket : public StreamSocket
{
public:
  ConnectingSocket(const SocketAddr target_address);
  ~ConnectingSocket(void) {}
};

				// Create an IP address space entity
SocketAddr::SocketAddr(const IPaddress host, const short port_no)
{
  sin_family = AF_INET;
  sin_port = htons((short)port_no);
  sin_addr.s_addr = host.net_addr();
}

			// Give a STATIC string representation of a socket
			// address
SocketAddr::operator const char * (void) const
{
  assert( sin_family == AF_INET );
  static char buffer[80];
  
  sprintf(buffer,"%s:%d",(const char *)IPaddress(sin_addr.s_addr),
	  ntohs((short)sin_port));
  return buffer;
}

			// Create an unbound stream socket
StreamSocket::StreamSocket(void)
	: socket_handle(socket(AF_INET,SOCK_STREAM,0))
{
  do_well( socket_handle );
  status = Good;
}

				// Wrap a class around a (supposedly open)
				// socket
StreamSocket::StreamSocket(const int _socket_handle)
	: socket_handle(_socket_handle)
{
  if( socket_handle < 0 )
    _error("Bad _socket_handle was passed to the class StreamSocket");
  status = Good;
}

				// Close the socket
StreamSocket::~StreamSocket(void)
{
  assert( socket_handle >= 0 );
  assert( close(socket_handle) == 0 );
  status = Failure;
}

				// Get the name of this socket
SocketAddr StreamSocket::get_my_name(void) const
{
  SocketAddr sock_addr;
  int namelen = sizeof(sock_addr);
  do_well( ::getsockname(socket_handle, (sockaddr *)sock_addr, &namelen) );
  assert( namelen == sizeof(sock_addr) );
  return sock_addr;
}

				// Get the name of the socket this socket is
				// connected to
SocketAddr StreamSocket::get_peer_name(void) const
{
  SocketAddr sock_addr;
  int namelen = sizeof(sock_addr);
  do_well( ::getpeername(socket_handle, (sockaddr *)sock_addr, &namelen) );
  assert( namelen == sizeof(sock_addr) );
  return sock_addr;
}

					// Print the socket status
void StreamSocket::print(const char * title) const
{
  message("%s socket %s ",title,(const char *)get_my_name());
  message("is connected to %s ",(const char *)get_peer_name());
  switch(status)
  {
    case Good:
         message("\n");
	 break;

    case EOF:
         message("(got EOF)\n");
	 break;

    case Failure:
         message("Failed\n");
	 break;

    default:
	 _error("socket is in a weird state %d",status);
  }
}

				// Set blocking/non-blocking I/O
				// If the blocking i/o is set, reading
				// on socket blocks the process until
				// the packet arrives. If i/o is non-blocking,
				// read() returns -1 with errno=EWOULDBLOCK
				// if there is nothing to read
void StreamSocket::set_blocking_io(const BOOL onoff)
{
  int arg = fcntl(socket_handle,F_GETFL,0);
  do_well( arg );
  do_well( fcntl(socket_handle,F_SETFL,onoff ? arg & ~FNDELAY :
		 arg | FNDELAY) );
}

				// Enable/disable SIGIO upon arriving of a
				// new packet
void StreamSocket::enable_sigio(const BOOL onoff)
{
  int arg = fcntl(socket_handle,F_GETFL,0);
  do_well( arg );
  do_well( fcntl(socket_handle,F_SETFL,onoff ? arg | FASYNC :
		 arg & ~FASYNC) );
  if( onoff )				// Tell which process should get SIGIO
    do_well( fcntl(socket_handle,F_SETOWN,-getpid()) );
//  message("after enable_sigio: flags 0x%x\n",fcntl(socket_handle,F_GETFL,0));
//  message("owner %d\n",fcntl(socket_handle,F_GETOWN,0));
}


					// Write to the socket, return the
					// status
StreamSocket::Status
StreamSocket::write(const void * buffer, const unsigned int len)
{
  assert( status == Good );
  assert( len != 0 );
  if( ::write(socket_handle,buffer,len) < 0 )
  {
    perror("Socket write failed");
    return status = Failure;
  }
  return Good;
}
					// Read from the socket returning the
					// size of the block read
StreamSocket::ReadResult
StreamSocket::read(void * buffer, const unsigned int len)
{
  assert( status == Good );
  assert( len != 0 );
  ReadResult result = { ::read(socket_handle,buffer,len), Good };
  if( result.len == 0 )
    return result.status = (status = Eof), result;
  if( result.len == (unsigned)(-1L) )
    if( errno == EWOULDBLOCK )
      return result.len = 0, result.status = Nodata, result;
    else
    {
      perror("reading from the socket");
      return result.len = 0, result.status = (status = Failure), result;
    }
  return result;
}
				// Create a bound socket
Listen::Listen(const SocketAddr bind_address)
{
  do_well( ::bind(socket_handle, (sockaddr *)bind_address,
		  sizeof(bind_address)) );
}

				// Listen for a connection and accept one
StreamSocket * Listen::accept(const int backlog)
{
  do_well( ::listen(socket_handle,backlog) );
  const int accepted_socket = ::accept(socket_handle,0,0);
  do_well( accepted_socket );
  return new StreamSocket(accepted_socket);
}

				// Initiate a TCP connection
ConnectingSocket::ConnectingSocket(const SocketAddr target_address)
{
  do_well( ::connect(socket_handle, (sockaddr *)target_address,
		  sizeof(target_address)) );
}

/*
 *------------------------------------------------------------------------
 *		    Here's where all relaying is going on...
 * Note we're forced to use static members in the PacketRelay class
 * because the relay function is called asynchronously, upon the SIGIO (that
 * is, when some packet arrives), so the relay function has to be static.
 */

class PacketRelay
{
  StreamSocket& sock1;
  StreamSocket& sock2;

  char sock1_name[80];			// Socket names for easy reference
  char sock2_name[80];			// (name cache)

  char * buffer;
  enum { IO_buffer_size=12000 };

  BOOL done;

  static PacketRelay * the_PacketRelay;

  void intr_enable(void);
  void intr_disable(void);

				// Finite State Machine "Actors"
  enum StatusCode {OK=0, NoData, Eof, Failed};	// what Actors return
  enum SOCK_ID { id_sock1, id_sock2 };
  typedef StatusCode (*Actor)(PacketRelay * _this, const SOCK_ID id);

					// Meaning write to another socket and
					// keep the log
  StatusCode relay_socket(const SOCK_ID id);
  BOOL done_loop;
  StatusCode is_loop_done(const SOCK_ID id)    { return done_loop ? Eof : OK; }
  StatusCode set_loop_done(const SOCK_ID id)   { done_loop = TRUE; return OK; }
  StatusCode reset_loop_done(const SOCK_ID id) { done_loop = FALSE; return OK;}
  StatusCode set_finished(const SOCK_ID id)    { done = TRUE; return OK;}

				// Relay Finite State Machine
  struct FSMAction { Actor   actor;
		     SOCK_ID socket_to_use;
		     short   next_state[Eof+1]; };
  static FSMAction FSMProgram [];

public:
  PacketRelay(StreamSocket& _sock1, StreamSocket& _sock2);
  ~PacketRelay(void);
  BOOL q_done(void) const	{ return done; }
  static int relay(void);		// That's what relays (if anything)
};

PacketRelay * PacketRelay::the_PacketRelay = 0;

				// Here is the FSM relay program
PacketRelay::FSMAction PacketRelay::FSMProgram [] =
{
 /* 0 */ {0,	id_sock1,	{0,0,0}},	// Dummy (stop) state

//  State	What to do     on which socket   OK ND EOF
 /* 1 */ { (Actor)&relay_socket,  id_sock1,	{2,  3,  5} },	// start state
 /* 2 */ { (Actor)&relay_socket,  id_sock2,	{1,  4,  6} },
 /* 3 */ { (Actor)&set_loop_done, id_sock1,	{2, -1, -1} },
 /* 4 */ { (Actor)&is_loop_done,  id_sock1,	{1,  0,  0} },
 /* 5 */ { (Actor)&relay_socket,  id_sock2,	{5,  7,  7} },
 /* 6 */ { (Actor)&relay_socket,  id_sock1,	{6,  7,  7} },
 /* 7 */ { (Actor)&set_finished,  id_sock1,	{0,  0,  0} }
};

				// Initialize relay and set up signal
				// handling
PacketRelay::PacketRelay(StreamSocket& _sock1, StreamSocket& _sock2)
	: sock1(_sock1), sock2(_sock2), done(FALSE), done_loop(FALSE)
{
  the_PacketRelay = this;		// For ref by static members
  buffer = new char[IO_buffer_size];
  assert( buffer != 0 );
  strncpy(sock1_name,sock1.get_peer_name(),sizeof(sock1_name)-2);
  strncpy(sock2_name,sock2.get_peer_name(),sizeof(sock2_name)-2);
  sock1.set_blocking_io(FALSE);
  sock2.set_blocking_io(FALSE);
  intr_enable();
  sock1.enable_sigio(TRUE);
  sock2.enable_sigio(TRUE);
  relay();				// Push to start
}

				// Do some minor clean-up
PacketRelay::~PacketRelay(void)
{
  sock1.enable_sigio(FALSE);		// do it first do disable all further
  sock2.enable_sigio(FALSE);		// signals
  intr_disable();			// now we can disengage our handler
  assert( buffer != 0 );
  delete buffer;
  buffer = 0;
  the_PacketRelay = 0;
}

void PacketRelay::intr_enable(void)
{
  do_well( (int)signal(SIGIO,relay) );
}

void PacketRelay::intr_disable(void)
{
  do_well( (int)signal(SIGIO,SIG_DFL) );
}

				// Read the specified socket and relay
				// data to the other socket
				// and keep the log
PacketRelay::StatusCode PacketRelay::relay_socket(const SOCK_ID id)
{
  StreamSocket::ReadResult read_result = 
    (id == id_sock1 ? sock1 : sock2).read(buffer,IO_buffer_size);

  if( read_result.status == StreamSocket::Nodata )
    return NoData;
  else if( read_result.status == StreamSocket::Eof )
  {
    message("\nEOF from '%s':\n",id == id_sock1 ? sock1_name : sock2_name);
    return Eof;
  }
  else if( read_result.status != StreamSocket::Good )
    return Failed;

  assert( read_result.len > 0 );
  message("\ndata from '%s':\n", id == id_sock1 ? sock1_name : sock2_name);
  for(register char * bp=buffer; bp < buffer+read_result.len; bp++)
    if( *bp != '\r' )
      message("%c",*bp);

  StreamSocket& socket = id == id_sock1 ? sock2 : sock1;
  if( socket.write(buffer,read_result.len) != StreamSocket::Good )
    return Failed;
  return reset_loop_done(id);
}

				// This is where it relays: the Finite
				// State Machine
				// Start with the state #1, do the action and
				// pick up the next state based on the action's
				// return code. Keep doing it until hit the
				// stop state (pc=0)
int PacketRelay::relay(void)
{
  PacketRelay& me = *the_PacketRelay;
  if( me.done )
    return 0;

  int pc;			// "Program" counter
  for(pc=1; pc != 0; )
  {
    assert( pc > 0 );
    const FSMAction& action = FSMProgram[pc];
    StatusCode result = action.actor(&me,action.socket_to_use);
    if( result == Failed )
      return me.done = TRUE, 0;
    pc = action.next_state[result];
  }
  return 0;
}

/*
 *------------------------------------------------------------------------
 *				Root module
 */

main(void)
{
  Listen listen_socket(SocketAddr(IPaddress(),Listen_port));
  message("\nPrepare to listen on port %d\n",Listen_port);

  for(;;)
  {
    StreamSocket& source_socket = *listen_socket.accept(1);
    source_socket.print("Got connection");
    ConnectingSocket target_socket(SocketAddr(Target_host,Target_port));
    target_socket.print("Target");

   {
     PacketRelay relay(source_socket,target_socket);

     while( !relay.q_done() )
       pause();
   }			// relay is to be destroyed here

    delete &source_socket;
  }
}

