/*
 *	dynamic.h
 *
 *	This file exports the classes and macros needed to implement
 *	dynamic message passing in C++.
 *
 *	(C) Copyright 1996, Pierre Arnaud, OPaC bright ideas
 *		CH-1437 Suscevaz, Pierre.Arnaud@di.epfl.ch
 *		http://diwww.epfl.ch/w3lami/team/arnaud/
 */

#ifndef _DYNAMIC_H_
#define	_DYNAMIC_H_

class Object;
class Message;

//	Define a boolean as an enum (not every compiler knows 'bool')

enum Bool
{
	FALSE,
	TRUE
};

//	Exceptions which may be thrown by the 'Message' class:

enum MessageExc
{
	MSG_EXC_TOO_MANY_PARAMS,	//	too many parameters requested
	MSG_EXC_BAD_PARAM_TYPE		//	parameter has bad type
};

//	Pointer to trampoline function:

typedef Bool (*TrampoFunc)(Message& message, Object* object);

/*
 *	Define the 'Message' class which will be used to store the
 *	arguments needed for the dynamic message passing.
 */

class Message
{
	enum Type					//	list of supported types
	{
		PT_INT,
		PT_FLOAT,
		PT_STR
	};
	
	union Value					//	union of supported parameters
	{
		int			_int;
		float		_float;
		const char*	_str;
	};
	
	struct List					//	description of a parameter
	{
		List*		next;		//	pointer to next one
		Value		value;		//	value of parameter
		Type		type;		//	type of parameter
		
		List (Value v, Type t) : next (0), value (v), type (t) { }
		~ List () { if (next) delete next; }
	};
	
private:
	const char*		method;		//	name of invoked method
	List*			head;		//	pointer to first parameter
	List*			tail;		//	pointer to last parameter
	List*			iter;		//	next parameter for GetValue
	
private:
	Message& SetValue (Value value, Type type);
	Message& GetValue (Value& value, Type type);

	void Clear ();

public:
	Message (const char* m = 0)
				    : method (m), head (0), tail (0), iter (0) { }
	
	~ Message () { this->Clear (); }
	
	void SetMethod (const char* name);
	Bool Send (Object* target);
	
	//	'-' is used in the 'object <- message << arg' construct:

	Message& operator - () { this->Clear (); return *this; }
	
	//	'<<' store the arguments for the invoked method.
	
	Message& operator << (int v)
	{ Value x; x._int   = v; return SetValue (x, PT_INT); }
	
	Message& operator << (float v)
	{ Value x; x._float = v; return SetValue (x, PT_FLOAT); }

	Message& operator << (const char* v)
	{ Value x; x._str   = v; return SetValue (x, PT_STR); }
	
	//	'GetValue' return the nth argument or throw an exception.

	Message& operator >> (int& v)
	{ Value x; GetValue (x, PT_INT);   v = x._int;   return *this; }

	Message& operator >> (float& v)
	{ Value x; GetValue (x, PT_FLOAT); v = x._float; return *this; }

	Message& operator >> (const char*& v)
	{ Value x; GetValue (x, PT_STR);   v = x._str;   return *this; }
};

/*
 *	Define the run-time information about an 'Object' class. It
 *	maintains the list of known methods.
 */

class Runtime
{
	struct List					//	description of method
	{
		List*		next;		//	pointer to next one
		const char*	name;		//	name of that method
		TrampoFunc	func;		//	pointer to trampoline function
		
		List (const char* n, TrampoFunc f)
								: next (0), name (n), func (f) { }
	};

private:
	const char*		name;		//	name of represented class
	List*			head;		//	pointer to first method
	List*			tail;		//	pointer to last method
	
public:
	Runtime (const char* n) : name (n), head (0), tail (0) { }
	
	void AddFunction (const char* name, TrampoFunc func);
	
	TrampoFunc GetFunction (const char* name) const;
	const char* GetClassName () const { return this->name; }
	const char* GetNthFunctionName () const;
};

/*
 *	Special automatic setup class which calls a specified function
 *	at its construction. This is used by the 'PUBLISH_xxx' macros.
 */

class AutoRuntimeSetup
{
public:
	AutoRuntimeSetup (const char* m, Runtime& run, TrampoFunc func)
	{ run.AddFunction (m, func); }
};

/*
 *	Base class used to derive every other class which supports
 *	dynamic message passing.
 */

class Object
{
public:
	Object ();
	virtual ~ Object ();

	virtual Runtime* GetRuntime () const;	//	always override!
	
	//	'<' is used in the 'object <- message << arg' construct:

	Bool operator < (Message& m) { return m.Send (this); }
};

/*
 *	Use 'PUBLISH_CLASS' exactly once for each class derived from
 *	'Object'. It will provide the 'GetRuntime' method.
 */

#define	PUBLISH_CLASS(Class)									  \
																  \
Runtime runtime_##Class (#Class);								  \
																  \
Runtime*														  \
Class::GetRuntime () const										  \
{																  \
	return & runtime_##Class;									  \
}																  \
																  \
class Object /* swallow the semicolon */

/*
 *	Use 'PUBLISH_METHOD' to declare dynamically callable methods
 *	of a class. It should be followed by some PUBLISH_ARG macros
 *	and exactly one PUBLISH_CALL macro.
 */

#define	PUBLISH_METHOD(Class,method)							  \
																  \
Bool trampo_##Class##method (Message&, Object*);				  \
																  \
AutoRuntimeSetup auto_##Class##method (#method, runtime_##Class,  \
									   trampo_##Class##method);	  \
																  \
Bool															  \
trampo_##Class##method (Message& message, Object* object)		  \
{																  \
	Class* instance = (Class*)(object)

/*
 *	Use 'PUBLISH_ARG' to declare (named) arguments for a method.
 */

#define	PUBLISH_ARG(arg,type)									  \
	type arg; message >> arg

/*
 *	Use 'PUBLISH_CALL' at the end of a method publication; this
 *	tells the compiler how to call the method using the specified
 *	arguments.
 */

#define	PUBLISH_CALL(call)										  \
	return (instance) ? instance->call : FALSE;					  \
}																  \
																  \
class Object /* swallow the semicolon */

#endif
