//======================================================================
//
// DBLSCAN.C - main module for VxD DBLSCAN
//
// Copyright (C) 1995, Mark Russinovich and Bryce Cogswell
//
// You have the right to use and publish this code provided you
// give credit to the authors.
// 
//======================================================================

#define   DEVICE_MAIN
#include  "dblscan.h"
#undef    DEVICE_MAIN
#include  <blockdev.h>

Declare_Virtual_Device(DBLSCAN)

// our protected mode interface identifying string
CHAR VendorString[] = DBLSCAN_ID_STRING;

// pager types for windows program
#define UPAGERDOS         0
#define UPAGERBIOS        1
#define UPAGERBLCK        2
#define UPAGERIFS         3

WORD       UserPagerType = -1;

// protected mode command definitions
#define VDBLSCND_INIT       0
#define VDBLSCND_SCAN       1

// size of memory
WORD       memSize;

// are we running an experiment
BYTE       RunningExperiment = FALSE;
BYTE       RanOnce = FALSE;


// pager type
BYTE       PagerType;

// if DOS paging, keep track of paging buffer
PBYTE      DOSBuffer = NULL;

// if BLOCKDEV paging, keep track of callback (as an identifier)
DWORD      PagerCallback = 0;

// 
// this is the definition for the stats data structure shared with 
// the user interface. This declaration is identical to the one in
// that module
//
typedef struct {
  DWORD            DiskWrites;
  DWORD            PagesInRAM;
  DWORD            PagesOnDisk;
} vstats;

// the stats structure pointer
vstats             *vdblscnd_stats;

// these variables keep track of the number of hits we have on the
// string going to disk and in memory
DWORD        DiskHits;
DWORD        MemHits;

// this is a pointer to the string passed to use by the protected
// mode app. On pages that are uncompressed, this string will fill
// the page.
PBYTE              vdblscnd_string;
WORD               vdblscnd_strlen;

// for hooking int21 pager activity
WORD               int21Segment;
DWORD              int21Offset;

// pointer to the original pagefile_readwrite service that we hook
DWORD              (*BlockdevCompleteService)();
DWORD              (*IFSMgrFileIO)();

// thunks
DeviceService_THUNK   thunkBlockdevCompleteHook;
DeviceService_THUNK   thunkIFSMgrFileHook;
V86Int_THUNK          thunkInt13;
V86Call_THUNK         thunkInt21;


//------------------------------------------------------------
//
// getMemSize
//
// Uses CMOS to get the size of physical memory in KB. Rounds
// up to the next 1MB for safety.
//
//------------------------------------------------------------
WORD getMemSize()
{
  WORD       low, high, size;
  BYTE       val;
  PDWORD     *cr3reg;

  // get the page table representing conventional memory
  _asm {
    mov    eax, cr3
    mov    cr3reg, eax
  }
  cr3reg = _MapPhysToLinear(cr3reg, 0x1000, 0);

  // determine size of physical memory from CMOS and grab half
  _disable();
  val = 0x80 | 0x30;
  _outp(0x70, val);
  low = _inp(0x71);
  val = 0x80 | 0x31;
  _outp(0x70, val);
  high = _inp(0x71);
  _outp(0x70, 0);
  _enable();

  // calc memsize and round up to 1 meg boundary  
  size = ((high << 8 + low)+0x1000)&0xF000;
  
  // convert to pages
  size >>= 2;
  return size;
}


//----------------------------------------------------------------------
//
// ChainService
//
// Passes a call through to the original service, setting and restoring
// pDS registers in the process.
//
//----------------------------------------------------------------------
VOID ChainService( PDSFRAME pDS, DWORD (*Service)() )
{
  DWORD  REAX, REBX, RECX, REDX, RESI, REDI;

  REAX = pDS->REAX;
  REBX = pDS->REBX;
  RECX = pDS->RECX;
  REDX = pDS->REDX;
  RESI = pDS->RESI;
  REDI = pDS->REDI;

  _asm {
    pusha
    mov   eax, REAX
    mov   ebx, REBX
    mov   ecx, RECX
    mov   edx, REDX
    mov   esi, RESI
    mov   edi, REDI
    call  Service
    mov   REAX, eax
    mov   REBX, ebx
    mov   RECX, ecx
    mov   REDX, edx
    mov   RESI, esi
    mov   REDI, edi
    popa
  }

  pDS->REAX = REAX;
  pDS->REBX = REBX;
  pDS->RECX = RECX;
  pDS->REDX = REDX;
  pDS->RESI = RESI;
  pDS->REDI = REDI;
}


//----------------------------------------------------------------------
//
// VdblscndClear
//
// After we have found our identifying string in a buffer ready to
// go to disk, we clear it so that we won't see it again for some
// reason (like if its in a buffer in memory that isn't cleared before
// we do the RAM scan). We know that global allocs are paragraph aligned
// and our string is 8 bytes so we can scan for it.
//
//----------------------------------------------------------------------
VOID VdblscndClear( PBYTE pagedata )
{
  int    i = 0; 

  while( i < 0x1000 ) {
    if( !strncmp( &pagedata[i], vdblscnd_string, vdblscnd_strlen )) 
      memset( &pagedata[i], 0, vdblscnd_strlen );
    i += vdblscnd_strlen;
  }
}


//----------------------------------------------------------------------
//
// MemScan
//
// Returns whether or not the passed page contains only the identifying
// string repeated.
//
//----------------------------------------------------------------------
WORD MemScan( PBYTE pagedata )
{
  int    i = 0, hit = 0;

  while( i < 0x1000 ) {
    if( !strncmp( &pagedata[i], vdblscnd_string, vdblscnd_strlen ) ) 
      hit++;
    i += vdblscnd_strlen;
  }
  
  // if more than half the page has hits, we assume its part
  // of one of the application's uncompresses pages (the other
  // part won't be counted since it will be less than half a page).
  return hit;
}


//----------------------------------------------------------------------
//
// int13Hook
//
// If we are running in Window 3.1 and we are told that DOS paging is
// in effect, all paging requests will eventually end up passing 
// through the BIOS. So we hook int13 and look at reads or writes.
//
//----------------------------------------------------------------------
int __stdcall Int13Hook( DWORD Interrupt, 
			 VMHANDLE hVM, PCLIENT_STRUCT pRegs )
{
  PBYTE   buffer;
  DWORD   i, numpages;

  if( !RunningExperiment || UserPagerType == UPAGERDOS ) return FALSE;

  if( pRegs->CBRS.Client_AH == BIOSWRITE )
    vdblscnd_stats->DiskWrites++;

  // if its a write, look for and zap our string
  if( (pRegs->CBRS.Client_AH == BIOSWRITE) &&
     pRegs->CBRS.Client_AL == 8 ) {

    // it loook like a page file access - if it is a write, check
    // it right away because it is a BIOS call that may eventually
    // get cached and then get sent through DOS
    buffer = (PBYTE) Map_Flat( CLIENT_ES, CLIENT_BX );

    // scan through sectors looking for signature
    numpages = pRegs->CBRS.Client_AL/8;
    for( i=0; i < numpages; i++ ) {
      if( DiskHits += MemScan( buffer ) ) {

	// BIOS is being used
	UserPagerType = UPAGERBIOS;

	VdblscndClear( buffer );
      }
      buffer = (PBYTE) ((DWORD)buffer + 0x1000);
    }
  }
  return FALSE;
}


//----------------------------------------------------------------------
//
// int21Hook
//
// If we are using windows 3.1, we might be using DOS for paging or
// the BIOS (the pagefile won't tell us which it is). So we have to
// initially hook both and wait and see if DOS is being used (if
// we see are signature being used, we decide that DOS is the ticket).
//
//----------------------------------------------------------------------
VOID __stdcall Int21Hook( VMHANDLE hVM, PVOID RefData, PCLIENT_STRUCT pRegs )
{
  PBYTE   buffer;
  DWORD   i, numpages;

  // first thing, revector to original entry point
  pRegs->CRS.Client_CS = int21Segment;
  pRegs->CRS.Client_EIP = (DWORD) int21Offset;

  if( !RunningExperiment || UserPagerType == UPAGERBIOS ) return;

  if( pRegs->CBRS.Client_AH == DOSWRITE ) {
    vdblscnd_stats->DiskWrites++;

    // get the buffer address
    buffer = (PBYTE) Map_Flat( CLIENT_DS, CLIENT_DX );

    // if DOS is being used for pagin, all requests are
    // one page in length
    numpages = pRegs->CWRS.Client_CX/0x1000;
    for( i=0; i < numpages; i++ ) {
      if( DiskHits += MemScan( buffer ) ) {
	// DOS is being used for sure!
	UserPagerType = UPAGERDOS;

	// make sure its a real paging request
	if( !DOSBuffer ) DOSBuffer = buffer;
	else if( DOSBuffer != buffer ) break;

	VdblscndClear( DOSBuffer );
      }
      buffer = (PBYTE) ((DWORD)buffer + 0x1000);
    }
  }
  return;
}


//----------------------------------------------------------------------
//
// BlockdevCompleteHook
//
// After any file I/O has completed, we look to see if it could
// have been a pager initiated command. They have the characteristic
// that each request is 8 sectors (4K) and is high-priority.
//
//----------------------------------------------------------------------
VOID __stdcall BlockdevCompleteHook( PDSFRAME pDS )
{
  PBCB   cmdblock;

  if( !RunningExperiment) {
    // pass the service on through
    ChainService( pDS, BlockdevCompleteService );
    return;
  }

  // get pointer to blockdev command
  cmdblock = (PBCB) pDS->RESI;

  // count disk writes
  if( cmdblock->BD_CB_Command == BDC_Write ) {
    vdblscnd_stats->DiskWrites++;

    // if command blocks are 8 sectors  (1 page) long, and
    // successful commands (i.e. not cancelled) and are page 
    // aligned, scan for our id string
    if( cmdblock && ( cmdblock->BD_CB_Count == 8) &&
       !(cmdblock->BD_CB_Buffer_Ptr & 0xFFF) &&
       (cmdblock->BD_CB_Flags & BDCF_HIGH_PRIORITY) &&
       (cmdblock->BD_CB_Cmd_Status < BDS_FIRST_ERROR_CODE ) &&
       !( PagerCallback && 
	 (cmdblock->BD_CB_Cmd_Cplt_Proc != PagerCallback ))) {

      if( DiskHits += MemScan( (PBYTE) cmdblock->BD_CB_Buffer_Ptr ) ) {

	// got a live one so remember what the paging callback is
        if( !PagerCallback )
	  PagerCallback = cmdblock->BD_CB_Cmd_Cplt_Proc;

	// now, adjust the counts 
	VdblscndClear( (PBYTE) cmdblock->BD_CB_Buffer_Ptr );
      }
    }
  }

  // pass the service on through
  ChainService( pDS, BlockdevCompleteService );
}


//----------------------------------------------------------------------
//
// IFSMgrFileHook
//
// Windows 95 uses IFS Manager services to do all file I/O. So we
// monitor all read and write operations, looking for commands
// that are filled with our signature. We then keep a cummulative
// record of total pages written - total pages read to represent the
// number of our pages that are out on disk in the paging file.
//
//----------------------------------------------------------------------
VOID __stdcall IFSMgrFileHook( PDSFRAME pDS )
{
  DWORD    cmd, len;
  PBYTE    buffer;

  // extract parameters before they're trashed by the service itself
  cmd = pDS->REAX;
  len = pDS->RECX;
  buffer = (PBYTE) pDS->RESI;

  // count disk writes if running experiment
  if( cmd == R0_WRITEFILE && RunningExperiment) {
    vdblscnd_stats->DiskWrites++;

    // is it a page aligned request (necessary for paging) and a multiple
    // of pagesize in len?
    if( !((DWORD) buffer & 0xFFF) && !(len%0x1000)) {
       
      // yep, lets start scanning
      while( len ) {

	// do we have a hit?
        if( DiskHits += MemScan( buffer ) ) {
	  VdblscndClear( buffer );
	}
	buffer = (PBYTE) ((DWORD) buffer+0x1000);
	len -= 0x1000;
      }
    }
  }

  // now, chain this and then look at the result afterwards
  ChainService( pDS, IFSMgrFileIO );
}


//----------------------------------------------------------------------
//
// VdblscndScan
//
// Scans memory for the identifier string passed at initialization. 
// For every page in memory that we find filled with the string, we
// increment the PagesinRAM count.
//
//----------------------------------------------------------------------
VOID VdblscndScan( PCLIENT_STRUCT pRegs )
{
  DWORD    page;
  PBYTE    buffer;

  // start scanning
  _asm cli;
  RunningExperiment = FALSE;
  for( page=0 ; page < memSize; page++ ) {
    buffer = (PBYTE) _MapPhysToLinear( (PVOID) (page << 0xC),
				      0x1000, 0 );
    MemHits += MemScan( buffer );
  }
  _asm sti;
  pRegs->CWRS.Client_AX = UserPagerType;

  // now we can tally up the score
  vdblscnd_stats->PagesInRAM  = MemHits/(0x1000/vdblscnd_strlen);
  vdblscnd_stats->PagesOnDisk = DiskHits/(0x1000/vdblscnd_strlen);
  
  // see if a page happens to be split between disk and memory,
  // and if so, count it as being in memory
  if( vdblscnd_stats->PagesInRAM + vdblscnd_stats->PagesOnDisk !=
     (MemHits + DiskHits)/(0x1000/vdblscnd_strlen))
    vdblscnd_stats->PagesInRAM++;
}


//----------------------------------------------------------------------
//
// VdblscndInit
// 
// Translate pointers to the statistics data structure and the 
// identifying string that are passed to us by the user interface.
// Then hook the pagefile_readwrite service.
//
//----------------------------------------------------------------------
VOID VdblscndInit( PCLIENT_STRUCT pRegs )
{
  DWORD           pagerversion;

  // we receive the address of the PM application's data structure
  vdblscnd_string = (PBYTE) Map_Flat( CLIENT_DS, CLIENT_BX );
  vdblscnd_strlen = strlen( vdblscnd_string );
  vdblscnd_stats = (vstats *) Map_Flat( CLIENT_DS, CLIENT_AX );

  // init the structure
  vdblscnd_stats->DiskWrites = 0;
  vdblscnd_stats->PagesInRAM = 0;
  vdblscnd_stats->PagesOnDisk = 0;  

  // initialize counters
  DiskHits = 0;
  MemHits  = 0;

  // return the number of pages in memory
  memSize = getMemSize();
  pRegs->CWRS.Client_AX = memSize;

  // starting experiment
  RunningExperiment = TRUE;

  // if we already hooked things because we ran once, don't do it again
  if( RanOnce ) return;
  RanOnce = TRUE;

  // see what kind of paging is going on
  VxDCall( PageFile_Get_Version );
  _asm mov     PagerType, bl;
  _asm mov     pagerversion, eax;

  // hook the appropriate disk service
  switch( PagerType ) {

  case PAGERNONE:
    // do nothing - no pager
    break;

  case PAGERDOS:
    // hook int13 and int21 to look for writes of our signature
    Hook_V86_Int_Chain( 0x13, Int13Hook, &thunkInt13 );
    break;

  case PAGERFAST:

    // under windows 95 this is ifsmanager - under window 3.1 this
    // is the blockdev device
    // NOTE: softram returns a version number of 0xC03 for Windows 3.1
    if( pagerversion >= 0x400 && pagerversion < 0x500 ) {

      // windows 95: hook ifs manager
      UserPagerType = UPAGERIFS;
      IFSMgrFileIO = Hook_Device_Service( (IFSMGR_DEVICE_ID << 16) | 
		     GetVxDServiceOrdinal(IFSMgr_Ring0_FileIO),
		     IFSMgrFileHook,
		     &thunkIFSMgrFileHook );
    } else {
      
      // windows 3.X: hook blockdev service
      UserPagerType = UPAGERBLCK;
      BlockdevCompleteService = Hook_Device_Service( 
                      (BLOCKDEV_DEVICE_ID << 16) | 
		      GetVxDServiceOrdinal(_BlockDev_Command_Complete), 
		      BlockdevCompleteHook,
		      &thunkBlockdevCompleteHook );
    }
    break;

  }
}


//----------------------------------------------------------------------
//
// PM_Vendor_Api_Handler
//
// This routine handles commands from the Dblscan user interface.
//
//----------------------------------------------------------------------
VOID __stdcall PM_Vendor_Api_Handler(VMHANDLE hVM, PVOID pRefdata, 
				     PCLIENT_STRUCT pRegs)
{
  // command is in the DX register
  switch(pRegs->CWRS.Client_DX) {

  case VDBLSCND_INIT:
    VdblscndInit( pRegs );
    break;

  case VDBLSCND_SCAN:
    // after we scan, win program can look at the results
    VdblscndScan( pRegs );
    break;
  }

  Simulate_Far_Ret();
}


//----------------------------------------------------------------------
//
// ControlDispatcher
//
// VxD entry point. Only used for device initialization.
//
//----------------------------------------------------------------------
DefineControlHandler(SYS_CRITICAL_INIT, OnSysCriticalInit);
DefineControlHandler(SYS_CRITICAL_EXIT, OnSysCriticalExit);
DefineControlHandler(BEGIN_PM_APP, OnBeginPmApp);
DefineControlHandler(END_PM_APP, OnEndPmApp);
BOOL ControlDispatcher(
	DWORD dwControlMessage,
	DWORD EBX,
	DWORD EDX,
	DWORD ESI,
	DWORD EDI)
{
  START_CONTROL_DISPATCH

    ON_SYS_CRITICAL_INIT(OnSysCriticalInit);
    ON_SYS_CRITICAL_EXIT(OnSysCriticalExit);
    ON_BEGIN_PM_APP(OnBeginPmApp);
    ON_END_PM_APP(OnEndPmApp);

  END_CONTROL_DISPATCH

  return TRUE;
}


//----------------------------------------------------------------------
// 
// OnBeginPmApp
//
// Register the entry point for the app.
//
//----------------------------------------------------------------------
BOOL OnBeginPmApp(VMHANDLE hVM, DWORD flags, PPMCB pmcb)
{
	Set_PM_Vendor_Entry(VendorString, PM_Vendor_Api_Handler);
	return TRUE;
}


//----------------------------------------------------------------------
// 
// OnEndPmApp
//
// Unregister the entry point for the app.
//
//----------------------------------------------------------------------
VOID OnEndPmApp(VMHANDLE hVM, PPMCB pmcb)
{
	End_PM_Vendor_Entry();
}


//----------------------------------------------------------------------
//
// OnSysCriticalInit
//
// We have too hook the DOS chain before the pagefile initializes
// and reads the DOS entry point as the vector at that time.
//
//----------------------------------------------------------------------
BOOL OnSysCriticalInit( VMHANDLE hVM, PCHAR CommandTail, DWORD refData )
{
  DWORD     int21callback;

  // we have too hook DOS now, because the pager will get the
  // vector when it initializes and if we hook later we won't
  // see the calls
  Get_V86_Int_Vector( 0x21, &int21Segment, &int21Offset );
  int21callback =  Allocate_V86_Call_Back( NULL, Int21Hook,
                             &thunkInt21 );
  Set_V86_Int_Vector( 0x21, (WORD) (int21callback >> 16), 
             int21callback & 0xFFFF );
  return TRUE;
}


//----------------------------------------------------------------------
//
// OnSysCriticalExit
//
// Restore the int21 interrupt handler.
//
//----------------------------------------------------------------------
VOID OnSysCriticalExit()
{
   Set_V86_Int_Vector( 0x21, int21Segment, int21Offset );
}


