#include "elliptic.hpp"
#include <math.h>

#ifndef FALSE
#define FALSE 0
#endif
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// ALGEBRAIC DERIVATIVE FUNCTIONS:
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// Function for elliptic integrals of the 1st kind, over algebraic interval.
long double FAR CDECL ldEifk( long double ldT, const long double *pldM )
{
   const long double ldTsq = ldT*ldT;
   const long double ldOne = 1.0e0L;
   return ldOne / sqrtl( ( ldOne - ldTsq ) * ( ldOne - (*pldM) * ldTsq ) );
}

// Function for elliptic integrals of the 2nd kind, over algebraic interval.
long double FAR CDECL ldEisk( long double ldT, const long double *pldM )
{
   const long double ldTsq = ldT*ldT;
   const long double ldOne = 1.0e0L;
   return sqrtl( ( ldOne - (*pldM) * ldTsq ) / ( ldOne - ldTsq ) );
}

// Function for elliptic integrals of the 3rd kind, over algebraic interval.
long double FAR CDECL ldEitk( long double ldT, const long double *pldData )
{
   const long double ldTsq = ldT*ldT;
   const long double ldOne = 1.0e0L;
   long double ldM = pldData[M_POSITION];
   long double ldN = pldData[N_POSITION];
   long double ldTerm = ( ldOne - ldTsq ) * ( ldOne - ldM * ldTsq );
   if (ldTerm > 0)
      ldTerm = sqrtl( ldTerm);
   return ldOne / ( ( ldOne - ldN*ldTsq ) * ldTerm );
}

//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// Trigonometric DERIVATIVE FUNCTIONS:
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// Function for elliptic integrals of the 1st kind, over Trigonometric interval.
long double FAR CDECL ldEifkTrig( long double ldT, const long double *pldM )
{
   const long double ldOne = 1.0e0L;
   long double ldSinSquaredT = sinl( ldT );
   ldSinSquaredT *= ldSinSquaredT;
   return ldOne / sqrtl( ( ldOne - (*pldM) * ldSinSquaredT ) );
}

// Function for elliptic integrals of the 2nd kind, over Trigonometric interval.
long double FAR CDECL ldEiskTrig( long double ldT, const long double *pldM )
{
   const long double ldOne = 1.0e0L;
   long double ldSinSquaredT = sinl( ldT );
   ldSinSquaredT *= ldSinSquaredT;
   return sqrtl( ( ldOne - (*pldM) * ldSinSquaredT ) );
}

// Function for elliptic integrals of the 3rd kind, over Trigonometric interval.
long double FAR CDECL ldEitkTrig( long double ldT, const long double *pldData )
{
   const long double ldOne = 1.0e0L;
   long double ldSinSquaredT = sinl( ldT );
   ldSinSquaredT *= ldSinSquaredT;
   long double ldM = pldData[M_POSITION];
   long double ldN = pldData[N_POSITION];
   long double ldTerm = ldOne - ldM * ldSinSquaredT;
   if (ldTerm > 0)
      ldTerm = sqrtl( ldTerm );
   return ldOne/( ( ldOne - ldN*ldSinSquaredT ) * ldTerm );
}

//////////////////////////////////////////////////////////////////////////
// ELLIPTIC INTEGRALS Class
//////////////////////////////////////////////////////////////////////////
// Constructor:
EllipticIntegral::EllipticIntegral()
{
   const long double ldZero = 0.0e0L;
   ldK = ldZero;         // Constant for the integration
   ldM = ldZero;         // Constant for the integration
   ldPhi = ldZero;       // Constant for the integration
   fOkToProcess = 0;     // bitflag set indicates true if all test bits are on.
   vSetX0( ldZero );     // Always true for elliptic integrals
   vSetX1( ldZero );     // Zero length interval until supplied
   // Set flag for Trigonometric interval to FALSE
   // (Algebraic integration is faster and more accurate)
   bSetTrigonometric( FALSE );
}

// Destructor
EllipticIntegral::~EllipticIntegral()
{
}

//////////////////////////////////////////////////////////////////////////
// Inquiry Methods
//////////////////////////////////////////////////////////////////////////
char EllipticIntegral::bOkToProcess() // Return True if OK to do integration
{
return ( char ) ( fOkToProcess & M_SUPPLIED &&
                  fOkToProcess & K_SUPPLIED &&
                  fOkToProcess & X_SUPPLIED );
}
char EllipticIntegral::bTrigonometric() // Return True if Trigonometric interval
{
   return ( fTrigonometric );
}

long double EllipticIntegral::ldGetK()  // Get the constant K for integration
{
   return ldK;
}

long double EllipticIntegral::ldGetM()  // Get the constant M for integration
{
   return ldM;
}


long double EllipticIntegral::ldGetPhi()  // Get the constant N for integration
{
   return ldPhi;
}

long double EllipticIntegral::ldGetEllipticResult( void ) // Get the result of integration
{
   return ldGetIntegralResult();
}
//////////////////////////////////////////////////////////////////////////
// Modifier Methods
//////////////////////////////////////////////////////////////////////////

char EllipticIntegral::bSetTrigonometric( char fFlag ) // Set Trigonometric interval
{
   fTrigonometric = fFlag;
   ldSetPhi( ldPhi );
   ldSetK( ldK );
   return ( fTrigonometric );
}
long double EllipticIntegral::ldSetK( long double ldKIn )
{

   if ( ( ldKIn > 0.0e0L? ldKIn : -ldKIn )  >= 1.0e0L)
   {
      // raise domain error
   }
   else
   {
      ldM = ldKIn*ldKIn;
      ldK = ldKIn;
      fOkToProcess |= K_SUPPLIED;
      fOkToProcess |= M_SUPPLIED;
   }
   return ldK;
}

long double EllipticIntegral::ldSetM( long double ldMIn )
{
   if (ldMIn < 0.0e0L || ldMIn >= 1.0e0L)
   {
      // raise domain error
   }
   else
   {
      ldM = ldMIn;
      ldK = sqrtl( ldM );
      fOkToProcess |= K_SUPPLIED;
      fOkToProcess |= M_SUPPLIED;
   }
   return ldM;
}

long double EllipticIntegral::ldSetPhi( long double ldPhiIn )
{
   if (ldPhiIn > PIOVER2)
      ldPhiIn = fmodl( ldPhiIn, PIOVER2 );
   ldPhi = ldPhiIn;
   if (bTrigonometric())
      vSetX1( ldPhiIn );
   else
      vSetX1( sinl( ldPhiIn ) );
   fOkToProcess |= X_SUPPLIED;
   return ldPhi;
}

long double EllipticIntegral::ldEvaluate()
{
   long double ldResult = 0.0e0L;
   if (bOkToProcess())
   {
      Evaluate();
      ldResult = ldGetIntegralResult();
   }
   else
   {
      // raise error...
   }
   return ldResult;
}

EllipticIntegralOfThe1StKind::EllipticIntegralOfThe1StKind(long double ldK, long double ldPhi)
{
   ldSetK( ldK );
   ldSetPhi( ldPhi );
   pldConstArr[0] = ldGetM();
   vSetConstArr( pldConstArr );
   Algebraic();
   ldEvaluate();
}

EllipticIntegralOfThe1StKind::~EllipticIntegralOfThe1StKind()
{
}



EllipticIntegralOfThe1StKind::Trigonometric()
{
   vSetFprime( ::ldEifkTrig );
   return bSetTrigonometric( TRUE );
}

EllipticIntegralOfThe1StKind::Algebraic()
{
   vSetFprime( ::ldEifk );
   return bSetTrigonometric( FALSE );
}


EllipticIntegralOfThe2ndKind::EllipticIntegralOfThe2ndKind(long double ldK, long double ldPhi)
{
   ldSetK( ldK );
   ldSetPhi( ldPhi );
   pldConstArr[0] = ldGetM();
   vSetConstArr( pldConstArr );
   Algebraic();
   ldEvaluate();
}


EllipticIntegralOfThe2ndKind::~EllipticIntegralOfThe2ndKind()
{
}

EllipticIntegralOfThe2ndKind::Trigonometric()
{
   vSetFprime( ::ldEiskTrig );
   return bSetTrigonometric( TRUE );
}

EllipticIntegralOfThe2ndKind::Algebraic()
{
   vSetFprime( ::ldEisk );
   return bSetTrigonometric( FALSE );
}

EllipticIntegralOfThe3rdKind::EllipticIntegralOfThe3rdKind(long double ldK, long double ldPhi, long double ldN)
{
   ldSetN( ldN );
   bNwasSupplied = TRUE;
   ldSetK( ldK );
   ldSetPhi( ldPhi );
   pldConstArr[M_POSITION] = ldGetM();
   pldConstArr[N_POSITION] = ldGetN();
   vSetConstArr( pldConstArr );
   Algebraic();
   ldEvaluate();
}

EllipticIntegralOfThe3rdKind::~EllipticIntegralOfThe3rdKind()
{
}

EllipticIntegralOfThe3rdKind::Trigonometric()
{
   vSetFprime( ::ldEitkTrig );
   return bSetTrigonometric( TRUE );
}


EllipticIntegralOfThe3rdKind::Algebraic()
{
   vSetFprime( ::ldEitk );
   return bSetTrigonometric( FALSE );
}

long double EllipticIntegralOfThe3rdKind::ldGetN() // Get the constant N for integration
{
   return ldN;
}

long double EllipticIntegralOfThe3rdKind::ldSetN( long double ldNIn )
{
   bNwasSupplied = TRUE;
   ldN = ldNIn;
   return ldN;
}
