#include "integral.hpp"
#include <float.h>

// internal prototype:
static long double ldGaussian(
_ldIntegralFunction_t ldF,
long double ldA,
long double ldB,
long double *pldConstArray
);

///////////////////////////////////////////////////////////////////////////
// Function ldIntegrate() performs numerical integration over a supplied
// interval (input parameters ldX0 to ldX1) for the supplied function.
//
// This formula used and the associated constants are from:
//
// the "Encyclopedic Dictionary of Mathematics, Volume 1"
// "1 Abel to 304 Numerical Solution of Partial Differential Equations"
// By  Kiyosi Ito.
//
// Imprint Cambridge, MA : MIT Press, 1993.
//
// ISBN    0262590204 ( set : PB )
//         0262090260 ( set : HB )
//
// LCCN    93139017
// Article 299 ( XV.7 ) Numerical Integration, pp 1120-1.
//
///////////////////////////////////////////////////////////////////////////
// Constructor:
//////////////////////////////////////////////////////////////////////////
EXPORT Integral::Integral()
{
   // Set up the integration interval values;
   ldX0 = 0.0e0L;
   ldX1 = 0.0e0L;
   ldFprime = ( _ldIntegralFunction_t ) NULL;
   pldConstArray = ( long double * ) NULL;
}


EXPORT Integral::Integral(
long double ldStart,
long double ldEnd,
_ldIntegralFunction_t ldF
)
{
   // Set up the integration interval values;
   ldX0 = ldStart;
   ldX1 = ldEnd;
   ldFprime = ldF;
   pldConstArray = ( long double * ) NULL;
   Evaluate();
}
EXPORT Integral::Integral(
long double ldStart,
long double ldEnd,
_ldIntegralFunction_t ldF,
long double *pldConstArrayIn
)
{
   // Set up the integration interval values;
   ldX0 = ldStart;
   ldX1 = ldEnd;
   ldFprime = ldF;
   pldConstArray = pldConstArrayIn;
   Evaluate();
}
void EXPORT Integral::Evaluate()
{
   // We use this constant to divide the interval into two equal parts:
   const long double ldHalf = 0.5e0L;

   // Empirically derived for long double stepsize:
   // The number .21 is due to the 22 weights in our Gaussian formula
   // creating 21 subdivisions.  This will give a stepsize of .01,
   // when .21 is divided 21 times.
   // Theoretically, the error should be less than that of Newton-Cotes.
   const long double ldSmallEnoughInterval = 0.21e0L;

   // If interval is too large, divide the integral into two equal sections.
   if ( ( ldX1 - ldX0 ) > ldSmallEnoughInterval )
   {
        long double ldMidpoint = ldX0 + ( ldX1 - ldX0 )*ldHalf;
        Integral i0( ldX0, ldMidpoint, ldFprime, pldConstArray );
        Integral I1( ldMidpoint, ldX1, ldFprime, pldConstArray );
        ldIntegralResult = i0.ldGetIntegralResult() + I1.ldGetIntegralResult();
   }
   else
   {
      ldIntegralResult = ldGaussian( ldFprime, ldX0, ldX1, pldConstArray );
   }
}

///////////////////////////////////////////////////////////////////////////
// Destructor:
//////////////////////////////////////////////////////////////////////////
EXPORT Integral::~Integral()
{
}

//////////////////////////////////////////////////////////////////////////
// Inquiry Methods:
//////////////////////////////////////////////////////////////////////////
void EXPORT Integral::vDisplayResultsToConsole()    // Display the results to the console
{
}
long double EXPORT Integral::ldGetIntegralResult(void)
{
   return ldIntegralResult;
}
long double EXPORT Integral::ldGetX0() // Get the start of the interval of integration
{
   return ldX0;
}
long double EXPORT Integral::ldGetX1() // Get the end of the interval of integration
{
   return ldX1;
}

//////////////////////////////////////////////////////////////////////////
// Modifier Methods:
//////////////////////////////////////////////////////////////////////////
// Set the Derivative Function
void EXPORT Integral::vSetFprime( _ldIntegralFunction_t ldF )
{
   ldFprime = ldF;
}

void EXPORT Integral::vSetX0( long double ldX ) // Set the start of the interval
{
   ldX0 = ldX;
}
void EXPORT Integral::vSetX1( long double ldX ) // Set the end of the interval
{
   ldX1 = ldX;
}
void EXPORT Integral::vSetConstArr( long double *pldConst ) // Set the end of the interval
{
   pldConstArray = pldConst;
}

//////////////////////////////////////////////////////////////////////////
// Internal Method, Gaussian rule for integral evaluation:
//////////////////////////////////////////////////////////////////////////
static long double ldGaussian(
_ldIntegralFunction_t ldF,
long double ldA,
long double ldB,
long double *pldConstArr
)
{
   const long double ldGLWeights[] = {
      2.78342835580868333037e-002L,
      6.27901847324523123016e-002L,
      9.31451054638671257124e-002L,
      1.16596882295995240001e-001L,
      1.31402272255123331128e-001L,
      1.36462543388950315360e-001L,
      1.31402272255123331128e-001L,
      1.16596882295995240001e-001L,
      9.31451054638671257124e-002L,
      6.27901847324523123016e-002L,
      2.78342835580868333037e-002L,
   };
   const long double ldGLAbscissae[] = {
      1.08856709269715036124e-002L,
      5.64687001159523504590e-002L,
      1.34923997212975337962e-001L,
      2.40451935396594092037e-001L,
      3.65228422023827513853e-001L,
      5.00000000000000000000e-001L,
      6.34771577976172486147e-001L,
      7.59548064603405907963e-001L,
      8.65076002787024662065e-001L,
      9.43531299884047649541e-001L,
      9.89114329073028496360e-001L,
   };
   long j;                /* Array index */
   long double ldXOffset; /* X scale factor for the interval */
   long double ldSum;     /* Partial summation term */
   long double ldCenter = ( ldB + ldA ) * 0.5; /* Average of b and a */
   long double ldH = ( ldB - ldA ) * 0.5; /* Half the width of the interval */

   long double ldIntegralResult = 0.0e0L;

   for ( j = 0; j < 11; j++ )
   {
      /* Calculate f(x) at two symmetrical offsets from the center: */
      ldXOffset = ldH * ldGLAbscissae[j];
      ldSum = ( *ldF )( ldCenter - ldXOffset, pldConstArr ) +
              ( *ldF )( ldCenter + ldXOffset, pldConstArr );
      ldIntegralResult += ldGLWeights[j] * ldSum;
   }
   return ( ldIntegralResult * ldH );
}
