#include "integral.hpp"
#include <float.h>
// internal prototype:
static long double qxrul(
_ldIntegralFunction_t ldF,
long double ldA,
long double ldB,
int  iRuleChoice,
long double *pldConstArray,
long double *pldVectorOne,
long double *pldVectorTwo,
int *pI1,
int *pI2
);

#define HALF_KNOT_COUNT 21
#define ABS( a ) ( a ) > 0 ? ( a ) : -( a )
///////////////////////////////////////////////////////////////////////////
// Function ldIntegrate() performs numerical integration over a supplied
// interval (input parameters ldX0 to ldX1) for the supplied function.
//
// This formula used and the associated constants are from:
//
// the "Encyclopedic Dictionary of Mathematics, Volume 1"
// "1 Abel to 304 Numerical Solution of Partial Differential Equations"
// By  Kiyosi Ito.
//
// Imprint Cambridge, MA : MIT Press, 1993.
//
// ISBN    0262590204 ( set : PB )
//         0262090260 ( set : HB )
//
// LCCN    93139017
// Article 299 ( XV.7 ) Numerical Integration, pp 1120-1.
//
///////////////////////////////////////////////////////////////////////////
// Constructor:
//////////////////////////////////////////////////////////////////////////
EXPORT Integral::Integral()
{
   // Set up the integration interval values;
   ldX0 = 0.0e0L;
   ldX1 = 0.0e0L;
   ldFprime = ( _ldIntegralFunction_t ) NULL;
   pldConstArray = ( long double * ) NULL;
}


EXPORT Integral::Integral(
long double ldStart,
long double ldEnd,
_ldIntegralFunction_t ldF
)
{
   // Set up the integration interval values;
   ldX0 = ldStart;
   ldX1 = ldEnd;
   ldFprime = ldF;
   pldConstArray = ( long double * ) NULL;
   Evaluate();
}
EXPORT Integral::Integral(
long double ldStart,
long double ldEnd,
_ldIntegralFunction_t ldF,
long double *pldConstArrayIn
)
{
   // Set up the integration interval values;
   ldX0 = ldStart;
   ldX1 = ldEnd;
   ldFprime = ldF;
   pldConstArray = pldConstArrayIn;
   Evaluate();
}
void EXPORT Integral::Evaluate()
{
   // Empirically derived for long double stepsize:
   const long double ldSmallEnoughInterval = 0.25e0L;

   // If interval is too large, divide the integral into two equal sections.
   if ( ( ldX1 - ldX0 ) > ldSmallEnoughInterval )
   {
           long double ldMidpoint = ldX0 + ( ldX1 - ldX0 ) * 0.5e0L;
           Integral i0( ldX0, ldMidpoint, ldFprime, pldConstArray );
           Integral I1( ldMidpoint, ldX1, ldFprime, pldConstArray );
           ldIntegralResult = i0.ldGetIntegralResult() + I1.ldGetIntegralResult();
   }
   else
   {
      const long double ldTol = LDBL_EPSILON*2.0e0L;
      int I1 = -1;
      int I2 = -1;
      long double pldVectorOne[HALF_KNOT_COUNT] = { 0 };
      long double pldVectorTwo[HALF_KNOT_COUNT] = { 0 };
      int iRuleChoice = 0;
      long double ldResult0;
      long double ldResult1;
      ldResult0 = qxrul( ldFprime, ldX0, ldX1, iRuleChoice++, pldConstArray, pldVectorOne, pldVectorTwo, &I1, &I2 );
      ldResult1 = qxrul( ldFprime, ldX0, ldX1, iRuleChoice++, pldConstArray, pldVectorOne, pldVectorTwo, &I1, &I2 );
      if ( ABS( ( ldResult0 - ldResult1 )/ldResult1 ) > ldTol )
      {
         ldResult0 = ldResult1;
         ldResult1 = qxrul( ldFprime, ldX0, ldX1, iRuleChoice++, pldConstArray, pldVectorOne, pldVectorTwo, &I1, &I2 );
         if ( ABS( ( ldResult0 - ldResult1 )/ldResult1 ) > ldTol )
         {
            ldResult1 = qxrul( ldFprime, ldX0, ldX1, iRuleChoice, pldConstArray, pldVectorOne, pldVectorTwo, &I1, &I2 );
         }
      }
      ldIntegralResult = ldResult1;
   }
}

///////////////////////////////////////////////////////////////////////////
// Destructor:
//////////////////////////////////////////////////////////////////////////
EXPORT Integral::~Integral()
{
}

//////////////////////////////////////////////////////////////////////////
// Inquiry Methods:
//////////////////////////////////////////////////////////////////////////
void EXPORT Integral::vDisplayResultsToConsole()    // Display the results to the console
{
}
long double EXPORT Integral::ldGetIntegralResult(void)
{
   return ldIntegralResult;
}
long double EXPORT Integral::ldGetX0() // Get the start of the interval of integration
{
   return ldX0;
}
long double EXPORT Integral::ldGetX1() // Get the end of the interval of integration
{
   return ldX1;
}

//////////////////////////////////////////////////////////////////////////
// Modifier Methods:
//////////////////////////////////////////////////////////////////////////
// Set the Derivative Function
void EXPORT Integral::vSetFprime( _ldIntegralFunction_t ldF )
{
   ldFprime = ldF;
}

void EXPORT Integral::vSetX0( long double ldX ) // Set the start of the interval
{
   ldX0 = ldX;
}
void EXPORT Integral::vSetX1( long double ldX ) // Set the end of the interval
{
   ldX1 = ldX;
}
void EXPORT Integral::vSetConstArr( long double *pldConst ) // Set the end of the interval
{
   pldConstArray = pldConst;
}

/*
** These routines are derived from function QXRUL from:
** ALGORITHM 691, COLLECTED ALGORITHMS FROM ACM.
** THIS WORK PUBLISHED IN TRANSACTIONS ON MATHEMATICAL SOFTWARE,
** VOL. 17, NO. 2, PP. HALF_KNOT_COUNT8-232. JUNE, 1991.
**
** This was a component of the SLATEC FORTRAN library created by:
**
** LAWRENCE LIVERMORE NATIONAL LABORATORY
**
** To compute I = integral of f() over ( ldA, ldB ), with error estimate
**
** Parameters on entry:
** f() - long double function to be integrated
**
** ldA - long double lower limit of integration
**
** ldB - long double upper limit of integration
**
**
** iRuleChoice - int
**    Selector for choice of local integration rule. Rule used with:
**       13 points if iRuleChoice = 0,
**       19 points if iRuleChoice = 1,
**       27 points if iRuleChoice = 2,
**       42 points if iRuleChoice = 3
**
**
** pldVectorOne - long double
**    vector containing pI1 saved functional values of positive abscissas.
**
** pldVectorTwo - long double
**    vector containing pI2 saved functional values of negative abscissas.
**
** pI1, pI2 - int  number of elements in pldVectorOne, pldVectorTwo respectively.
**
** On return
** ldY - long double approximation to the integral I.
**
**
** pldVectorOne - long double
**    vector containing pI1 saved functional values of positive abscissas.
**
** pldVectorTwo - long double
**    vector containing pI2 saved functional values of negative abscissas.
**
** pI1, pI2 - int  number of elements in pldVectorOne, pldVectorTwo respectively.
*/
static long double qxrul(
_ldIntegralFunction_t ldF,
long double ldA,
long double ldB,
int  iRuleChoice,
long double *pldConstArray,
long double *pldVectorOne,
long double *pldVectorTwo,
int *pI1,
int *pI2
)
{
   /* Starting positions into the weights array: */
   int iStart[] = { 0, 7, 17, 31 };

   /* Lengths of the weight sequence to process: */
   int iLen[]   = { 7, 10, 14, HALF_KNOT_COUNT };

   /* Symmetrical offsets +/- for unit interval */
   const long double ldAbscissae[HALF_KNOT_COUNT] =
   {
      0.0000000e0L, 0.2500e0L, 0.5000e0L, 0.750000e0L, 0.87500e0L,
      0.9375000e0L, 1.0000e0L, 0.3750e0L, 0.625000e0L, 0.96875e0L,
      0.1250000e0L, 0.6875e0L, 0.8125e0L, 0.984375e0L, 0.18750e0L,
      0.3125000e0L, 0.4375e0L, 0.5625e0L, 0.843750e0L, 0.90625e0L,
      0.9921875e0L
   };
   const long double ldWeights[52] =
   {
      0.1303262173284849021810473057638590518409112513421e0L,
      0.2390632866847646220320329836544615917290026806242e0L,
      0.2630626354774670227333506083741355715758124943143e0L,
      0.2186819313830574175167853094864355208948886875898e0L,
      0.027578976466428368658596011976074715743366742067e0L,
      0.105575010053845844336503487908666979130555049383e0L,
      0.01571194260595182254168429283636656908546309467968e0L,
      0.1298751627936015783241173611320651866834051160074e0L,
      0.2249996826462523640447834514709508786970828213187e0L,
      0.1680415725925575286319046726692683040162290325505e0L,
      0.1415567675701225879892811622832845252125600939627e0L,
      0.1006482260551160175038684459742336605269707889822e0L,
      0.02510604860724282479058338820428989444699235030871e0L,
      0.009402964360009747110031098328922608224934320397592e0L,
      0.0554269923329587516840678369514364633827480535978e0L,
      0.09986735247403367525720377847755415293097913496236e0L,
      0.04507523056810492466415880450799432587809828791196e0L,
      0.06300942249647773931746170540321811473310938661469e0L,
      0.1261383225537664703012999637242003647020326905948e0L,
      0.127386443358102827287870998185030736345352311788e0L,
      0.08576500414311820514214087864326799153427368592787e0L,
      0.07102884842310253397447305465997026228407227220665e0L,
      0.05026383572857942403759829860675892897279675661654e0L,
      0.004683670010609093810432609684738393586390722052124e0L,
      0.1235837891364555000245004813294817451524633100256e0L,
      0.1148933497158144016800199601785309838604146040215e0L,
      0.0125257577422612263339147770259358530725452719807e0L,
      0.123957239623183424219418967424381861904228081664e0L,
      0.02501306413750310579525950767549691151739047969345e0L,
      0.04915957918146130094258849161350510503556792927578e0L,
      0.02259167374956474713302030584548274729936249753832e0L,
      0.06362762978782724559269342300509058175967124446839e0L,
      0.09950065827346794643193261975720606296171462239514e0L,
      0.07048220002718565366098742295389607994441704889441e0L,
      0.06512297339398335645872697307762912795346716454337e0L,
      0.03998229150313659724790527138690215186863915308702e0L,
      0.03456512257080287509832054272964315588028252136044e0L,
      0.002212167975884114432760321569298651047876071264944e0L,
      0.08140326425945938045967829319725797511040878579808e0L,
      0.06583213447600552906273539578430361199084485578379e0L,
      0.02592913726450792546064232192976262988065252032902e0L,
      0.1187141856692283347609436153545356484256869129472e0L,
      0.05999947605385971985589674757013565610751028128731e0L,
      0.0550093798019804173691025798834610183906258148982e0L,
      0.005264422421764655969760271538981443718440340270116e0L,
      0.01533126874056586959338368742803997744815413565014e0L,
      0.03527159369750123100455704702965541866345781113903e0L,
      0.05000556431653955124212795201196389006184693561679e0L,
      0.05744164831179720106340717579281831675999717767532e0L,
      0.01598823797283813438301248206397233634639162043386e0L,
      0.02635660410220884993472478832884065450876913559421e0L,
      0.01196003937945541091670106760660561117114584656319e0L
   };
   long double ldY = 0.0e0L;
   long double ldOffset;
   long double ldMidpoint;
   long double ldHalfInterval;
   int         iKnots;
   int         i;

   iKnots = iLen[iRuleChoice];
   ldHalfInterval = ( ldB - ldA ) * 0.5;
   ldMidpoint = ldA + ldHalfInterval;
   for ( i = 0; i < iKnots; i++ )
   {
      if ( i >= *pI1 || i >= *pI2 )
      {
         ldOffset = ldHalfInterval * ldAbscissae[i];
         if ( i >= *pI1 )
         {
            pldVectorOne[i] = ( *ldF )( ldMidpoint + ldOffset, pldConstArray );
         }
         if ( i >= *pI2 )
         {
            pldVectorTwo[i] = ( *ldF )( ldMidpoint - ldOffset, pldConstArray );
         }
      }
      ldY += ( pldVectorOne[i] + pldVectorTwo[i] ) * ldWeights[iStart[iRuleChoice] + i];
   }
   ldY *= ldHalfInterval;
   if ( *pI1 <= iKnots )
   {
      *pI1 = iKnots;
   }
   if ( *pI2 <= iKnots )
   {
      *pI2 = iKnots;
   }
   return ldY;
}
