#include "integral.hpp"
///////////////////////////////////////////////////////////////////////////
// Function ldIntegrate() performs numerical integration over a supplied
// interval (input parameters ldX0 to ldX1) for the supplied function.
//
// This formula used and the associated constants are from:
//
// the "Encyclopedic Dictionary of Mathematics, Volume 1"
// "1 Abel to 304 Numerical Solution of Partial Differential Equations"
// By  Kiyosi Ito.
//
// Imprint Cambridge, MA : MIT Press, 1993.
//
// ISBN    0262590204 ( set : PB )
//         0262090260 ( set : HB )
//
// LCCN    93139017
// Article 299 ( XV.7 ) Numerical Integration, pp 1120-1.
//
///////////////////////////////////////////////////////////////////////////
// Constructor:
//////////////////////////////////////////////////////////////////////////
EXPORT Integral::Integral()
{
   // Set up the integration interval values;
   ldX0 = 0.0e0L;
   ldX1 = 0.0e0L;
   ldFprime = ( _ldIntegralFunction_t ) NULL;
   pldConstArray = ( long double * ) NULL;
}


EXPORT Integral::Integral(
long double ldStart,
long double ldEnd,
_ldIntegralFunction_t ldF
)
{
   // Set up the integration interval values;
   ldX0 = ldStart;
   ldX1 = ldEnd;
   ldFprime = ldF;
   pldConstArray = ( long double * ) NULL;
   Evaluate();
}
EXPORT Integral::Integral(
long double ldStart,
long double ldEnd,
_ldIntegralFunction_t ldF,
long double *pldConstArrayIn
)
{
   // Set up the integration interval values;
   ldX0 = ldStart;
   ldX1 = ldEnd;
   ldFprime = ldF;
   pldConstArray = pldConstArrayIn;
   Evaluate();
}
void EXPORT Integral::Evaluate()
{
   // We use this constant to divide the interval into two equal parts:
   const long double ldHalf = 0.5e0L;

   // Empirically derived for long double stepsize:
   const long double ldStepsizeForBestAccuracy = 0.01e0L;

   // The stepsize used for the integral estimation
   // The number .1 is due to the 11 terms in the Newton-Cotes formula
   // creating 10 sub-intervals.
   const long double ldH = ( ldX1 - ldX0 )*0.1e0L;

   // If stepsize is too large, divide the integral into two equal sections.
   // Note that the maximum possible error is:
   // -(1,346,350/326,918,592)*h^13*epsilon, where epsilon is some value
   // of the 12th derivative of the supplied function in our sub-interval.
   // See "Handbook of Mathematical Functions" by Abramowitz and Stegun,
   // formula 25.4.20 on page 887.
   if ( ldH > ldStepsizeForBestAccuracy )
   {
           long double ldMidpoint = ldX0 + ( ldX1 - ldX0 )*ldHalf;
           Integral i0( ldX0, ldMidpoint, ldFprime, pldConstArray );
           Integral i1( ldMidpoint, ldX1, ldFprime, pldConstArray );
           ldIntegralResult = i0.ldGetIntegralResult() + i1.ldGetIntegralResult();
   }
   else
   {  // Newton-Cotes constant for calculating the integral:
           const long double A = 5.0e0L/299376.0e0L;

           // Perform the summation.
           ldIntegralResult = (
                16067e0L * ( ldFprime( ldX0, pldConstArray )
                            +ldFprime( ldX0 + ldH*10, pldConstArray ) )
             + 106300e0L * ( ldFprime( ldX0 + ldH,    pldConstArray )
                            +ldFprime( ldX0 + ldH*9,  pldConstArray ) )
              - 48525e0L * ( ldFprime( ldX0 + ldH*2,  pldConstArray )
                            +ldFprime( ldX0 + ldH*8,  pldConstArray ) )
             + 272400e0L * ( ldFprime( ldX0 + ldH*3,  pldConstArray )
                            +ldFprime( ldX0 + ldH*7,  pldConstArray ) )
             - 260550e0L * ( ldFprime( ldX0 + ldH*4,  pldConstArray )
                            +ldFprime( ldX0 + ldH*6,  pldConstArray ) )
             + 427368e0L *   ldFprime( ldX0 + ldH*5,  pldConstArray )
             ) * ldH * A;
   }
}

///////////////////////////////////////////////////////////////////////////
// Destructor:
//////////////////////////////////////////////////////////////////////////
EXPORT Integral::~Integral()
{
}

//////////////////////////////////////////////////////////////////////////
// Inquiry Methods:
//////////////////////////////////////////////////////////////////////////
void EXPORT Integral::vDisplayResultsToConsole()    // Display the results to the console
{
}
long double EXPORT Integral::ldGetIntegralResult(void)
{
   return ldIntegralResult;
}
long double EXPORT Integral::ldGetX0() // Get the start of the interval of integration
{
   return ldX0;
}
long double EXPORT Integral::ldGetX1() // Get the end of the interval of integration
{
   return ldX1;
}

//////////////////////////////////////////////////////////////////////////
// Modifier Methods:
//////////////////////////////////////////////////////////////////////////
// Set the Derivative Function
void EXPORT Integral::vSetFprime( _ldIntegralFunction_t ldF )
{
   ldFprime = ldF;
}

void EXPORT Integral::vSetX0( long double ldX ) // Set the start of the interval
{
   ldX0 = ldX;
}
void EXPORT Integral::vSetX1( long double ldX ) // Set the end of the interval
{
   ldX1 = ldX;
}
void EXPORT Integral::vSetConstArr( long double *pldConst ) // Set the end of the interval
{
   pldConstArray = pldConst;
}
