// File: bkthread.cpp

#include <somd.hh>

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#include <os2.h>

#include <ithread.hpp>   // IThread from IBM's OpenClass Library

#include "TChat.hh"      // Our SOM Class Declaration
#include "TChat.h"       // Resource ID for Our Icon
#include "Useful.hpp"    // Some Useful Items for DSOM Work
#include "Chatroom.hh"   // Chatroom DSOM Class
#include "BkThread.hpp"  // Background Thread Class

/**********************************************************************
 *        Background Thread for Issuing DSOM Requests over LAN
 *
 * Since DSOM requests, potentially being issued over a LAN, can take
 * non-zero time, we don't want the desktop to be frozen for that time
 * so we issue the requests on a background thread.  There is one such
 * thread per instance of the TChat class, not one per TChat object.
 *
 **********************************************************************/

/*******************************
 * Global Constants and Macros
 *******************************/

/*******************************
 *    Structures and Types
 *******************************/

/*******************************
 *     Function Prototypes
 *******************************/
static void handleJoinRequest(TJoinRequest *req);
static void handleQuitRequest(TQuitRequest *req);
static void handlePostRequest(TPostRequest *req);

extern void ShowMsg(const char *title, const char *text);

/*******************************
 *      Global Variables
 *******************************/

/**********************************************************************
 *            Constructor for a Background Thread Object
 *
 * This method initializes the object and then starts the actual thread.
 *
 **********************************************************************/
BackThread::BackThread(M_TChat *metaclass)
: _running(FALSE),
  _wantStop(FALSE),
  __SOMEnv( SOM_CreateLocalEnvironment() ),
  _metaclass(metaclass),
  _reqQueue("TChat")
{
    _reqThread.setStackSize(65536);
    _reqThread.start(this, true /* Allocate a PM Message Queue */ );
}
/**********************************************************************
 *              Destructor foa Background Thread Object
 *
 * This method stops the thread and destroys the object.  It tries to
 * be nice by first asking the thread to terminate and then killing it
 * if it does not comply.  Killing a thread is a bad thing as it stops
 * it from getting a change at cleaning up but sometimes you must.
 *
 **********************************************************************/
BackThread::~BackThread()
{
    if (isRunning()) {

        // Set flag telling service thread to stop service and allows
        // a little time for the current request to finish.  Usually
        // things work out where it ends before we kill it.
        _wantStop = TRUE;
        IThread::current().sleep(250); // Pause Freezes the Desktop (.25 Sec)

        if (isRunning()) { // If Still Running, Then Kill It
            try {
                _reqThread.stop(); // Kill the Thread by Force
            } catch (IException& exc) {
                // Thread Must Not Actually Be Running
                somPrintf("Warning - Unable to Kill Background Thread\n");
            }
        }
    }

    if (__SOMEnv)
        SOM_DestroyLocalEnvironment(__SOMEnv);
}
/**********************************************************************
 *     Thread that Processes Incoming SOM Requests of Our Objects
 *
 * This method is the actual thread execution code itself.  It loops
 * processing incoming requests and issuing DSOM calls to remote
 * objects.
 *
 **********************************************************************/
void
BackThread::run()
{
    SOMLocalEnv   env;            // We Need a Locally-Scoped SOM Environment
    Environment  *__SOMEnv = env.base(); // And Set the DTS-Magic Variable
    TRequest *req;

    somPrintf("Background Thread Started\n");

    // ********************************************
    // Insure DSOM Communications Daemon is Running
    //           (Experimental but Risky)
    // ********************************************
// if (!WinIsSOMDDReady()) {      // If DSOM Daemon is Not Running,
//     somPrintf("(Daemon SOMDD.EXE Not Already Running, Starting It...)\n");
//     if (WinRestartSOMDD(1)) {  // Then Start It
//         somPrintf("ERROR - Cannot Find and Unable to Start SOMDD.EXE\n");
//         return; // ERROR - Unable to Find/Start SOMDD.EXE Daemon
//     }
// }

    // ****************************************************
    // Loop and Handle Incoming Requests Until Told to Stop
    // ****************************************************
    _running = TRUE; // We're Not Running Until We Can Take Requests
    while (!_wantStop && (req = _reqQueue.getMsg(TRUE)) != NULL) {

        switch (req->cmd()) {
        case JOIN_REQUEST:
            handleJoinRequest((TJoinRequest *)req);
            break;

        case POST_REQUEST:
            handlePostRequest((TPostRequest *)req);
            break;

        case QUIT_REQUEST:
            handleQuitRequest((TQuitRequest *)req);
            break;

        default:
            somPrintf("Got Request %d (UNKNOWN)\n", req->cmd());
            break;
        }

        delete req;
    }

    somPrintf("Background Thread Stopped\n");
    _running = FALSE;
}
/**********************************************************************
 *    Process a Request from the Desktop to Join a Remote Chatroom
 *
 * This method joins a desktop object to a chatroom, and will quit from
 * an existing chatroom if a link already exists.
 *
 **********************************************************************/
static void
handleJoinRequest(TJoinRequest *req)
{
    SOMLocalEnv   env; // We Need a Locally-Scoped SOM Environment
    Environment  *__SOMEnv = env.base(); // And Set the DTS-Magic Variable

    somPrintf("Got Request to Join Session '%s' as User '%s'\n",
        req->room(), req->user());

    Chatroom *roomObj;

    switch (req->client()->state()) {
    case TChat::CONNECTED: // Joining But Already Connected - Quit First
        req->client()->setState(TChat::DISCONNECTING);

        roomObj = req->client()->roomObject();
        if (!roomObj) {
            somPrintf("ASSERT in %s - Room Object Should Not Be Null\n",
                      __FUNCTION__);
            break;
        }

        somPrintf("Invoking Remote Method quitSession()...\n");
        if (roomObj->quitSession(req->client()))
            somPrintf("... Quit Session OK\n");
        else
            somPrintf("... Unable to Quit Session\n");

        SOMD_ObjectMgr->somdReleaseObject(roomObj);

        req->client()->setRoomObject(NULL);
        req->client()->setState(TChat::DISCONNECTED);

        // Fall-Thru Intentional

    case TChat::DISCONNECTED: // Not Joined Already So Just Join Now
        roomObj = (Chatroom *)SOMD_ObjectMgr->somdFindServerByName((string)req->room());
        if (roomObj) {
            req->client()->setState(TChat::CONNECTING);
            req->client()->setRoomObject(roomObj);

            somPrintf("Invoking Remote Method joinSession()...\n");
            if (roomObj->joinSession(req->client(), req->user()))
                somPrintf("Joined Session OK\n");
            else
                somPrintf("Unable to Join Session\n");

            // Now Joined to a Room, Set Our Variables
            req->client()->setUserName( req->user() );
            req->client()->setRoomName( req->room() );
            req->client()->setState(TChat::CONNECTED);

        } else {
            somPrintf("ERROR in TChat SOM Example, Unable to Find Chatroom\n");
            ShowMsg("ERROR in TChat SOM Example", "Unable to Find Chatroom");
        }
        break;

    default:
        somPrintf("Invalid Connection State in Background Thread\n");
        break;
    }
}
/**********************************************************************
 *    Process a Request from the Desktop to Leave a Remote Chatroom
 *
 * This method disconnects a desktop object from a chatroom.
 *
 **********************************************************************/
static void
handleQuitRequest(TQuitRequest *req)
{
    SOMLocalEnv   env; // We Need a Locally-Scoped SOM Environment
    Environment  *__SOMEnv = env.base(); // And Set the DTS-Magic Variable

    somPrintf("Got Request to Quit Session '%s' as User '%s'\n",
        req->client()->roomName(), req->client()->userName());

    Chatroom *roomObj;

    switch (req->client()->state()) {
    case TChat::CONNECTED:
        req->client()->setState(TChat::DISCONNECTING);

        somPrintf("Invoking Remote Method quitSession()...\n");
        if (req->client()->roomObject()->quitSession(req->client())) {
            somPrintf("... Quit Session OK\n");
        } else {
            somPrintf("... Unable to Quit Session\n");
        }

        roomObj = req->client()->roomObject();

        somPrintf("In %s, About to Release Server Object 0x%08lX\n",
            __FUNCTION__, roomObj);

        if (roomObj)
            SOMD_ObjectMgr->somdReleaseObject(roomObj);

        req->client()->setRoomObject(NULL);
        req->client()->setState(TChat::DISCONNECTED);
        break;

    case TChat::DISCONNECTED:
        break;

    default:
        somPrintf("Invalid Connection State in Background Thread\n");
        break;
    }

    req->client()->reallyDelete();
}
/**********************************************************************
 *        Process a Request from the Desktop to Post Some Text
 *
 * This method sends text to the currently joined chatroom for sharing
 * among the other participants.  If there is no connection to a
 * chatroom, the text is dropped.
 *
 **********************************************************************/
static void
handlePostRequest(TPostRequest *req)
{
    SOMLocalEnv   env; // We Need a Locally-Scoped SOM Environment
    Environment  *__SOMEnv = env.base(); // And Set the DTS-Magic Variable

    somPrintf("Got Request to Post Text '%s' to Session '%s' as User '%s'\n",
        req->text(), req->client()->roomName(), req->client()->userName());

    if (req->client()->state() == TChat::CONNECTED) {
        if (req->client()->roomObject())
            req->client()->roomObject()->postSession(req->text());
        else
            somPrintf("ASSERT in %s - Room Object Should Not Be Null\n",
                       __FUNCTION__);
    }
}
/**********************************************************************
 *     Constructor for the Class Wrapper of OS/2's QUEUE Facility
 *
 * Since we use queues for sending pointers to requests from one thread
 * to another, there is no need for many of the features of the QUEUE
 * facility such as length and passing the data through shared memory
 * between processes.  We just want to pass a pointer in a single
 * address space.
 *
 **********************************************************************/
TQueue::TQueue(const char *name, BOOL createfirst)
: _handle(NULL)
{
    char    qname[CCHMAXPATH];
    PID     hisPid;
    APIRET  rc;

    strcpy(qname, "\\QUEUES\\");
    strcat(qname, name);

    if (createfirst) // Create a New Queue
        rc = DosCreateQueue(&_handle, QUE_FIFO|QUE_CONVERT_ADDRESS, qname);
    else             // Open an Existing Queue
        rc = DosOpenQueue(&hisPid, &_handle, qname);

    if (rc != NO_ERROR)  // If An Error Occurred,
        _handle = 0;      // Then There is No Handle
}
/**********************************************************************
 *      Destructor for the Class Wrapper of OS/2's QUEUE Facility
 *
 **********************************************************************/
TQueue::~TQueue()
{
    if (_handle)
        DosCloseQueue(_handle);
}
/**********************************************************************
 *            Return a Pointer to the Next Request Queued
 *
 * This method gets a pointer to the request, with an option whether to
 * block the process or just return if no request is pending.
 *
 **********************************************************************/
TRequest *
TQueue::getMsg(BOOL waitflag)
{
    REQUESTDATA  reqinfo;
    ULONG        reqlen;
    void        *dataptr;
    BYTE         priority;
    APIRET       rc;

    rc = DosReadQueue(_handle, &reqinfo, &reqlen, &dataptr, 0,
                      waitflag ? DCWW_WAIT : DCWW_NOWAIT, &priority, 0L);

    return (rc == NO_ERROR) ? (TRequest *)dataptr : NULL;
}
/**********************************************************************
 *            Stuff a Pointer to a Request into a Queue
 *
 **********************************************************************/
void
TQueue::putMsg(TRequest *req)
{
    DosWriteQueue(_handle, req->cmd(), 0L, (void *)req, 0);
}
/**********************************************************************
 *          Construct a Request to Join a Chatroom Session
 *
 **********************************************************************/
TJoinRequest::TJoinRequest(TChat       *client,
                           const char  *room,
                           const char  *asUser)
: TRequest(),
  chatobj(client),
  roomName( strdup(room) ),
  userName( strdup(asUser) )
{
}
/**********************************************************************
 *          Clean Up a Request to Join a Chatroom Session
 *
 **********************************************************************/
TJoinRequest::~TJoinRequest()
{
    delete roomName;
    delete userName;
}
/**********************************************************************
 *          Construct a Request to Leave a Chatroom Session
 *
 **********************************************************************/
TQuitRequest::TQuitRequest(TChat *client)
: TRequest(),
  chatobj(client)
{
}
/**********************************************************************
 *    Construct a Request to Post Some Text into a Chatroom Session
 *
 **********************************************************************/
TPostRequest::TPostRequest(TChat *client, const char *text)
: TRequest(),
  chatobj(client),
  sometext( strdup(text) )
{
}
/**********************************************************************
 *
 **********************************************************************/

