// File: chatroom.cpp

#define INCL_DOSPROCESS
#define INCL_NOPMAPI
#include <os2.h>

#include "Chatroom.hh"  // DTS SOM Declaration of Our Chatroom Class

/**********************************************************************
 *                      Chatroom DSOM Class
 *
 * A DSOM class is usually packaged as a DLL and this one is no different.
 * We define the _DLL_InitTerm() function that OS/2 invokes upon the
 * loading and unloading of each DLL, in order to monitor this activity.
 *
 * This DLL implements the remote server that all TChat objects
 * register with and post text to.  The server then echos all text
 * back to all participants for display.  This server has nothing to
 * do with the Workplace Shell and runs quietly in the background on
 * one machine in the LAN.
 *
 * This DLL must be registered in the implementation repository for it
 * to start up automatically.  See the PREGIMPL.EXE command for how.
 *
 **********************************************************************/

/*******************************
 * Global Constants and Macros
 *******************************/

/*******************************
 *    Structures and Types
 *******************************/

/*******************************
 *     Function Prototypes
 *******************************/
extern "C" int   _CRT_init(void); // C Runtime Init Function (0 = Success, -1 = Failure)

#ifdef __cplusplus
extern "C" void  __ctordtorInit(void); // C++ Constructor/Destructor Init
extern "C" void  __ctordtorTerm(void); // C++ Constructor/Destructor Init
#endif

#if defined(__IMPORTLIB__)
/* A clean up routine registered with DosExitList must be used if runtime */
/* calls are required and the runtime is dynamically linked.  This will   */
/* guarantee that this clean up routine is run before the library DLL is  */
/* terminated.                                                            */
static void APIENTRY _DLL_ExitHandler(ULONG exitReason);

#else
extern "C" void _CRT_term(void); // Runtime Terminator - Static Link Only
#endif

/*******************************
 *      Global Variables
 *******************************/
char *_dll_name = "Chatroom.dll"; // Convenient Ref to Our DLL Name

struct MemberReg { // Participant Tracking Record
    string      objref; // String Form of DSOM Object Reference
    SOMObject  *objptr; // Ptr to Proxy of Remote DSOM Object Itself
};

// Simple Table of Members - Could Be Much Improved
MemberReg members[10];  // Table of Up To 10 Participants per Chatroom
int numMembers = 0;

/**********************************************************************
 *                           _DLL_InitTerm
 *
 * This function gets called by the operating system loader when it
 * loads and frees this DLL for each process that accesses this DLL.
 * However, it only gets called the first time the DLL is loaded and
 * the last time it is freed for a particular process, not per thread.
 *
 **********************************************************************/
ULONG _System
_DLL_InitTerm(ULONG hModule, ULONG ulFlag)
{
    somPrintf("%s %s\n", ulFlag ? "Unloading" : "Loading", _dll_name);

    if (ulFlag) {        // DLL is Being Unloaded from This Process
        #if !defined(__IMPORTLIB__)
            _ctordtorTerm();
            _CRT_term(); // Only Called if Runtime is Statically Linked-In
        #else
            DosExitList(EXLST_REMOVE, _DLL_ExitHandler);
        #endif

        return TRUE; // Return Non-Zero Value to Indicate Success

    } else {           // DLL is Being Loaded into New Process
        if (_CRT_init() == -1) // Initialize the C Runtime DLL
            return FALSE; // ERROR - Unable to Initialize the C Runtime

        else {
            #if defined(__IMPORTLIB__)
                // Use a DosExitList routine if C Runtime is Dynamically Linked
                DosExitList(0x0000FF00 | EXLST_ADD, _DLL_ExitHandler);
            #endif

            __ctordtorInit();
        }
        return TRUE; // Return Non-Zero Value to Indicate Success
    }
}
/**********************************************************************
 *                       _DLL_ExitHandler
 *
 * This Dos Exit Handler gets called when the client process terminates,
 * in order to let the DLL perform any necessary cleanup.  It gets
 * called *BEFORE* _DLL_InitTerm().
 *
 * This routine currently does nothing useful except print a message
 * telling us when the exit handler was called.  This is good for
 * debugging but could be removed in production.
 *
 **********************************************************************/
#if defined(__IMPORTLIB__)
static void APIENTRY
_DLL_ExitHandler(ULONG exitReason)
{
    somPrintf("_DLL_ExitHandler of %s Called\n", _dll_name);

    switch (exitReason) {
    case TC_EXIT:        // Normal Process Exit
    case TC_HARDERROR:   // Hard Error Occurred
    case TC_TRAP:        // Trap in a 16-Bit Child Process
    case TC_KILLPROCESS: // Unintercepted DosKillProcess Call
    case TC_EXCEPTION:   // Exception in a 32-Bit Child Process
    default:
        break;
    }

    // NOTE: Destructors of Local C++ Objects NOT Called Due to Early-Exit
    DosExitList(EXLST_EXIT, 0); // Does Not Return - No Local DTors Called!!
}
#endif
/**********************************************************************
 * Function Called when This DLL is Dynamically Loaded via somFindClass
 *
 * This function does not appear to always be called, depending upon
 * how the SOM class is made available.  It seems to depend upon
 * whether the DLL is statically or dynamically linked into the client
 * and a host of other factors.  It also matters whether you define
 * the 'SOMInitModule' entry point in your .DEF file.
 *
 * It doesn't seem to be very important as the DTS compiler seems to
 * automatically add code to initialize classes when first seen.  This
 * is a poorly documented corner of SOM.
 *
 **********************************************************************/
SOMEXTERN void SOMLINK
SOMInitModule(long majorVersion, long minorVersion)
{
    somPrintf("SOMInitModule() of %s Called\n", _dll_name);

    ChatroomNewClass(0, 0); // Initialize Our Class with SOM Manager
}
/**********************************************************************
 *                     Construct a Chatroom
 *
 * This method exists solely for debugging purposes so we can watch.
 *
 **********************************************************************/
Chatroom::Chatroom()
{
    somPrintf("Invoking %s\n", __FUNCTION__);
}
/**********************************************************************
 *                       Destruct a Chatroom
 *
 * This method exists solely for debugging purposes so we can watch.
 *
 **********************************************************************/
Chatroom::~Chatroom()
{
    somPrintf("Invoking %s\n", __FUNCTION__);
}
/**********************************************************************
 *       Handle a Request to Post Some Text to All Participants
 * 
 * This method implements one of the magic parts of DSOM; asynchronous
 * invocation of a method in a remote object by textual name only.
 *
 * We iterate over all room members and send each of them a copy of
 * the text string.  Since each room member might be a different type
 * of object, we must locate the receiving method by name.  There is
 * no required inheritance relationship in room participants as a
 * result, just a need to define a method named 'displayText'.
 *
 **********************************************************************/
boolean
Chatroom::postSession(const char *postText)
{
    typedef void SOMLINK notifyFunc(SOMObject  *obj,  // Receiving Object
                                    const char *msg); // Method-Specific Args

    for (int i = 0; i < numMembers; i++) { // Echo to Each Participant...

        // ************************************************************
        // Prepare to Call a Method Named 'displayText' if 'obj' Supports It
        // We resolve by name to avoid dependency on the class of 'obj'.
        // ************************************************************
        somMethodProc *mp = somResolveByName(members[i].objptr,
                                             "displayText"); // Method Name

        if (!mp) { // If Method Not Supported,
            somPrintf("ERROR - Unable to Resolve 'displayText' Method\n");
            return FALSE;
        }

        // NOTE: All client 'displayText' members must take the same
        // set of arguments.  Since we are sending to WPS objects
        // which use the old IDL style (no environment parameter),
        // our 'displayText' method must omit one as well.  Because
        // we looked up the method name, we must use the non-object
        // calling style:
        //
        //     (*fnptr)(dest_obj, args...)
        //
        // rather than the form:
        //
        //     dest_obj->method(args...)

        ((notifyFunc *)mp)(members[i].objptr, postText);
    }
    return TRUE;
}
/**********************************************************************
 *            Handle a Request to Join a Chatroom Session
 *
 * This method registers a remote client object with the chatroom so it
 * will receive any copies of text posted.  The choice of room is
 * implicit since there is one 'chatroom' object per room and if we got
 * here, its for this room.  The user alias is not currently used but
 * could be used to prefix his comments or put up a graphic icon of who
 * is speaking.
 *
 * As the key, we use the string form of the client's DSOM object
 * reference rather than the address of the proxy object since it is
 * independent of where the proxy is in memory.  In other words, there
 * is no guarantee that the proxy created during the joining request
 * will be at the same address at the one created during the quit request.
 *
 **********************************************************************/
boolean
Chatroom::joinSession(SOMObject *obj, const char *userAlias)
{
    somPrintf("Chatroom: Object 0x%08lX is Joining as User %s\n",
        obj, userAlias);

    string objref = SOMD_ObjectMgr->somdGetIdFromObject(obj);
    somPrintf("Chatroom::joinSession objref 0x%08lX (%s)\n", objref, objref);

    for (int i = 0; i < numMembers; i++) { // Already Registered?
        if (strcmp(members[i].objref, objref) == 0) {

            somPrintf("Chatroom: Object Already In Table\n");
            return FALSE; // Already in Table
        }
    }

    // We convert the string reference back into a new proxy since the
    // proxy we receive in our arguments ('obj') will be destroyed when
    // we return from this method.  We need a proxy that will stay
    // around until the client quits from the room, so we can
    // asynchronously post text to the client object.

    members[numMembers].objptr = SOMD_ObjectMgr->somdGetObjectFromId(objref);
    members[numMembers].objref = objref; // Keep the String Ref
    numMembers++; // And Increment Count of Room Members

    return TRUE;
}
/**********************************************************************
 *           Handle a Request to Leave a Chatroom Session
 *
 * This method deregisters the remote client object from the chatroom.
 * As the key, it uses the string form of the client's DSOM object
 * reference rather than the address of the proxy object since it is
 * independent of where the proxy is in memory.  In other words, there
 * is no guarantee that the proxy created during the joining request
 * will be at the same address at the one created during the quit request.
 *
 * An alternative implementation would have been to return a handle or
 * key from the join method that must be provided to quit.
 *
 **********************************************************************/
boolean
Chatroom::quitSession(SOMObject *obj)
{
    somPrintf("Chatroom: Object 0x%08lX is Quitting\n", obj);

    string objref = SOMD_ObjectMgr->somdGetIdFromObject(obj);
    somPrintf("Chatroom::quitSession objref 0x%08lX (%s)\n", objref, objref);

    for (int i = 0; i < numMembers; i++) {
        if (strcmp(members[i].objref, objref) == 0) { // Found the Member

            SOMFree(members[i].objref); // Free String Form of Object Ref

            // Destroy Our Proxy to the Remote Object
            //   (But Not the Remote Object Itself)
            SOMD_ObjectMgr->somdReleaseObject(members[i].objptr);

            SOMFree(objref); // Free Our Temporary String Form of Ref

            for (int j = i + 1; j < numMembers; j++)
                members[i] = members[j]; // Slide the Entries Upward

            numMembers--; // And Reduce the Count of Room Members
            return TRUE;
        }
    }

    return FALSE;
}
/**********************************************************************
 *
 **********************************************************************/

