// File: chatview.cpp

#include <stdio.h>
#include <stdlib.h>

#include <imsgbox.hpp>   // IMessageBox from IBM's OpenClass Library
#include <ifont.hpp>     // IFont from IBM's OpenClass Library

#include "ChatView.hpp"  // Declaration of OpenClass-Derived ChatView Class
#include "TChat.h"       // Resource IDs for GUI Elements of TChat

/**********************************************************************
 *          View of TChat Object When Opened on the Desktop
 *            (Implemented Using IBM's OpenClass Library)
 *
 * The class 'ChatView' represents the window that appears when an
 * instance of the TChat class is opened on the desktop.  It presents
 * a split-canvas where incoming keystrokes appear in the top pane and
 * the user types into the lower pane.  There is also a one-line status
 * line on the bottom of the window to report the status of the link
 * to the Chatroom object.
 *
 * Also in this file is the 'ChatInput' class which is just a wrapper
 * of the MultiLineEdit object that represents the lower pane.  We use
 * it to intercept keystrokes into the pane and transmit them to the
 * Chatroom object.
 *
 **********************************************************************/

/*******************************
 * Global Constants and Macros
 *******************************/

/*******************************
 *    Structures and Types
 *******************************/

/*******************************
 *     Function Prototypes
 *******************************/

/*******************************
 *      Global Variables
 *******************************/

/**********************************************************************
 *        Constructor for the Composite ChatView Window Object
 *
 * Each instance of a ChatView window 'belongs' to a corresponding SOM
 * object that acts as its Workplace Shell anchor.  A WPS object can
 * own multiple views, but each view only has one owning WPS object.
 *
 **********************************************************************/
ChatView::ChatView(TChat *obj)
: IFrameWindow(),                     // Our View is a Simple FrameWindow
  myTitle(this, obj->wpQueryTitle()), // And Our Title is Our WPS Object Name
  chatArea(ID_CHATAREA, this, this),
  upperPane(ID_UPPERPANE, &chatArea, &chatArea, IRectangle(),
            IMultiLineEdit::defaultStyle() | IMultiLineEdit::readOnly),
  lowerPane(this, ID_LOWERPANE, &chatArea, &chatArea, IRectangle(),
            IMultiLineEdit::defaultStyle()),
  statusArea(ID_STATUSAREA, this, this)
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    somObj = obj; // Stash a Pointer to Our Representative SOM Object

    upperPane.setLimit(2048);  // Allow 2048 Characters in Upper Pane
    lowerPane.setLimit(2048);  // Allow 2048 Characters in Lower Pane

    sizeTo( ISize(640, 400) ); // Chose a Nice Starting Window Size
    chatArea.setOrientation( ISplitCanvas::horizontalSplit );

    setClient(&chatArea);
    handleEventsFor(this);

    IFont tempFont("Helv", 12); // Chose a Nice Font for Text Display
    statusArea.setFont(tempFont)
              .setForegroundColor( IColor(IColor::blue) );

    addExtension(&statusArea,   // Add the Status Line to the Bottom
                 IFrameWindow::belowClient,
                 IFont(&statusArea).maxCharHeight());

    setState(somObj->state()); // Set Visible State from Internal State

    // ************************************
    // Construct a WPS View Tracking Record
    // ************************************
    myUseItem = (PUSEITEM)somObj->wpAllocMem(
                          sizeof(USEITEM) + sizeof(VIEWITEM), NULL);

    PVIEWITEM pViewItem = (PVIEWITEM)(myUseItem + 1);
    memset(myUseItem, 0, sizeof(USEITEM) + sizeof(VIEWITEM));

    myUseItem->type   = USAGE_OPENVIEW;
    pViewItem->view   = viewID();
    pViewItem->handle = handle();

    somObj->wpAddToObjUseList(myUseItem);
    somObj->wpRegisterView(handle(), "Chat View");

    // ********************
    // Finish Up Loose Ends
    // ********************
    setAutoDeleteObject(); // Tell WPS to Automatically Clean Up Object
    chatArea.setFocus();   // Make New View the Focus on the Desktop
    show();                // And Paint the Screen to Show It
}
/**********************************************************************
 *        Destructor for the Composite ChatView Window Object
 *
 * Clean up after the removal of a view, getting rid of its view
 * tracking record for WPS.
 *
 **********************************************************************/
ChatView::~ChatView()
{
    somObj->wpDeleteFromObjUseList(myUseItem);
    somObj->wpFreeMem((PBYTE)myUseItem);
}
/**********************************************************************
 *            Add Text to the Incoming-Keystroke Window
 *
 * This method is invoked by the owning TChat object whenever it gets
 * room participant's keystrokes from the remote Chatroom object.
 *
 **********************************************************************/
void
ChatView::displayText(const char *str)
{
    // Append Text to End of Upper MultiLineEdit Window
    try {
        upperPane.addAsLast(str);

        if (str[ strlen(str)-1 ] == '\r') // If Last Character is <ENTER>
            upperPane.addLineAsLast("");

    } catch (IException& exc) { // Just In Case of an OpenClass Exception
        ; // Ignore Inability to Append Character
        somPrintf("exception in ChatView::displayText()!!\n");

        IMessageBox  mbox(this);
        mbox.show(exc);  // Provide a Simple Warning Dialog Box
    }
}
/**********************************************************************
 *         Transmit the Text to the Others in the Conference
 *
 * This method is invoked when a keystroke is typed into the lower
 * pane.  It then passes it on to the owning TChat object for
 * transmission to the remote Chatroom object.
 *
 **********************************************************************/
void
ChatView::emitText(const char *str)
{
    somObj->emitText(str);
}
/**********************************************************************
 *   Adjust the Visually-Indicated State of the Link to the Chatroom
 *
 * This method changes the message that appears in the status line
 * across the bottom of the window to reflect changes in the link.
 *
 **********************************************************************/
void
ChatView::setState(short state)
{
    char buffer[100];

    switch (state) {
    case TChat::DISCONNECTED:
        setStatus("Disconnected");
        show();
        break;

    case TChat::CONNECTING:
        sprintf(buffer, "Connecting to %s as %s...",
            somObj->roomName(), somObj->userName());
        setStatus(buffer);
        show();
        break;

    case TChat::CONNECTED:
        sprintf(buffer, "Connected to %s as %s",
            somObj->roomName(), somObj->userName());
        setStatus(buffer);
        show();
        break;

    case TChat::DISCONNECTING:
        sprintf(buffer, "Disconnecting from %s as %s...",
            somObj->roomName(), somObj->userName());
        setStatus(buffer);
        show();
        break;
    }
}
/**********************************************************************
 *    Construct a Class to Obtain Keystrokes and Send Into Chatroom
 *
 * This method constructs a instance of the lower pane so we can get
 * keystrokes from the user for sending to a chatroom.  It might have
 * been possible to avoid this by adding a handler to the upper pane
 * MLE itself.
 *
 **********************************************************************/
ChatInput::ChatInput(ChatView      *recv,
                     unsigned long  id,
                     IWindow       *parent,
                     IWindow       *owner,
               const IRectangle&    initial,
               const Style&         style)
: IMultiLineEdit(id, parent, owner, initial, style)
{
    // Simple Pass-Thru Class But We Get to Override the Keyboard Functions
    _host = recv;

    IKeyboardHandler::handleEventsFor(this);
}
/**********************************************************************
 *           Handle the Keypress of a Displayable Character
 *
 **********************************************************************/
Boolean
ChatInput::characterKeyPress(IKeyboardEvent& event)
{
    if (event.isCtrlDown() || event.isAltDown())
        return false; // We Don't Transmit Control or Alt Keys

    char buf[2];

    buf[0] = event.character();
    buf[1] = '\0';

    _host->emitText(buf); // Tell Our TChat SOM Object to Transmit It

    return false; // Act Like We Didn't Handle It, Keeping Passing It
                  // (This Lets It Appear in the Lower Pane as Well)
}
/**********************************************************************
 *            Handle the Keypress of a Special Character
 *
 **********************************************************************/
Boolean
ChatInput::virtualKeyPress(IKeyboardEvent& event)
{
    switch (event.virtualKey()) {
    case IKeyboardEvent::tab:
        _host->emitText("\t"); // Tell Our TChat SOM Object to Transmit
        break;

    case IKeyboardEvent::newLine:
    case IKeyboardEvent::enter:
        _host->emitText("\r"); // Tell Our TChat SOM Object to Transmit
        break;

    case IKeyboardEvent::backSpace:
        _host->emitText("\b"); // Tell Our TChat SOM Object to Transmit
        break;

    default:
        break;
    }

    return false; // Act Like We Didn't Handle It, Keeping Passing It
                  // (This Lets It Appear in the Lower Pane as Well)
}
/**********************************************************************
 *
 **********************************************************************/

