// File: m_tchat.cpp

#include <somd.hh>

#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_WIN
#include <os2.h>

#include <ithread.hpp>   // IThread from IBM's OpenClass Library

#include "TChat.hh"      // Our SOM Class Declaration
#include "TChat.h"       // Resource ID for Our Icon
#include "Useful.hpp"    // Some Useful Items for DSOM Work
#include "Chatroom.hh"   // Chatroom DSOM Class
#include "BkThread.hpp"  // Background Thread Class

/**********************************************************************
 *                 Metaclass of TChat Class (M_TChat)
 *
 * This module contains the member functions of the M_TChat class,
 * which is the metaclass of the TChat class.  These functions relate
 * primarily to certain overhead operations of the Workplace Shell.
 *
 * A 'metaclass' under SOM means that these methods are for use by the
 * TChat class itself, NOT instances of the TChat class.  This is due
 * to classes being objects in their own right under SOM, unlike C++.
 *
 **********************************************************************/

/*******************************
 * Global Constants and Macros
 *******************************/

/*******************************
 *    Structures and Types
 *******************************/

/*******************************
 *     Function Prototypes
 *******************************/

/*******************************
 *      Global Variables
 *******************************/
extern HMODULE  MyModuleHandle;  // Module handle for our .DLL

/**********************************************************************
 *                   Constructor for M_TChat Class
 *
 * We just print a message here so we can watch the class initialize.
 * This more clearly reveals how SOM operates internally and makes
 * debugging a bit easier.  In a production object it would either be
 * absent or under a DEBUG #define.
 *
 **********************************************************************/
M_TChat::M_TChat()
: hAttn(NULL),
  hNorm(NULL),
  requestThread(NULL),
  instances(0)
{
    somPrintf("Invoking %s\n", __FUNCTION__);
}
/**********************************************************************
 *                    Destructor for M_TChat Class
 *
 * We just print a message here so we can watch the class terminate.
 * This more clearly reveals how SOM operates internally and makes
 * debugging a bit easier.  In a production object it would either be
 * absent or under a DEBUG #define.
 *
 **********************************************************************/
M_TChat::~M_TChat()
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    DosSleep(300); // Allow .3 Seconds for Any Pending Requests to Finish
    somPrintf("Stopping Background Thread\n");
    delete requestThread; // Terminate the Background Service Thread
}
/**********************************************************************
 *        Keep Track of TChat Instances In This Process Space
 *
 * There are many possible reasons for tracking the count of objects
 * created by M_TChat.  Our reason is to be able to create a background
 * thread for issuing DSOM requests when the first TChat object is
 * created (new object) or made active (WPS starting up with existing
 * objects).
 *
 * Theoretically we could do this in the constructor for the M_TChat
 * class but due to the way that DSOM creates proxies, that way won't
 * work quite right.  It would result in a background thread being
 * created both in the desktop process (which we want) and in the
 * chatroom server (which we don't want).
 *
 * This is because when we pass a pointer to our TChat object to the
 * Chatroom object in a remote process, DSOM creates a proxy in that
 * remote process to represent our object.  To create that proxy, it
 * must instantiate the M_TChat class, since it is responsible for
 * creating instances of TChat.  However, since it is creating proxies
 * of TChat and not TChat itself, all of its capabilities aren't used.
 * This means that the M_TChat ctor/dtor are invoked but no instances
 * TChat are created.  For this reason, we tie the creation of the
 * background thread to the creation of the first TChat object.
 *
 * This was not easy to figure out and appears undocumented.
 *
 **********************************************************************/
void
M_TChat::adjCount(int delta)
{
    if (instances == 0 && delta == 1) // First Instance Appearing
        requestThread = new BackThread(this);

    instances += delta;
}
/**********************************************************************
 *           Upon-Awakening Class Object Data Initialization
 *
 * This method is called when the class is first created -OR- is
 * awakened, as in someone opened a folder containing it or WPS just
 * booted up with existing instances already on the desktop.
 *
 **********************************************************************/
void
M_TChat::wpclsInitData()
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    Parent::wpclsInitData();  // Invoke Parent's Method First on Init

    // Initialize Any Class-Wide Data Here, Like Shared Icons
    hNorm = WinLoadPointer(HWND_DESKTOP, MyModuleHandle, ID_ICON);
    hAttn = WinLoadPointer(HWND_DESKTOP, MyModuleHandle, ID_ICON2);
}
/**********************************************************************
 *             Going-Dormant Class-Wide Data Destruction
 *
 * This method is called when the class is made dormant.  This
 * occurs when the *last* instance of the class is made dormant.
 *
 **********************************************************************/
void
M_TChat::wpclsUnInitData()
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    // ********************************
    // Destroy Any Class-Wide Data Here
    // ********************************
    if (hAttn)
        WinDestroyPointer(hAttn);

    if (hNorm)
        WinDestroyPointer(hNorm);

    Parent::wpclsUnInitData();  // Invoke Parent's Method Last on UnInit
}
/**********************************************************************
 *                Return PM Handle of Our Normal Icon
 **********************************************************************/
HPOINTER
M_TChat::queryNormalIcon()
{
    return hNorm; // Handle of Icon to Use When Chat Phone is NOT Ringing
}
/**********************************************************************
 *                Return PM Handle of Our Attn Icon
 **********************************************************************/
HPOINTER
M_TChat::queryAttnIcon()
{
    return hAttn; // Handle of Icon to Use When Chat Phone is Ringing
}
/**********************************************************************
 *                 Default Title for TChat Objects
 *
 * This title is used as a default for instances of this class as well
 * as to describe the class in facilities such as Find, Include,
 * Details and Sort.
 *
 * Note that this is the title of 'TChat', the class of which this is
 * a metaclass, so a title of 'M_TChat' is not appropriate.
 *
 **********************************************************************/
PSZ
M_TChat::wpclsQueryTitle()
{
    return "TChat Session";
}
/**********************************************************************
 *               Default Open View for TChat Objects
 **********************************************************************/
ULONG
M_TChat::wpclsQueryDefaultView()
{
    return TChat::VIEWID_CHATVIEW;
}
/**********************************************************************
 *            Define What Can Be Done to the TChat Objects
 *
 * This method is used to determine the default set of possible actions
 * permitted on instances, such as COPY, PRINT, DELETE, MOVE, RENAME
 * and such.  Each instance can also have its own set of permissions,
 * but if not, this is where you define the global set.
 *
 * One of the actions is whether a user can MANUALLY create a template
 * and since we don't want templates, we deny the action here.  This
 * causes the 'Template []' button to NOT be displayed on the GENERAL
 * page of the Settings notebook.
 *
 * NOTE: There is some confusion over the CLSSTYLE_NEVERTEMPLATE flag
 *       versus the CLSSTYLE_DONTTEMPLATE.  The former appears
 *       obsolete due to a conflict in bit assignment.
 *
 **********************************************************************/
ULONG
M_TChat::wpclsQueryStyle()
{
    return (Parent::wpclsQueryStyle() & ~CLSSTYLE_DONTTEMPLATE);
}
/**********************************************************************
 *           Prevent the Creation of a Default WPS Template
 *
 * This method is invoked when the class is first registered via
 * WinRegisterObjectClass().  The parameter is a pointer to the system
 * template folder and this method is given an opportunity to create
 * template object(s) itself, say with custom defaults and such.  This
 * method returns a boolean indicating whether it did so or whether it
 * wants the Workplace Shell to create one instead.
 *
 * We lie to WPS here by telling it we did when we didn't, in order to
 * avoid having a template.  This is desirable in a development setup
 * since having a template locks the DLL and prevents its replacement.
 *
 **********************************************************************/
BOOL
M_TChat::wpclsCreateDefaultTemplates(WPObject *templateFolder)
{
    return TRUE; // Do Not Create a Default Template in the Templates Folder
}
/**********************************************************************
 *                 Default Icon for TChat Objects
 *
 * This method indicates the icon to be used for instances of the TChat
 * class on the desktop.  It can represent this as the name of a .ICO
 * file, a module name and resource ID, or as a block of binary data.
 *
 * We use an icon compiled into the class DLL as a resource.
 *
 **********************************************************************/
ULONG
M_TChat::wpclsQueryIconData(ICONINFO *pIconInfo)
{
    if (pIconInfo) {
        pIconInfo->fFormat = ICON_RESOURCE;  // Its a Module/Resource ID
        pIconInfo->hmod    = MyModuleHandle; // Module Handle of Our DLL
        pIconInfo->resid   = ID_ICON;        // Resource ID of Our Icon
    }
    return sizeof(ICONINFO);
}
/**********************************************************************
 *    Central Function to Queue a Request on the Background Thread
 *
 * This method is invoked by the various instances of TChat in order to
 * queue a request to the single instance of the background thread.
 * Since there is one instance of M_TChat for all TChat objects (in a
 * single process), it is only natural for M_TChat to provide this
 * service.
 *
 **********************************************************************/
void
M_TChat::queueRequest(TRequest *req)
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    requestThread->queueRequest(req);
}
/**********************************************************************
 *
 **********************************************************************/

