// File: tchat.cpp

#include <somd.hh>

#define INCL_WIN
#define INCL_DOS
#include <os2.h>

#include "ChatView.hpp"  // Declaration of OpenClass-Derived ChatView Class

#include "TChat.hh"      // Our SOM Class Declaration
#include "TChat.h"       // Resource ID for Our Icon
#include "Useful.hpp"    // Some Useful Items for DSOM Work
#include "Chatroom.hh"   // Chatroom DSOM Class
#include "BkThread.hpp"  // Background Thread Class

/**********************************************************************
 *          TChat Class - A GUI Element for Chatting via DSOM
 *
 * A DSOM class is usually packaged as a DLL and this one is no different.
 * We define the _DLL_InitTerm() function that OS/2 invokes upon the
 * loading and unloading of each DLL, in order to monitor this activity.
 *
 * This DLL implements the TChat class which is a desktop object.  It
 * provides a desktop view into a chatroom where a user types and all
 * the other participants see what he types.
 *
 **********************************************************************/

/*******************************
 * Global Constants and Macros
 *******************************/
#define WPSITEM_USERNAME  1
#define WPSITEM_ROOMNAME  2

/*******************************
 *    Structures and Types
 *******************************/

/*******************************
 *     Function Prototypes
 *******************************/
void ShowMsg(const char *title, const char *text);

extern "C" int   _CRT_init(void); // C Runtime Init Function (0 = Success, -1 = Failure)

#ifdef __cplusplus
extern "C" void  __ctordtorInit(void); // C++ Constructor/Destructor Init
extern "C" void  __ctordtorTerm(void); // C++ Constructor/Destructor Init
#endif

#if defined(__IMPORTLIB__)
/* A clean up routine registered with DosExitList must be used if runtime */
/* calls are required and the runtime is dynamically linked.  This will   */
/* guarantee that this clean up routine is run before the library DLL is  */
/* terminated.                                                            */
static void APIENTRY _DLL_ExitHandler(ULONG exitReason);

#else
extern "C" void _CRT_term(void); // Runtime Terminator - Static Link Only
#endif

/*******************************
 *      Global Variables
 *******************************/
char *_dll_name = "TChat.dll";    // Convenient Ref to Our DLL Name

HMODULE          MyModuleHandle;  // Module handle for our .DLL

/**********************************************************************
 *                           _DLL_InitTerm
 *
 * This function gets called by the operating system loader when it
 * loads and frees this DLL for each process that accesses this DLL.
 * However, it only gets called the first time the DLL is loaded and
 * the last time it is freed for a particular process, not per thread.
 *
 **********************************************************************/
ULONG _System
_DLL_InitTerm(ULONG hModule, ULONG ulFlag)
{
    somPrintf("%s %s\n", ulFlag ? "Unloading" : "Loading", _dll_name);

    if (ulFlag) {        // DLL is Being Unloaded from This Process

        #if !defined(__IMPORTLIB__)
            _ctordtorTerm();
            _CRT_term(); // Only Called if Runtime is Statically Linked-In
        #else
            DosExitList(EXLST_REMOVE, _DLL_ExitHandler);
        #endif

        return TRUE; // Return Non-Zero Value to Indicate Success

    } else {           // DLL is Being Loaded into New Process
        MyModuleHandle = hModule;

        if (_CRT_init() == -1) // Initialize the C Runtime DLL
            return FALSE; // ERROR - Unable to Initialize the C Runtime

        else {
            #if defined(__IMPORTLIB__)
                // Use a DosExitList routine if C Runtime is Dynamically Linked
                DosExitList(0x0000FF00 | EXLST_ADD, _DLL_ExitHandler);
            #endif

            __ctordtorInit();
            return TRUE;
        }
        return TRUE; // Return Non-Zero Value to Indicate Success
    }
}
/**********************************************************************
 *                       _DLL_ExitHandler
 *
 * This Dos Exit Handler gets called when the client process terminates,
 * in order to let the DLL perform any necessary cleanup.  It gets
 * called *BEFORE* _DLL_InitTerm().
 *
 * In our case, it is useful only to report the ending of the WPS
 * process, which should never occur under normal circumstances.
 *
 **********************************************************************/
#if defined(__IMPORTLIB__)
static void APIENTRY
_DLL_ExitHandler(ULONG exitReason)
{
    somPrintf("_DLL_ExitHandler of %s Called\n", _dll_name);

    switch (exitReason) {
    case TC_EXIT:        // Normal Process Exit
    case TC_HARDERROR:   // Hard Error Occurred
    case TC_TRAP:        // Trap in a 16-Bit Child Process
    case TC_KILLPROCESS: // Unintercepted DosKillProcess Call
    case TC_EXCEPTION:   // Exception in a 32-Bit Child Process
    default:
        break;
    }

    // NOTE: Destructors of Local C++ Objects NOT Called Due to Early-Exit
    DosExitList(EXLST_EXIT, 0); // Does Not Return - No Local DTors Called!!
}
#endif
/**********************************************************************
 * Function Called when This DLL is Dynamically Loaded via somFindClass
 *
 * This function does not appear to always be called, depending upon
 * how the SOM class is made available.  It seems to depend upon
 * whether the DLL is statically or dynamically linked into the client
 * and a host of other factors.  It also matters whether you define
 * the 'SOMInitModule' entry point in your .DEF file.
 *
 * It doesn't seem to be very important as the DTS compiler seems to
 * automatically add code to initialize classes when first seen.  This
 * is a poorly documented corner of SOM.
 *
 **********************************************************************/
SOMEXTERN void SOMLINK
SOMInitModule(long majorVersion, long minorVersion)
{
    somPrintf("SOMInitModule() of %s Called\n", _dll_name);

    TChatNewClass(0, 0);   // Register Our TChat Class with SOM
    M_TChatNewClass(0, 0); // Register Our M_TChat Class with SOM
}
/**********************************************************************
 *                   Construct a TChat Object
 *
 * This method is only the first step in the creation of a WPS object
 * which ends with the invocation of the wpObjectReady() method later.
 * At the point this ctor executes, the persistent WPS image is NOT
 * ready so there is little you can do at this point.
 *
 **********************************************************************/
TChat::TChat()
: myClass(NULL),
  connectState(DISCONNECTED),
  roomObj(NULL)
{
    somPrintf("Invoking %s\n", __FUNCTION__);
}
/**********************************************************************
 *                     WPS Object Going Ready
 *
 * This method is invoked by WPS as the last stage in the WPS object
 * creation sequence.
 *
 **********************************************************************/
void
TChat::wpObjectReady(ULONG ulCode, WPObject* refObject)
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    Parent::wpObjectReady(ulCode, refObject); // Invoke Parent's Method

    SOMLocalEnv   env;            // We Need a Locally-Scoped SOM Environment
    Environment  *__SOMEnv = env.base(); // And Set the DTS-Magic Variable

    // For Ease of Reference, Stash a Ptr to My Class Object
    myClass = (M_TChat *)__ClassObject;

    // We lock the object so it won't ever go dormant.  WPS objects go
    // dormant when the folder they are in is closed.  Since we want to
    // listen for chatroom activity and change the icon image as needed,
    // we must never go dormant.
    wpLockObject(); // Prevent Dormancy Since We Listen for Chat Activity

    // Using adjCount() we inform our metaclass M_TChat that an object
    // has been created so we can start the per-class background thread
    // upon the appearance of the first instance.  Although the metaclass
    // invoked our ctor and could count instances then, a WPS object is
    // not completely created until the wpObjectReady() call, which our
    // metaclass does not see.
    myClass->adjCount(1); // Report Creation of a TChat Object
    myClass->queueRequest( new TJoinRequest(this, roomName(), userName()) );
}
/**********************************************************************
 *      Handle Initial Creation (w/Defaults) of a TChat Instance
 *
 * This method is invoked only once on a WPS object.  We use it to set
 * default values into a new object.  The parent method invokes our
 * wpSetup() method so we must set any defaults prior to that, in case
 * our setup string provides non-default values.
 *
 **********************************************************************/
BOOL
TChat::wpSetupOnce(PSZ pszSetupString)
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    setUserName("NoName");   // Defaults to Use if Not Specified on SETUP
    setRoomName("DobbsRoom");

    BOOL bStatus = Parent::wpSetupOnce(pszSetupString);

    wpSaveDeferred(); // Save Persistent Object to Disk Eventually
    return bStatus;
}
/**********************************************************************
 *             Parse Our WPS Setup String for New Keys
 *
 * This method gets invoked when our object is initially created via
 * the SysCreateObject() function (in our REXX script in this case)
 * -AND- when someone executes the WinSetObjectData() function to
 * alter our object attributes.
 *
 * It does NOT get invoked when our object goes dormant/active.
 *
 * We support two additional setup keys; "ROOM" and "ALIAS".
 *
 * REXX Example:
 *
 *     rc = SysCreateObject('TChat', 'Chat One', '<WP_DESKTOP>',,
 *             'OBJECTID=<TCHAT3>;ROOM=DobbsRoom;ALIAS=Laurel',,
 *             'REPLACE')
 *
 * Because of the possibility of WinSetObjectData(), we must allow
 * an existing connection to a chatroom to be yanked away and assigned
 * to another room in this routine.
 *
 **********************************************************************/
BOOL
TChat::wpSetup(PSZ pszSetupString)
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    ULONG cbValue;
    char  newRoom[80], newUser[80];

    BOOL bStatus = Parent::wpSetup(pszSetupString); // Invoke Parent First

    BOOL  bSaveObject = FALSE;
    if (bStatus && pszSetupString && *pszSetupString) {
        if (wpScanSetupString(pszSetupString, "ROOM", NULL, &cbValue)) {
            if (cbValue > sizeof(newRoom) - 1)
                cbValue = sizeof(newRoom) - 1;

            bStatus = wpScanSetupString(pszSetupString, "ROOM",
                                        newRoom, &cbValue);
            if (bStatus) {
                bSaveObject = TRUE;
                somPrintf("ROOM Changed Value in wpSetup\n");
            }
        }

        if (wpScanSetupString(pszSetupString, "ALIAS", NULL, &cbValue)) {
            if (cbValue > sizeof(newUser) - 1)
                cbValue = sizeof(newUser) - 1;

            bStatus = wpScanSetupString(pszSetupString, "ALIAS",
                                        newUser, &cbValue);
            if (bStatus) {
                bSaveObject = TRUE;
                somPrintf("ALIAS Changed Value in wpSetup\n");
            }
        }

        if (bSaveObject)                 // If ROOM or ALIAS Changed,
            reconnect(newRoom, newUser); // Then Reconnect to New Choices
    }
    return bStatus;
}
/**********************************************************************
 *         An Object is Leaving Existence, Tell Our Metaclass
 *
 * This method handles a TChat instance going away by informing our
 * metaclass of this fact.  It uses this information to track the count
 * of TChat instances and to stop the background thread when there are
 * no more.
 *
 * Unfortunately there is no opposite of the wpObjectReady() method
 * and the hierarchy of WPS method calls during the destruction of an
 * object is not as well documented as during its creation.
 *
 **********************************************************************/
TChat::~TChat()
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    myClass->adjCount(-1);
}
/**********************************************************************
 *       User Wishes to Delete a TChat Object from the Desktop
 *
 * This method handles the request to delete a TChat object.  When
 * deleting one we must quit any current chatroom we are in, and this
 * takes time (depending upon the LAN).  Since we don't want to freeze
 * the desktop during this time, we must perform the deletion on the
 * background thread.
 *
 * Hence this method queues a quit request to the background thread and,
 * to make it -look- like we deleted the object immediately, hides the
 * TChat object from the user.  When the background thread finishes
 * leaving the chatroom and is ready to delete the (now hidden) TChat
 * object, it invokes our reallyDelete() method which quietly does it.
 *
 * This does leave a window of trouble if the user is rapidly deleting
 * and recreating TChat objects from a REXX script but in practice,
 * this is rare.  Deletion of remote objects just takes time.
 *
 **********************************************************************/
ULONG
TChat::wpDelete(ULONG fConfirmations)
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    SOMLocalEnv   env;            // We Need a Locally-Scoped SOM Environment
    Environment  *__SOMEnv = env.base(); // And Set the DTS-Magic Variable

    wpModifyStyle(OBJSTYLE_NOTVISIBLE, OBJSTYLE_NOTVISIBLE);

    myClass->queueRequest( new TQuitRequest(this) );
    return NO_DELETE;
}
/**********************************************************************
 *         Handle Request to Really Delete the TChat Object
 *
 * This method is invoked by the background thread to actually perform
 * the deletion of the TChat object once its chatroom link is broken.
 *
 **********************************************************************/
ULONG
TChat::reallyDelete()
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    return Parent::wpDelete(NULL);
}
/**********************************************************************
 *           Handle Notifications of Changes in our State
 *
 * This method is called by various components when changes in our
 * internal remote-object link state occur.  It sets our instance
 * variable 'connectState' and then notifies each open view of ours
 * of the change, in case it matters.  In our case, the chat view
 * indicates state changes in the status line on the bottom of the view.
 *
 * If no views are open, they must query our state when they do open
 * in order to remain in-sync.
 *
 **********************************************************************/
void
TChat::setState(short newstate)
{
    ChatView   *aview;
    int         numViews = 0, myViews = 0;

    connectState = newstate; // Change Our Internal Instance Variable

    PVIEWITEM pViewItem = wpFindViewItem(VIEW_ANY, NULL);
    while (pViewItem) { // Iterate Over All Object Views on the Desktop
        if (pViewItem->view == VIEWID_CHATVIEW) { // If a View of Us,
            aview = (ChatView *)IWindow::windowWithHandle(pViewItem->handle);
            if (aview)
                aview->setState(newstate); // Notify a View

            myViews++;
        }
        pViewItem = wpFindViewItem(VIEW_ANY, pViewItem);
        numViews++;
    }
}
/**********************************************************************
 *         Receive Some Text and Display in the Chat Window
 *
 * This method receives a string from the chatroom and passes it
 * to each open chat view for display.
 *
 * This SOM method is defined as 'oneway' as an example of how.  This
 * means the invoker of this method doesn't care about results or
 * whether the method actually works.  Think datagram where the caller
 * sends a packet without error retry.  This also means that this
 * method must NEVER be changed to return a value in any way, even via
 * parameters.
 *
 **********************************************************************/
void
TChat::displayText(const char *str)
{
    ChatView   *aview;
    int         numViews = 0, myViews = 0;

    PVIEWITEM pViewItem = wpFindViewItem(VIEW_ANY, NULL);
    while (pViewItem) { // Iterate Over All Object Views on the Desktop
        if (pViewItem->view == VIEWID_CHATVIEW) { // If a View of Us,
            aview = (ChatView *)IWindow::windowWithHandle(pViewItem->handle);
            if (aview)
                aview->displayText(str); // Stuff Text into a View

            myViews++;
        }
        pViewItem = wpFindViewItem(VIEW_ANY, pViewItem);
        numViews++;
    }

    // Extra Feature: If we are receiving text from a chatroom and our
    // icon is closed, we change our icon image to alert the user.  We
    // could also have played sound effects.  Any text received while
    // our icon is closed is lost, although the code could be changed
    // to buffer it and squirt it to the view once it opened.

    if (numViews == 0) { // If No Views Open, Signal the User of Attention
        // Tell User Someone is Trying to Chat with Him
        wpSetIcon( myClass->queryAttnIcon() );
    }
}
/**********************************************************************
 *             Transmit a String to the Chatroom Object
 *
 * This method invokes a method on a (possibly) remote server, giving
 * it the string.  The Chatroom can then pass it out to the members
 * of the chat session.
 *
 **********************************************************************/
boolean
TChat::emitText(const char *str)
{
    somPrintf("Invoking %s with '%s' 0x%02lX\n", __FUNCTION__, str, *str);

    // Queue a Request on the Background Thread to Emit Some Text
    myClass->queueRequest( new TPostRequest(this, str) );
    return TRUE;
}
/**********************************************************************
 *            Handle a User's Request to Open a New View
 *
 **********************************************************************/
HWND
TChat::wpOpen(HWND hwndCnr, ULONG ulView, ULONG param)
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    switch (ulView) {
    case VIEWID_CHATVIEW: // Attempting to Open Our ChatView?

        // Opening a View, Reset Attention Icon to Normal
        wpSetIcon( myClass->queryNormalIcon() );
        return (new ChatView(this))->handle();

    default: // Not Our View, Just Call Our Parent to Handle the Others
        return Parent::wpOpen(hwndCnr, ulView, param);
    }
}
/**********************************************************************
 *             Handle Selection of Our Custom Menu Item
 *
 **********************************************************************/
BOOL
TChat::wpMenuItemSelected(HWND hwndFrame, ULONG ulMenuId)
{
    switch (ulMenuId) {
    case MENUID_CHATVIEW:
        wpViewObject(NULLHANDLE, ulMenuId, 0);
        break;

    default:
        break;
    }

    return Parent::wpMenuItemSelected(hwndFrame, ulMenuId);
}
/**********************************************************************
 *             Save Our Persistent Attributes to Disk
 *
 **********************************************************************/
BOOL
TChat::wpSaveState()
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    wpSaveString("TChat", WPSITEM_USERNAME, (PSZ)userName());
    wpSaveString("TChat", WPSITEM_ROOMNAME, (PSZ)roomName());

    return Parent::wpSaveState(); // Invoke Parent's Method Last on Save
}
/**********************************************************************
 *            Restore Our Persistent Attributes from Disk
 *
 **********************************************************************/
BOOL
TChat::wpRestoreState(ULONG ulReserved)
{
    somPrintf("Invoking %s\n", __FUNCTION__);

    ULONG buflen;
    BOOL bStatus;

    char newuser[80], newroom[80];

    // Invoke Parent's Method First on Restore
    if (Parent::wpRestoreState(ulReserved)) {
        if (wpRestoreString("TChat", WPSITEM_USERNAME, NULL, &buflen)) {
            if (buflen > sizeof(newuser) - 1)
                buflen = sizeof(newuser) - 1;

            if (wpRestoreString("TChat", WPSITEM_USERNAME, newuser, &buflen))
                setUserName(newuser);
            else
                return FALSE; // ERROR - Unable to Restore UserName Field
        } else
            return FALSE; // ERROR - Unable to Restore UserName Field

        if (wpRestoreString("TChat", WPSITEM_ROOMNAME, NULL, &buflen)) {
            if (buflen > sizeof(newroom) - 1)
                buflen = sizeof(newroom) - 1;

            if (wpRestoreString("TChat", WPSITEM_ROOMNAME, newroom, &buflen))
                setRoomName(newroom);
            else
                return FALSE; // ERROR - Unable to Restore RoomName Field
        } else
            return FALSE; // ERROR - Unable to Restore RoomName Field

        return TRUE;
    } else
        return FALSE; // ERROR - Unable to Restore Parent's State
}
/**********************************************************************
 *                 Modify Our UserName Attribute
 *
 **********************************************************************/
void
TChat::setUserName(const char *name)
{
    strncpy(myUserName, name, sizeof(myUserName));
    myUserName[ sizeof(myUserName)-1 ] = '\0';
    wpSaveDeferred(); // Attribute Changed - Save to Disk Later
}
/**********************************************************************
 *                 Modify Our RoomName Attribute
 *
 **********************************************************************/
void
TChat::setRoomName(const char *name)
{
    strncpy(myRoomName, name, sizeof(myRoomName));
    myRoomName[ sizeof(myRoomName)-1 ] = '\0';
    wpSaveDeferred(); // Attribute Changed - Save to Disk Later
}
/**********************************************************************
 *       Handle Need for Reconnect Due to Change in Attributes
 *
 **********************************************************************/
void
TChat::reconnect(const char *room, const char *user)
{
    if (wpIsObjectInitialized()) { // If After Object is Ready/Linked,
        myClass->queueRequest( new TJoinRequest(this, room, user) );

    } else {                       // Otherwise Just Set Our Attributes
        setUserName(user);
        setRoomName(room);
    }
}
/**********************************************************************
 *              Add Any Custom Items to Our Popup Menu
 *
 * This method allows an instance to add custom items to its popup
 * menu.  Here we add our CHATVIEW item to the set of OPEN choices.
 *
 **********************************************************************/
BOOL
TChat::wpModifyPopupMenu(HWND hwndMenu, HWND hwndCnr, ULONG iPosition)
{
    wpInsertPopupMenuItems(hwndMenu,  // Handle of the Popup Menu
                 0,                   // Relative Position to Insert At
                 MyModuleHandle,      // Handle of DLL Holding Resources
                 MENUID_CHATVIEWMENU, // ID of Menu Item to Insert
                 WPMENUID_OPEN);      // ID of Submenu to Put It On

    return Parent::wpModifyPopupMenu(hwndMenu, hwndCnr, iPosition);
}
/**********************************************************************
 *                 A Simple Popup [OK] Dialog Box
 *
 * This function is a simple popup box I found useful during debugging
 * and to handle a few catastrophic error conditions.
 *
 **********************************************************************/
void
ShowMsg(const char *title, const char *text)
{
    WinMessageBox(HWND_DESKTOP, HWND_DESKTOP,
                  (PSZ)text, (PSZ)title, 20,
                  MB_OK | MB_INFORMATION | MB_MOVEABLE);
}
/**********************************************************************
 *
 **********************************************************************/

