#! /usr/bin/perl
#=============================================================================
#
#   htmlpp.pl   HTML pre-processor - stage 1
#
#   Written:    96/03/27   Pieter Hintjens <ph@imatix.com>
#   Revised:    96/03/31
#
#   Copyright (c) 1996 Pieter Hintjens.  May be freely distributed.
#=============================================================================

require 'htmlpp1.d';                    #   Include dialog interpreter


##########################   INITIALISE THE PROGRAM   #########################

sub initialise_the_program
{
    print "htmlpp 1.0 - by Pieter Hintjens\n";

    if ($#ARGV == 0) {                  #   Exactly 1 argument in @ARGV[0]?
        $main_document = @ARGV [0];
        $the_next_event = $ok_event;
    } else {
        print "syntax: htmlpp <filename>\n";
        $the_next_event = $error_event;
    }
}


#########################   INITIALISE PROGRAM DATA   #########################

sub initialise_program_data
{
    #   These are the preprocessor keywords that we recognise
    $keyword {"define"}  = $define_event;
    $keyword {"include"} = $include_event;
}


############################   OPEN MAIN DOCUMENT   ###########################

sub open_main_document
{
    $document = $main_document;
    &open_the_document;
}


############################   OPEN THE DOCUMENT   ############################

sub open_the_document
{
    #   We use an indirect filehandle, whose name is the document name.
    #   To read from the file, we use <$document>
    if (open ($document, $document)) {
        $file_is_open {$document} = 1;  #   Keep track of open documents
    } else {
        print "htmlpp E: ($document $.) can't open $document: $!";
        &raise_exception ($exception_event);
    }
}


##########################   GET NEXT DOCUMENT LINE   #########################

sub get_next_document_line
{
    if ($_ = <$document>) {             #   Get next line of input
        chop;                           #   Remove trailing newline
        if (/^$/) {                     #   Blank lines
            $the_next_event = $blank_line_event;
        } elsif (/^\.-/) {              #   Comments
            $the_next_event = $comment_event;
        } elsif (/^\./) {               #   Line starts with a dot
            /^\.(\w+)/;                 #   Get word after dot
            if (defined ($keyword {$1})) {
                $the_next_event = $keyword {$1};
            } else {
                &syntax_error;
            }
        } else {
            $the_next_event = $body_text_event;
        }
    } else {
        $the_next_event = $finished_event;
    }
}

sub syntax_error {
    print "$_\n";
    print "htmlpp E: ($document $.) syntax error\n";
    &raise_exception ($exception_event);
}


#########################   STORE SYMBOL DEFINITION   #########################

sub store_symbol_definition
{
    #   Symbol name can consist of letters, digits, -._
    #   We re-parse the line to extract the symbol name and value:

    if (/^\.\w+\s+([A-Za-z0-9-\._]+)\s+(.*)/) {
        #   Stick name and value into associative array '$symbols'
        $symbols {$1} = $2;
    } else {
        &syntax_error;
    }
}


##########################   EXPAND SYMBOLS IN LINE   #########################

sub expand_symbols_in_line
{
    #   Expands symbols in $_ variable
    #
    #   Repeatedly expand symbols like this:
    #   $(xxx) - value of variable
    #
    #   Note that the entire symbol must be on one line; if the symbol or
    #   its label is broken over two lines it won't be expanded.

    for (;;) {
        if (/\$\(([A-Za-z0-9-_\.]+)\)/) {
            $_ = $`.&valueof ($1).$';
        } else {
            last;
        }
    }
}

#   Subroutine returns the value of the specified symbol; it issues a
#   warning message and returns 'UNDEF' if the symbol is not defined.
#
sub valueof {
    local ($symbol_name) = "@_";        #   Argument is symbol name

    defined ($symbols {$symbol_name}) && return $symbols {$symbol_name};
    print "$_\n";
    print "htmlpp E: ($document $.) undefined symbol \"$symbol_name\"\n";
    $symbols {$symbol_name} = "UNDEF";
    return $symbols {$symbol_name};
}


##########################   TAKE INCLUDE FILE NAME   #########################

sub take_include_file_name
{
    #   Get filename after .include
    if (/^\.\w+\s+(\S+)/) {
        if ($file_is_open {$1}) {
            print "$_\n";
            print "htmlpp E: ($document $.) $1 is already open";
            &raise_exception ($exception_event);
        };
        #   Save current document name and switch to new document
        push (@document_stack, $document);
        $document = $1;
    } else {
        &syntax_error;
    }
}


############################   CLOSE THE DOCUMENT   ###########################

sub close_the_document
{
    close ($document);
    undef $file_is_open {$document};
}


########################   UNSTACK PREVIOUS DOCUMENT   ########################

sub unstack_previous_document
{
    $document = pop (@document_stack);
    $the_next_event = $document eq ""? $finished_event: $ok_event;
}


############################   GET EXTERNAL EVENT   ###########################

sub get_external_event
{
}


##########################   TERMINATE THE PROGRAM    #########################

sub terminate_the_program
{
    $the_next_event = $terminate_event;
}
