/**********************************************************
Author:     Dale Roberts
Date:       11/19/95
Program:    DMAREAD.EXE
Compile:    cl dmaread.c

Purpose:

  Demonstrate use of USERDMA device driver.  Use a large
buffer to allow saving of samples to disk.  Use a simple
double buffering scheme.  When the buffer is half filled,
write out the filled part to disk, then wait for the next
half, and so on.  Demonstrates use of the
IOCTL_USERDMA_CUR_OFFSET call.
  The data is saved in raw, binary format.  The samples
are short integers, and each 16 samples represents a
complete scan of 16 channels.
**********************************************************/
#define WIN32

#include <stdio.h>
#include <windows.h>
#include <winioctl.h>
#include <string.h>
#include "userdma.h"

#define BUFSIZE_SAMPS   1000

int saved_low_flag,     // TRUE if low/high half of buffer
    saved_high_flag;    //  has been written out to disk.
int saving_flag;        // TRUE if we should be saving the
                        //  data now.
int save_from_index;    // lowest sample to write out to
                        //  the file
short *Buf;             // pointer to sample buffer.
int BufSize;            // size of buffer in 16 bit words
FILE *outfile;          // Output file for data.
HANDLE hUserDMA;        // handle to USERDMA driver.
USERDMA_INFO dmainfo;   // DMA information structure.

/**********************************************************
  Call the device driver to get the current DMA pointer.
**********************************************************/
int GetCurIndex(void)
{
    int nchar;
    DeviceIoControl(hUserDMA, IOCTL_USERDMA_CUR_OFFSET,
                    &dmainfo, sizeof(USERDMA_INFO),
                    &dmainfo, sizeof(USERDMA_INFO),
                    &nchar, 0);
    return dmainfo.CurDMAWord;
}

/**********************************************************
  If user requests that we start saving data, set up all
of our data structures and go.
**********************************************************/
void StartSaving(void)
{
    save_from_index = GetCurIndex();
    if(save_from_index < BufSize/2) {
        saved_low_flag = 0;
        saved_high_flag = 1;
    } else {
        saved_low_flag = 1;
        saved_high_flag = 0;
    }
}

/**********************************************************
  Write the Buf[] entries from From to To inclusive.  If
From > To, then wrap around at the end of the buffer.
This takes an index into the Buf[] array, which is an
array of SHORT integers (2 bytes).  So the counts must be
multiplied by 2.
**********************************************************/
void WriteBuf(int From, int To)
{
    if(From < To)
        fwrite(&Buf[From], (To-From+1)*2, 1, outfile);
    else {
        fwrite(&Buf[From], (BufSize-From)*2, 1, outfile);
        fwrite(&Buf[0], (To+1)*2, 1, outfile);
    }
}

/**********************************************************
  This routine is called periodically to see if we need
to write out any data to the file.  We write out half a
buffer at a time.
**********************************************************/
void CheckSave(void)
{
    int CurIndex;

    // Update dmainfo.CurDMAWord, and get current index
    CurIndex = GetCurIndex();

    if(!saving_flag)
        return;
    if(saved_low_flag == 0 && CurIndex >= BufSize/2) {
        saved_low_flag = 1;
        saved_high_flag = 0;
        WriteBuf(save_from_index, BufSize/2-1);
        save_from_index = BufSize/2;
    } else if(saved_high_flag == 0 &&
                                CurIndex < BufSize/2) {
        saved_high_flag = 1;
        saved_low_flag = 0;
        WriteBuf(save_from_index, BufSize-1);
        save_from_index = 0;
    }
}

/**********************************************************
  Write out the last data to the file.
**********************************************************/
void StopSaving(void)
{
    WriteBuf(save_from_index, GetCurIndex());
}

/**********************************************************
  Open file and start DMA.
**********************************************************/
main(int argc, char *argv[])
{
    int numread, i, nchar;
    char FileName[50];
    char ch;

    if(argc > 1)
        strncpy(FileName, argv[1], sizeof(FileName));
    else
        strcpy(FileName, "nul");

    // Open the output data file.
    outfile = fopen(FileName, "wb");
    if(!outfile) {
        printf("Could not open file '%s' for writing\n",
                    FileName);
        exit(-1);
    }

    // Get handle to device driver.
    hUserDMA = CreateFile("\\\\.\\userdma", GENERIC_READ,
                0, NULL, OPEN_EXISTING,
                FILE_ATTRIBUTE_NORMAL, NULL);
    if(hUserDMA == INVALID_HANDLE_VALUE) {
        printf("Couldn't access USERDMA device\n");
        return -1;
    }

    // Set our requested buffer size.
    dmainfo.NumSamps = BUFSIZE_SAMPS;

    // Start up the DMA.
    if(!DeviceIoControl(hUserDMA,
                            IOCTL_USERDMA_START,
                            &dmainfo, sizeof(dmainfo),
                            &dmainfo, sizeof(dmainfo),
                            &nchar, 0)) {
        printf("Error doing IOCTL\n");
        goto endit;
    }

    // If another process has already started DMA with
    // a small buffer, quit.
    if(dmainfo.NumSamps < BUFSIZE_SAMPS) {
        printf(
            "USERDMA already started with small buffer.\n");
        goto endit;
    }
    BufSize = dmainfo.NumSamps * dmainfo.NumChans;
    Buf = dmainfo.buf;

    printf("Press SPACE to toggle saving; 'q' to quit.\n");
    while(1) {
        if(kbhit()) {
            ch = getch();
            if(ch == ' ') {
                saving_flag ^= 1;
                if(saving_flag)
                    StartSaving();
                else
                    StopSaving();
            } else if(ch == 'q')
                break;
        }
        CheckSave();
        printf("%7s ", saving_flag ? "SAVING" : "");
        for(i=0; i < 8; ++i)
            printf(" %7d", Buf[dmainfo.CurDMAWord+i]);
        printf("\r");

        // don't hog the processor
        Sleep(100);
    }
    printf("\n");

endit:
    // Stop DMA.
    DeviceIoControl(hUserDMA,
                    IOCTL_USERDMA_STOP,
                    &dmainfo, sizeof(dmainfo),
                    &dmainfo, sizeof(dmainfo),
                    &nchar, 0);

    CloseHandle(hUserDMA);
    fclose(outfile);
    return 0;
}
