/**********************************************************
Author:     Dale Roberts
Date:       11/19/95
Program:    for USERDMA.SYS
Compile:    Use DDK BUILD facility

  This code is responsible for setting up and stopping
the National Instruments AT-MIO16X data acquisition board.
The two functions used by the main driver are StartAD(),
and StopAD().
**********************************************************/
#include "userdma.h"

/*
 *  Register defines.
 */
#define NI_BASE         0x220

#define NI_CMD1         (NI_BASE+0x00)
#define NI_CMD2         (NI_BASE+0x02)
#define NI_CMD3         (NI_BASE+0x04)
#define NI_CMD4         (NI_BASE+0x06)
#define NI_STAT1        (NI_BASE+0x18)
#define NI_STAT2        (NI_BASE+0x1a)
#define NI_DAC0         (NI_BASE+0x10)
#define NI_DAC1         (NI_BASE+0x12)
#define NI_DAC_UPD      (NI_BASE+0x18)
#define NI_CONFIGMEM    (NI_BASE+0x08)
#define NI_CONFIGMEM_CLR (NI_BASE+0x1b)
#define NI_CONFIGMEM_LD (NI_BASE+0x1b)
#define NI_DAQ_CLR      (NI_BASE+0x19)
#define NI_DMATCA_CLR   (NI_BASE+0x19)
#define NI_DMATCB_CLR   (NI_BASE+0x09)
#define NI_DMA_CHAN_CLR (NI_BASE+0x0b)
#define NI_TMRREQ_CLR   (NI_BASE+0x1f)
#define NI_ADC_CAL      (NI_BASE+0x1f)
#define NI_RTSI_SHIFT   (NI_BASE+0x0c)
#define NI_RTSI_STROBE  (NI_BASE+0x0e)
#define NI_DAQ_START    (NI_BASE+0x1d)

#define AM9513_DATA     (NI_BASE+0x14)
#define AM9513_CMD      (NI_BASE+0x16)
#define AM9513_STATUS   (NI_BASE+0x16)

#define SRC2_RATE       5000000

/*
 *  Command register bits.
 */
#define NI_CMD1_INTGATE_DIS 0x0400
#define NI_CMD1_RETRIG_DIS  0x0200
#define NI_CMD1_DAQEN       0x0100
#define NI_CMD1_SCANEN      0x0080
#define NI_CMD1_SCN2        0x0040
#define NI_CMD4_EXTRIGDIS   0x0001

/**********************************************************
  Stop the AM9513 timer.  Use the master reset command.
**********************************************************/
void stop_timer(void)
{
    outpw(AM9513_CMD, 0xffff);      // master reset
    outpw(AM9513_CMD, 0xffef);      // 16 bit bus mode
}

/**********************************************************
 Start the AM9513 timer.  Give speed in ticks/sec.

  Counter 3 is used as the sample interval counter.  This
is the time between each channel sample.  Counter 4 is the
sample counter.  We set this to 16, to match the number of
samples in the scan array.  Counter 2 is the scan interval.
It is the time between scanning the whole 16 channels.

  The BIG TRICK here is to program counter 4, which
normally controls the total number of samples to acquire,
to have a permanently low output.  This causes true
"continuous" acquisition forever.  What the book calls
"continuous" mode actually stops at the terminal count
of counter 4.
**********************************************************/
void start_timer(long speed)
{
    outpw(AM9513_CMD, 0xffff);      // master reset
    outpw(AM9513_CMD, 0xffef);      // 16 bit bus mode
    outpw(AM9513_CMD, 0xff17);      // adress mstr mode reg
    outpw(AM9513_DATA, 0xa000);     // 16 bit bus, dec div
    outpw(AM9513_CMD, 0xff02);      // mode/load/hold regs

                                    // program counter 2,
                                    //  scan interval
    outpw(AM9513_DATA, 0x8225);     // SRC2 5MHz, repetitive
                                    //  active low
                                    // set acquisition speed
    outpw(AM9513_DATA, SRC2_RATE / speed);
    outpw(AM9513_DATA, 0x00);
                                    // program counter 3,
                                    //  sample interval
    outpw(AM9513_DATA, 0x8225);     // SRC2 5MHz, repetitive
                                    //  active low
                                    // set speed btwn chans
    outpw(AM9513_DATA, SRC2_RATE / 50000);
    outpw(AM9513_DATA, 0x00);
                                    // counter 4, #samples
    outpw(AM9513_DATA, 0x0000);     // go "forever"
                                    // output always low
    outpw(AM9513_DATA, 0x00);
    outpw(AM9513_DATA, 0x00);

    outpw(AM9513_CMD, 0xff6e);      // load and arm ctrs
                                    //   2, 3 & 4
}

/**********************************************************
  Initialize the National Instruments board to get A/D
using DMA.  Set up to acquire all 16 channels continuously.
**********************************************************/
void init_ni(void)
{
    int i;

    outpw(NI_CMD1, 0);
    outpw(NI_CMD2, 0);
    outpw(NI_CMD3, 0);
    outpw(NI_CMD4, 0);

    inp(NI_CONFIGMEM_CLR);          // clear everything
    inp(NI_DAQ_CLR);
    outp(NI_DMATCA_CLR, 0);
    inp(NI_DMATCB_CLR);
    inp(NI_DMA_CHAN_CLR);
    inp(NI_TMRREQ_CLR);             // clear timer int req

    for(i=0; i < 56; ++i)           // zero the RTSI
                                    //  input shift reg
        outp(NI_RTSI_SHIFT, 0);
    outp(NI_RTSI_STROBE, 0);        // load the zeros
                                    //  into the RTSI

    //  Set up the scan configuration memory.
    inp(NI_CONFIGMEM_CLR);          // clear the chan scan
                                    //  memory
    for(i=0; i < 15; ++i)
        // single ended, bipolar, gain 1
        outpw(NI_CONFIGMEM, (0x9008 | (i << 6)));
    // set last chan bit
    outpw(NI_CONFIGMEM, (1 | 0x900c | (i << 6)));

    outpw(NI_CMD1,
            NI_CMD1_INTGATE_DIS     // disable conversions
            | NI_CMD1_DAQEN
            | NI_CMD1_SCN2
            | NI_CMD1_SCANEN
    );

    // DAC bipolar, set DMA chan
    outpw(NI_CMD2, 0x0c00 + USERDMA_CHANNEL);
    // DMA CH A enable, ADC request enable
    outpw(NI_CMD3, 0x6140);
    // disable external trigger line
    outpw(NI_CMD4, NI_CMD4_EXTRIGDIS);

    inp(NI_DAQ_CLR);                // clear and init ADC
    outp(NI_CONFIGMEM_LD, 0);       // load first chan
                                    //  scan config value
    outpw(NI_CMD1, 0                // clear intgate bit
            | NI_CMD1_DAQEN         //  to enable A/D
            | NI_CMD1_SCN2
            | NI_CMD1_SCANEN
    );
}

/**********************************************************
  Called by main driver to start the DMA.
**********************************************************/
void StartAD(void)
{
    stop_timer();
    init_ni();                      // set up A/D stuff
    start_timer(USERDMA_RATE);      // set up timer
    inp(NI_DAQ_START);              // make it go!
}

/**********************************************************
  Called by main driver to start the DMA.
**********************************************************/
void StopAD(void)
{
    stop_timer();
}
