/**********************************************************
Author:     Dale Roberts
Date:       11/19/95
Program:    USERDMA.SYS
Compile:    Use DDK BUILD facility

Purpose:

Example DMA device driver.  Drives an A/D board.  Uses
autoinitialize DMA to continuously read samples into
a buffer.  Passes a user mode pointer to the DMA buffer
up to the real mode caller, so they can read the data
easily.  Any number of applications can be simultaneously
using the device driver.
**********************************************************/
#include <ntddk.h>
#include <devioctl.h>
#include "userdma.h"

#define DEVICE_NAME_STRING  L"USERDMA"
#define USERDMA_DEVICE_NAME L"\\Device\\" DEVICE_NAME_STRING
#define USERDMA_DOS_NAME    L"\\DosDevices\\" \
                                DEVICE_NAME_STRING

#define USERDMA_BUFFER_SIZE     32768

/**********************************************************
  Device extension.  Holds all info about USERDMA driver.
**********************************************************/
typedef struct {
    PDEVICE_OBJECT  deviceObject;
    int             NumUsers;
    PADAPTER_OBJECT AdapterObject;
    ULONG           MaximumMapRegisters;
    PVOID           BufVirtualAddress;
    ULONG           BufLength;
    ULONG           BytesTransfered;
    PMDL            pmdl;
    PHYSICAL_ADDRESS LogicalAddress;
    PVOID           MapRegisterBase;
    ULONG           DMALength;
    int             NumADChans;
    int             BytesPerFullSample;
} DEVICE_EXTENSION, *PDEVICE_EXTENSION;

/**********************************************************
  Stop DMA (if it isn't already) and release resources.
**********************************************************/
VOID DeviceUnload(IN  PDRIVER_OBJECT  DriverObject)
{
    PDEVICE_EXTENSION ext;
    WCHAR DOSNameBuffer[] = USERDMA_DOS_NAME;
    UNICODE_STRING uniDOSString;
    KIRQL oldIrql;

    ext = (PDEVICE_EXTENSION)
            DriverObject->DeviceObject->DeviceExtension;
    StopAD();
    oldIrql = 255;
    if(KeGetCurrentIrql() < DISPATCH_LEVEL)
        KeRaiseIrql(DISPATCH_LEVEL, &oldIrql);
    if(ext->AdapterObject)
        IoFreeAdapterChannel(ext->AdapterObject);
    if(ext->pmdl)
        IoFreeMdl(ext->pmdl);
    if(ext->BufVirtualAddress)
        HalFreeCommonBuffer(ext->AdapterObject,
                            ext->BufLength,
                            ext->LogicalAddress,
                            ext->BufVirtualAddress, FALSE);
    if(oldIrql != 255)
        KeLowerIrql(oldIrql);

    RtlInitUnicodeString(&uniDOSString, DOSNameBuffer);
    IoDeleteSymbolicLink (&uniDOSString);
    IoDeleteDevice(DriverObject->DeviceObject);
}

/**********************************************************
  Generic service handler for Create and Close functions.
**********************************************************/
NTSTATUS CreateDispatch(PDEVICE_OBJECT DeviceObject,
                        PIRP Irp)
{
    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);
    return STATUS_SUCCESS;
}

/**********************************************************
  This routine maps the DMA buffer into the caller's
address space.  Must pass a PHYSICAL address, not a
kernel virtual address.  Generally follows example given
by MAPMEM sample code in DDK.
**********************************************************/
PVOID MapBuffer(PVOID PhysAddrLow, ULONG Length)
{
    UNICODE_STRING     uniNameString;
    OBJECT_ATTRIBUTES  attrib;
    HANDLE             hPhysMem = 0;
    PVOID              MemObject = 0;
    PHYSICAL_ADDRESS   physicalAddress, viewBase;
    PVOID              UserAddress = 0;

    physicalAddress.HighPart = 0;
    physicalAddress.LowPart = (ULONG)PhysAddrLow;
    RtlInitUnicodeString(&uniNameString,
                         L"\\Device\\PhysicalMemory");
    InitializeObjectAttributes(&attrib, &uniNameString,
                    OBJ_CASE_INSENSITIVE, (HANDLE) NULL,
                    (PSECURITY_DESCRIPTOR) NULL);
    if(!NT_SUCCESS(ZwOpenSection(&hPhysMem,
                    SECTION_ALL_ACCESS, &attrib)))
        return 0;

    if(!NT_SUCCESS(ObReferenceObjectByHandle(hPhysMem,
                    SECTION_ALL_ACCESS, (POBJECT_TYPE) NULL,
                    KernelMode, &MemObject,
                    (POBJECT_HANDLE_INFORMATION) NULL)))
        goto close;

    viewBase = physicalAddress;
    if(!NT_SUCCESS(ZwMapViewOfSection(hPhysMem, (HANDLE) -1,
                    &UserAddress, 0L, Length, &viewBase,
                    &Length, ViewShare, 0,
                    PAGE_READWRITE | PAGE_NOCACHE)))
        goto close;

    (ULONG)UserAddress += (ULONG)physicalAddress.LowPart -
                              (ULONG)viewBase.LowPart;
close:
    ZwClose (hPhysMem);
    return UserAddress;
}

/**********************************************************
  AdapterControl() is a callback specified in the
IoAllocateAdapterChannel() call.  It is responsible for
actually starting the DMA by calling IoMapTransfer() to
initialize the system DMA hardware, then calling whatever
routine is necessary to start the A/D board.
**********************************************************/
IO_ALLOCATION_ACTION AdapterControl(
    PDEVICE_OBJECT deviceObject,
    PIRP Irp, PVOID MapRegisterBase, PVOID Context)
{
    PDEVICE_EXTENSION ext;
    PHYSICAL_ADDRESS phys;

    ext = (PDEVICE_EXTENSION)
            deviceObject->DeviceExtension;
    ext->MapRegisterBase = MapRegisterBase;
    phys = IoMapTransfer(ext->AdapterObject, ext->pmdl,
                    MapRegisterBase,
                    ext->BufVirtualAddress,
                    &(ext->DMALength), FALSE);
    StartAD();
    return KeepObject;
}

/**********************************************************
  GoDMA() allocates the system DMA channel and, through
IoAllocateAdapterChannel(), calls
AdapterControl() to start the DMA.  We generally follow
the "common buffer" style of DMA described in the DDK.
*********************************************************/
NTSTATUS GoDMA(PDEVICE_EXTENSION ext)
{
    KIRQL oldIrql;
    ULONG NumMapPages;
    NTSTATUS status;

    // Make sure DMA length is multiple of sample size.
    ext->DMALength /= ext->BytesPerFullSample;
    ext->DMALength *= ext->BytesPerFullSample;

    oldIrql = 255;
    if(KeGetCurrentIrql() < DISPATCH_LEVEL)
        KeRaiseIrql(DISPATCH_LEVEL, &oldIrql);

    NumMapPages = ADDRESS_AND_SIZE_TO_SPAN_PAGES(
                ext->BufVirtualAddress, ext->BufLength);

    status = IoAllocateAdapterChannel(ext->AdapterObject,
                        ext->deviceObject, NumMapPages,
                        AdapterControl, 0);
    if(oldIrql != 255)
        KeLowerIrql(oldIrql);
    return status;
}

/**********************************************************
  Initialize device description and set up DMA buffer.
**********************************************************/
int DMAAllocateBuffer(PDEVICE_OBJECT deviceObject)
{
    DEVICE_DESCRIPTION deviceDescription;
    PDEVICE_EXTENSION ext;

    ext = (PDEVICE_EXTENSION)deviceObject->
                                        DeviceExtension;
    ext->BufLength = USERDMA_BUFFER_SIZE;
    ext->NumADChans = USERDMA_NUM_AD_CHANS;
    ext->BytesPerFullSample = ext->NumADChans * 2;

    RtlZeroMemory(&deviceDescription,
                    sizeof(deviceDescription));

    deviceDescription.Version = 0;
    deviceDescription.Master = FALSE;
    deviceDescription.ScatterGather = FALSE;
    deviceDescription.DemandMode = FALSE;
    deviceDescription.AutoInitialize = TRUE;
    deviceDescription.Dma32BitAddresses = FALSE;
    deviceDescription.BusNumber = 0;
    deviceDescription.DmaChannel = USERDMA_CHANNEL;
    deviceDescription.InterfaceType = Isa;
    deviceDescription.DmaWidth = Width16Bits;
    deviceDescription.DmaSpeed = Compatible;
    deviceDescription.MaximumLength = ext->BufLength;
    deviceDescription.DmaPort = 0;

    // Get AdapterObject using above deviceDescription.
    ext->AdapterObject = HalGetAdapter(&deviceDescription,
                            &(ext->MaximumMapRegisters));
    if(!ext->AdapterObject)
        return 0;

    // A Map Register corresponds to 4K of memory.
    // Make sure we don't use more than are available.
    if(ext->MaximumMapRegisters*4096 < ext->BufLength)
        ext->BufLength = ext->MaximumMapRegisters * 4096;

    // Allocate a buffer for the DMA data.  According to
    // DDK, we can only do this while initializing driver.
    ext->BufVirtualAddress =
        HalAllocateCommonBuffer(ext->AdapterObject,
                                ext->BufLength,
                                &(ext->LogicalAddress),
                                FALSE);
    if(!(ext->BufVirtualAddress))
        return 0;

    // Map the buffer.
    ext->pmdl = IoAllocateMdl(ext->BufVirtualAddress,
                        ext->BufLength, FALSE, FALSE, 0);
    if(!ext->pmdl)
        return 0;
    MmBuildMdlForNonPagedPool(ext->pmdl);
    return 1;
}

/**********************************************************
  Called from user DeviceControl().  Starts up DMA.
**********************************************************/
NTSTATUS
    DMAStart(USERDMA_INFO *dmainfo, PDEVICE_EXTENSION ext)
{
    NTSTATUS status = STATUS_SUCCESS;

    dmainfo->NumChans = ext->NumADChans;
    // Start DMA only if no one else is already using it.
    if(!ext->NumUsers) {
        // Take caller's suggestion for NumSamps
        ext->DMALength = dmainfo->NumSamps *
                            ext->BytesPerFullSample;
        // See if we have enough room.
        if(ext->DMALength > ext->BufLength)
            ext->DMALength = ext->BufLength;
        status = GoDMA(ext);
    }
    if(status != STATUS_SUCCESS)
        return status;

    // In any case, set the caller's NumSamps value
    dmainfo->NumSamps = ext->DMALength /
                        ext->BytesPerFullSample;

    // Keep track of how many callers are using us.
    ++(ext->NumUsers);

    // Give the caller a pointer to the DMA buffer.
    if(!dmainfo->buf)
        dmainfo->buf =
            MapBuffer((PVOID)(MmGetPhysicalAddress(
                ext->BufVirtualAddress).LowPart),
                ext->DMALength);
    return status;
}

/**********************************************************
  Called from DeviceControl() to stop DMA.
**********************************************************/
NTSTATUS DMAStop(
            USERDMA_INFO *dmainfo, PDEVICE_EXTENSION ext)
{
    KIRQL oldIrql;

    // Free user buffer mapping.
    if(dmainfo->buf) {
        ZwUnmapViewOfSection((HANDLE) -1, dmainfo->buf);
        dmainfo->buf = 0;
    }
    // Only stop DMA if this is the last user.
    if(ext->NumUsers > 0 && --(ext->NumUsers) == 0) {
        StopAD();
        IoFlushAdapterBuffers(ext->AdapterObject,
                ext->pmdl, ext->MapRegisterBase,
                ext->BufVirtualAddress,
                ext->BufLength, FALSE);
        oldIrql = 255;
        if(KeGetCurrentIrql() < DISPATCH_LEVEL)
            KeRaiseIrql(DISPATCH_LEVEL, &oldIrql);
        IoFreeAdapterChannel(ext->AdapterObject);
        if(oldIrql != 255)
            KeLowerIrql(oldIrql);
    }
    return STATUS_SUCCESS;
}

/**********************************************************
  Let the caller know the location of the most current
sample in the DMA buffer.  We use the HalReadDmaCounter()
routine to read the system DMA controller hardware, and
convert its "remaining bytes" value into an appropriate
index into the DMA buffer.
  We want to return a pointer to the most recent "full"
sample.  Since we sample more than one channel, the DMA
could be somewhere in the middle of a set of channel
samples.  We want the latest sample that has all channels
completed.
**********************************************************/
NTSTATUS DMACurOffset(USERDMA_INFO *dmainfo,
                    PDEVICE_EXTENSION ext)
{
    int CurIndex;

    CurIndex = ext->DMALength -
                HalReadDmaCounter(ext->AdapterObject);

    // Convert from bytes to full (all channels) samples.
    CurIndex /= ext->BytesPerFullSample;

    // Get last completed full sample.
    --CurIndex;

    // Convert from sample number to byte offset.
    CurIndex *= ext->BytesPerFullSample;

    // If we go below the start of the buffer, take the
    // last sample in the buffer.
    if(CurIndex < 0)
        CurIndex = ext->DMALength - ext->BytesPerFullSample;

    // Convert from byte offset to word offset.
    dmainfo->CurDMAWord = CurIndex / 2;

    return STATUS_SUCCESS;
}

/**********************************************************
  Process caller's DeviceIoControl() calls.  We recognize
three calls: start dma, stop dma, and get current index.
**********************************************************/
NTSTATUS DeviceControl(
        PDEVICE_OBJECT DeviceObject, PIRP Irp)
{
    PIO_STACK_LOCATION      irpSp;
    NTSTATUS                status;
    USERDMA_INFO            *dmainfo;
    PDEVICE_EXTENSION       ext;

    irpSp = IoGetCurrentIrpStackLocation(Irp);
    status = STATUS_UNSUCCESSFUL;
    Irp->IoStatus.Information = 0;

    if(irpSp->Parameters.DeviceIoControl.InputBufferLength
        < sizeof(USERDMA_INFO) ||
       irpSp->Parameters.DeviceIoControl.OutputBufferLength
        < sizeof(USERDMA_INFO))

        goto complete_request;          // bad parameters

    // Get pointer to caller's buffer.
    dmainfo = (USERDMA_INFO *)
                Irp->AssociatedIrp.SystemBuffer;
    ext = (PDEVICE_EXTENSION)DeviceObject->DeviceExtension;
    Irp->IoStatus.Information = sizeof(USERDMA_INFO);

    // Go to proper handler for the caller's request.
    switch(irpSp->
            Parameters.DeviceIoControl.IoControlCode) {
        case IOCTL_USERDMA_START:
                status = DMAStart(dmainfo, ext);
            break;
        case IOCTL_USERDMA_STOP:
                status = DMAStop(dmainfo, ext);
            break;
        case IOCTL_USERDMA_CUR_OFFSET:
                status = DMACurOffset(dmainfo, ext);
            break;
        default:
            goto complete_request;      // unknown IOCTL
    }

complete_request:
    Irp->IoStatus.Status = status;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);
    return status;
}

/**********************************************************
  Entry routine.  Set everything up.
**********************************************************/
NTSTATUS DriverEntry(PDRIVER_OBJECT DriverObject,
    PUNICODE_STRING RegistryPath)
{
    PDEVICE_OBJECT deviceObject;
    PDEVICE_EXTENSION ext;
    NTSTATUS status;
    WCHAR NameBuffer[] = USERDMA_DEVICE_NAME;
    WCHAR DOSNameBuffer[] = USERDMA_DOS_NAME;
    UNICODE_STRING uniNameString, uniDOSString;

    RtlInitUnicodeString(&uniNameString, NameBuffer);
    RtlInitUnicodeString(&uniDOSString, DOSNameBuffer);

    status = IoCreateDevice(DriverObject,
                    sizeof(DEVICE_EXTENSION),
                    &uniNameString,
                    FILE_DEVICE_UNKNOWN,
                    0, FALSE, &deviceObject);
    if(!NT_SUCCESS(status))
        return status;

    status = IoCreateSymbolicLink(&uniDOSString,
                             &uniNameString);
    if (!NT_SUCCESS(status)) {
        IoDeleteDevice(deviceObject);
        return status;
    }
    ext = (PDEVICE_EXTENSION)deviceObject->
                                        DeviceExtension;
    ext->deviceObject = deviceObject;
    deviceObject->Flags |= DO_BUFFERED_IO;

    DriverObject->MajorFunction[IRP_MJ_CREATE]
            = CreateDispatch;
    DriverObject->MajorFunction[IRP_MJ_CLOSE]
            = CreateDispatch;
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL]
            = DeviceControl;
    DriverObject->DriverUnload = DeviceUnload;

    if(!DMAAllocateBuffer(deviceObject)) {
        DeviceUnload(DriverObject);
        return STATUS_UNSUCCESSFUL;
    }
    return STATUS_SUCCESS;
}
