unit Main;

interface

uses
  SysUtils, WinTypes, WinProcs, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, Menus, ExtCtrls;

type
  TFontRec = record
    ColorText : tColor;
    ColorBack : tColor;
    FontHandle : hFont;
    FontSize : integer;
    FontName : tFontName;
    Escapement : integer;
    Style : tFontStyles;
  end;

  TMainForm = class(TForm)
    MainMenu: TMainMenu;
      mmFile: TMenuItem;
        smExit: TMenuItem;
      mmEdit: TMenuItem;
        smChooseFont: TMenuItem;
        smRedrawNow: TMenuItem;
        smAutoRedraw: TMenuItem;
    ToolBar: TPanel;
      FNamesCB: TComboBox;
      EscSBar: TScrollBar;
      EscLabel: TLabel;
      EscBevel: TBevel;
      SBtnAutoRedraw: TSpeedButton;
      SBtnRedrawNow: TSpeedButton;
      SBtnChooseFont: TSpeedButton;
    MainImage: TImage;
    MainEdit: TEdit;
    CommonFontDlg: TFontDialog;
    procedure FormCreate(Sender: TObject);
    procedure smExitClick(Sender: TObject);
    procedure smChooseFontClick(Sender: TObject);
    procedure smAutoRedrawClick(Sender: TObject);
    procedure smRedrawNowClick(Sender: TObject);
    procedure FNamesCBDrawItem(Control: TWinControl;
      Index: Integer; Rect: TRect; State: TOwnerDrawState);
    procedure MakeNewFontThenRedraw;
    function MakeNewFont : boolean;
    procedure MainImageRedraw;
    procedure OnChangeMaybeRedraw(Sender: tObject);
    procedure EscSBarScroll(Sender: TObject; ScrollCode:
      TScrollCode; var ScrollPos: Integer);
  private
    TheFontRec : TFontRec;
    TrueTypeBM : TBitmap;
    AutoRedraw : boolean;
  end;

var
  MainForm: TMainForm;


implementation

const
  DegreeSymbol = #176;

{$R *.DFM}
{$R FONTD2.RES}


function EnumFontsProc(var LogFont: tLogFont; var TextMetric:
  tTextMetric; FontType: integer; Data: pointer): integer; export;
begin
  Result := 1;
  if (TextMetric.tmPitchAndFamily and tmpf_TRUETYPE) > 0 then
    MainForm.FNamesCB.Items.AddObject(StrPas(LogFont.lfFaceName),
      pointer(True))  {if TrueType then store non-nil value}
  else                {else leave nil}
    MainForm.FNamesCB.Items.Add(StrPas(LogFont.lfFaceName));
end;


procedure TMainForm.FormCreate(Sender: TObject);
var
  DC: hDC;
begin
  TrueTypeBM := tBitmap.Create;
  try
    TrueTypeBM.Handle := LoadBitmap(hInstance, 'BMTRUE');
  except       {couldn't create or couldn't load bitmap}
    TrueTypeBM.Free;         {and display error message}
  end;

  DC := GetDC(0);
  EnumFontFamilies(DC, nil, @EnumFontsProc, nil);
  ReleaseDC(0, DC);

  with TheFontRec do
  begin
    ColorText := clBlue;
    ColorBack := clWhite;
    FontHandle := 0;
    FontSize := 14;
    FontName := 'Arial';
    Escapement := 0;
    Style := [];
  end;
  MainImage.Canvas.Brush.Color := TheFontRec.ColorBack;
  FNamesCB.ItemIndex := FNamesCB.Items.IndexOf(TheFontRec.FontName);
  EscLabel.Caption := IntToStr(EscSBar.Position) + DegreeSymbol;
  AutoRedraw := false;
  MakeNewFontThenRedraw;
end;


procedure TMainForm.smExitClick(Sender: TObject);
begin
  TrueTypeBM.Free; {OK, even if nil}
  Close;
end;


procedure tMainForm.OnChangeMaybeRedraw(Sender: tObject);
begin
  if AutoRedraw then
    if Sender = MainEdit then MainImageRedraw
    else MakeNewFontThenRedraw;
end;


procedure TMainForm.smChooseFontClick(Sender: TObject);
var
  NeedNewFont : boolean;
begin
  NeedNewFont := false;
  CommonFontDlg.Font := MainImage.Canvas.Font;
  if CommonFontDlg.Execute then
  begin
    {if only color changed don't make new font}
    if (CommonFontDlg.Font.Name <> TheFontRec.FontName)
    or (CommonFontDlg.Font.Size <> TheFontRec.FontSize)
    or (CommonFontDlg.Font.Style <> TheFontRec.Style)
    then NeedNewFont := true;
    FNamesCB.ItemIndex := FNamesCB.Items.
      IndexOf(CommonFontDlg.Font.Name);
    TheFontRec.FontSize := CommonFontDlg.Font.Size;
    TheFontRec.Style := CommonFontDlg.Font.Style;
    TheFontRec.ColorText := CommonFontDlg.Font.Color;
    if NeedNewFont then MakeNewFont;
    if AutoRedraw then MainImageRedraw;
  end;
end;


procedure TMainForm.smAutoRedrawClick(Sender: TObject);
begin
  AutoRedraw := not AutoRedraw;
  smAutoRedraw.Checked := AutoRedraw;
  smRedrawNow.Enabled := not AutoRedraw;
  SBtnAutoRedraw.Down := AutoRedraw;
  SBtnRedrawNow.Enabled := not AutoRedraw;
  if AutoRedraw then
    {if only text changed then don't make new font}
    if (TheFontRec.FontName <> FNamesCB.Text)
    or (TheFontRec.Escapement <> EscSBar.Position) then
      MakeNewFontThenRedraw else MainImageRedraw;
end;


procedure TMainForm.smRedrawNowClick(Sender: TObject);
begin
  MakeNewFontThenRedraw;
end;


procedure tMainForm.MakeNewFontThenRedraw;
begin
  if MakeNewFont then MainImageRedraw;
end;


function tMainForm.MakeNewFont : boolean;
var
  tempLogFont : tLogFont;
begin
  Result := true;
  GetObject(MainImage.Canvas.Font.Handle,
    SizeOf(tempLogFont), Addr(tempLogFont));
  with tempLogFont, TheFontRec do
  begin
    Escapement := EscSBar.Position;
    lfEscapement := Escapement * 10;
    FontName := FNamesCB.Text;
    StrPCopy(lfFaceName, FNamesCB.Text);
    lfHeight := - MulDiv(FontSize, Screen.PixelsPerInch, 72);
    lfWidth := 0; {let FontMapper map width to height}
    if fsBold in Style then lfWeight := fw_Bold
    else lfWeight := fw_Normal;
    lfItalic := byte(fsItalic in Style);
    lfUnderline := byte(fsUnderline in Style);
    lfStrikeOut := byte(fsStrikeOut in Style);

    FontHandle := CreateFontIndirect(tempLogFont);
    if FontHandle = 0 then
      Result := false  {and display error message}
    else
      MainImage.Canvas.Font.Handle := FontHandle;
  end;
end;


procedure tMainForm.MainImageRedraw;
begin
  MainImage.Canvas.FillRect(Rect(0, 0, MainImage.Picture.Width,
    MainImage.Picture.Height));
  if MainEdit.Text <>  '' then
  begin
    MainImage.Canvas.Font.Color := TheFontRec.ColorText;
    MainImage.Canvas.TextOut( MainImage.Picture.Width div 2,
      MainImage.Picture.Height div 2, MainEdit.Text);
  end;
end;


procedure TMainForm.FNamesCBDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
begin
  with FNamesCB do
  begin
    Canvas.FillRect(Rect);
    Canvas.TextRect(Rect, Rect.Left + 18, Rect.Top, Items[Index]);
    if (Items.Objects[Index] <>  nil)  {it's a TrueType font}
    and (TrueTypeBM <> nil) then       {and the bitmap is OK}
      Canvas.BrushCopy( Bounds(Rect.Left + 2, Rect.Top + 2,
          TrueTypeBM.Width, TrueTypeBM.Height),TrueTypeBM,
          Bounds(0, 0, TrueTypeBM.Width, TrueTypeBM.Height),
          clSilver);
  end;
end;


procedure TMainForm.EscSBarScroll(Sender: TObject;
  ScrollCode: TScrollCode; var ScrollPos: Integer);
begin
  EscLabel.Caption := IntToStr(ScrollPos) + DegreeSymbol;
  if AutoRedraw then
    if (ScrollCode in [scLineUp, scLineDown, scPageUp, scPageDown,
      scPosition, scEndScroll]) then MakeNewFontThenRedraw;
end;

end.
