		page	,132
;**********************************************************;
;							   ;
;   FILE NAME:	    ComPort.asm 			   ;
;							   ;
;   DESCRIPTION:    Manage I/Os to a serial port	   ;
;							   ;
;   NOTES:						   ;
;							   ;
;**********************************************************;

		.386p

WIN31COMPAT	equ	1

.xlist
include VMM.INC     ; Virtual Machine Manager definitions
.list
include comports.inc
include IrqMon.inc  ; Converted from irqmon.h
option casemap:notpublic ; broken by H2INC

BAUDS		equ	19200	    ; COM port speed

;==========================================================;
;		      Locked segments			   ;
;==========================================================;

VxD_LOCKED_DATA_SEG

	    public  dwPort
dwPort	    dd	    COM2    ; Serial port base I/O port

Old_IER     db	    ?	    ; UART Interrupt Enable Register

VxD_LOCKED_DATA_ENDS

VxD_LOCKED_CODE_SEG

;----------------------------------------------------------;
;							   ;
;   Function:	    Init_UART				   ;
;							   ;
;   Description:    Initialize the UART used for debug	   ;
;							   ;
;   Parameters:     None				   ;
;							   ;
;   Returns:	    None				   ;
;							   ;
;   Regs altered:   EAX, EBX, EDX			   ;
;							   ;
;   Notes:						   ;
;							   ;
;----------------------------------------------------------;

Init_UART	proc	C public

	pushfd
	cli		; Begin critical section

	; Set COM port speed

	mov	ebx, dwPort

	lea	edx, [ebx+LCR]
	mov	al, DLAB	; Select divisor latch
	out	dx, al

	lea	edx, [ebx+DLL]
	mov	al, LOW(115200 / BAUDS)
	out	dx, al

	lea	edx, [ebx+DLM]
	mov	al, HIGH(115200 / BAUDS)
	out	dx, al

	; Other COM port settings

	lea	edx, [ebx+LCR]
	mov	al, WLS8 + STB1 ; 8 bits, 1 stop bit, no par
	out	dx, al

	lea	edx, [ebx+IER]
	in	al, dx
	mov	Old_IER, al
	xor	al, al
	out	dx, al		; Disable all interrupts

	lea	edx, [ebx+FCR]
	mov	al, FFE + FFRR + FFXR
	out	dx, al		; Enable Fifos

	; Done

	popfd		; End critical section
	ret

Init_UART	endp

;----------------------------------------------------------;
;							   ;
;   Function:	    Reset_UART				   ;
;							   ;
;   Description:    Minimal cleanup of the UART 	   ;
;							   ;
;   Parameters:     None				   ;
;							   ;
;   Returns:	    None				   ;
;							   ;
;   Regs altered:   EAX, EBX, EDX			   ;
;							   ;
;   Notes:						   ;
;							   ;
;----------------------------------------------------------;

Reset_UART	proc	C public uses ESI

	pushfd
	cli		; Begin critical section

	mov	ebx, dwPort

	lea	edx, [ebx+IER]
	mov	al, Old_IER
	out	dx, al	; Restore all interrupts

	popfd		; End critical section
	ret

Reset_UART	endp

;----------------------------------------------------------;
;							   ;
;   Function:	    SendEAX				   ;
;							   ;
;   Description:    Send a hex dump of EAX on the COM port ;
;							   ;
;   Parameters:     EAX = Dword to dump 		   ;
;							   ;
;   Returns:	    None				   ;
;							   ;
;   Regs altered:   None				   ;
;							   ;
;   Notes:	    The critical section is not protected. ;
;		    Make sure interrupts are disabled	   ;
;		     before calling this routine if it may ;
;		     be reentered.			   ;
;							   ;
;----------------------------------------------------------;

SendEAX proc	C public

	ror	eax, 16
	call	SendAX

	rol	eax, 16
	jmp	SendAX

SendEAX endp

;----------------------------------------------------------;
;							   ;
;   Function:	    SendAX				   ;
;							   ;
;   Description:    Send a hex dump of AX on the COM port  ;
;							   ;
;   Parameters:     AX = Word to dump			   ;
;							   ;
;   Returns:	    None				   ;
;							   ;
;   Regs altered:   None				   ;
;							   ;
;   Notes:	    The critical section is not protected. ;
;		    Make sure interrupts are disabled	   ;
;		     before calling this routine if it may ;
;		     be reentered.			   ;
;							   ;
;----------------------------------------------------------;

SendAX	proc	C public

	xchg	al, ah
	call	SendAL

	xchg	al, ah
	jmp	SendAL

SendAX	endp

;----------------------------------------------------------;
;							   ;
;   Function:	    SendAL				   ;
;							   ;
;   Description:    Send a hex dump of AL on the COM port  ;
;							   ;
;   Parameters:     AL = Byte to dump			   ;
;							   ;
;   Returns:	    None				   ;
;							   ;
;   Regs altered:   None				   ;
;							   ;
;   Notes:	    The critical section is not protected. ;
;		    Make sure interrupts are disabled	   ;
;		     before calling this routine if it may ;
;		     be reentered.			   ;
;							   ;
;----------------------------------------------------------;

SendAL	proc	C public

	ror	al, 4
	call	SendNibble

	rol	al, 4
	jmp	SendNibble

SendAL	endp

;----------------------------------------------------------;
;							   ;
;   Function:	    SendNibble				   ;
;							   ;
;   Description:    Send a hex character from the lower    ;
;		     half of AL 			   ;
;							   ;
;   Parameters:     AL = nibble to dump.		   ;
;							   ;
;   Returns:	    None				   ;
;							   ;
;   Regs altered:   None				   ;
;							   ;
;   Notes:	    The upper nibble of AL is ignored	   ;
;							   ;
;		    Works even if DS is invalid.	   ;
;							   ;
;----------------------------------------------------------;

	align	4

hex	db	"0123456789ABCDEF"

SendNibble proc C public uses EAX

	and	eax, 0FH
	mov	al, cs:hex[eax]
	call	SendChar

	ret

SendNibble endp

;----------------------------------------------------------;
;							   ;
;   Function:	    SendChar				   ;
;							   ;
;   Description:    Send a character on the COM port	   ;
;							   ;
;   Parameters:     AL = character			   ;
;							   ;
;   Returns:	    None				   ;
;							   ;
;   Regs altered:   None				   ;
;							   ;
;   Notes:	    Works even if DS is invalid.	   ;
;							   ;
;----------------------------------------------------------;

SendChar  proc	C public uses EAX ECX EDX

	mov	ah, al

	mov	edx, ss:dwPort
	add	edx, LSR
	mov	ecx, 2 + (666666 / BAUDS)   ; Timeout in
					    ;  15us unit
	pushfd
	cli			; Begin critical section

read_line_status_loop:
	in	al, dx
	test	al, THRE    ; Trans holding register empty ?
	jnz	transmit_data
	call	wait_refresh
	loop	read_line_status_loop

	jmp	send_timeout

transmit_data:
	add	edx, THR-LSR	; Port data register address
	mov	al, ah		; Restore the byte to send
	out	dx, al

send_timeout:
	popfd			; End critical section
	ret

SendChar  endp

;----------------------------------------------------------;
;							   ;
;   Function:	    wait_refresh			   ;
;							   ;
;   Description:    Wait for the next refresh cycle.	   ;
;							   ;
;   Parameters:     None				   ;
;							   ;
;   Returns:	    None				   ;
;							   ;
;   Regs altered:   None				   ;
;							   ;
;   Notes:	    There's one refresh cycle every 15 us. ;
;							   ;
;----------------------------------------------------------;

SPU_CONTROL_PORT	equ	61H
REFRESH_MASK		equ	10H

wait_refresh	proc

	push	eax

	in	al, SPU_CONTROL_PORT
	mov	ah, al		    ; Reference value
@@:
	in	al, SPU_CONTROL_PORT
	xor	al, ah		    ; Which bits changed?
	test	al, REFRESH_MASK    ; Test refresh only
	jz	@B		    ; Loop if not changed.

	pop	eax
	ret

wait_refresh	endp

VxD_LOCKED_CODE_ENDS

END
