/**********************************************************\
*							   *
*   File name:	    IrqMon.c				   *
*							   *
*   Description:    VxD for intercepting interrupts	   *
*							   *
\**********************************************************/

#define WANTVXDWRAPS

#include <basedef.h>
#include <vmm.h>
#include <vxdwraps.h>

#include "IrqMon.h"	// Global constants for this program

// Manipulate words.
// Can be used both as an RVALUE and an LVALUE.
#define WORD0(dw) (*((WORD *)(&(dw))+0))
#define WORD1(dw) (*((WORD *)(&(dw))+1))

/* Segmentation */

#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG

/* Global variables */

int nIdt = 0;		// Number of IDTs located so far
PCALLGATEDESCRIPTOR pIdtList[NIDT]; // Addresses of the IDTs

PCALLGATEDESCRIPTOR pAbandonedIdt = NULL; // IDT about to go

/*--------------------------------------------------------*\
*							   *
|   Function:	    Add_IDT				   |
|							   |
|   Description:    Add an IDT to our list		   |
|							   |
|   Parameters:     Base of the IDT			   |
*							   *
\*--------------------------------------------------------*/

void Add_IDT1(PCALLGATEDESCRIPTOR pIdt)
  {
  int i;
  int i1stFr;	/* Index of first free IDT pointer */

  /* If closing a PM app, don't add its IDT back. */
  if (pIdt == pAbandonedIdt) return;
  pAbandonedIdt = NULL; /* Forget the abandonned IDT. */

  for (i=0, i1stFr = nIdt; i<nIdt; i++)
    {
    if (pIdt == pIdtList[i]) return;	/* Known IDT */
    if ((i1stFr > i) && (!pIdtList[i])) i1stFr = i;
    }

  SENDCHAR('&');
  SENDEAX(pIdt);	/* Display the new IDT base */

  if (i1stFr == NIDT) return; /* Prevent tbl overflow */

  pIdtList[i1stFr] = pIdt; /* Record the IDT address */

  /* Intercept the requested IRQs directly in the IDT */
  for (i=0; i<16; i++) if (dwIrqMask & (1 << i))
    {
    DWORD dw;
    int ix = 0x50+i;	/* IRQ handler index in an IDT */

    SENDCHAR('=');
    SENDNIBBLE(i);	/* Display the IRQ being hooked */

    WORD0(Old_IDT_IRQs[i1stFr][i]) = pIdt[ix].Offset_0_15;
    WORD1(Old_IDT_IRQs[i1stFr][i]) = pIdt[ix].Offset_16_31;

    SENDCHAR('-');
    dw = Old_IDT_IRQs[i1stFr][i];
    SENDEAX(dw);	/* Display the old handler */

    dw = (DWORD)My_IRQ_Handler+(HANDLER_SIZE*((i1stFr*16)+i));

    SENDCHAR('+');
    SENDEAX(dw);	/* Display the new handler */

    pIdt[0x50+i].Offset_0_15 = WORD0(dw);
    pIdt[0x50+i].Offset_16_31 = WORD1(dw);
    }

  if (i1stFr == nIdt) nIdt += 1;
  return;
  }

void Add_IDT(PCALLGATEDESCRIPTOR pIdt)
  {
  _asm pushfd
  _asm cli	    /* Begin critical section */

  Add_IDT1(pIdt);

  _asm popfd	    /* End critical section */
  }

/*--------------------------------------------------------*\
*							   *
|   Function:	    Remove_IDT				   |
|							   |
|   Description:    Remove an IDT from our list 	   |
|							   |
|   Parameters:     Base of the IDT			   |
*							   *
\*--------------------------------------------------------*/

void Remove_IDT(PCALLGATEDESCRIPTOR pIdt)
  {
  int i, j;

  _asm pushfd;
  _asm cli;	    /* Begin critical section */

  for (i=0; i<nIdt; i++)      /* Find the IDT in our list */
    {
    if (pIdt == pIdtList[i])	/* If found, remove it */
      {
      SENDCHAR('$');
      SENDEAX(pIdt);

      /* Restore the IDT handler addresses */
      for (j=0; j<16; j++) if (dwIrqMask & (1 << j))
	{
	int ix = 0x50+j;
	DWORD dw;

	SENDCHAR('=');
	SENDNIBBLE(j);

	SENDCHAR('-');
	WORD0(dw) = pIdt[ix].Offset_0_15;
	WORD1(dw) = pIdt[ix].Offset_16_31;
	SENDEAX(dw);

	dw = Old_IDT_IRQs[i][j];
	pIdt[ix].Offset_0_15 = WORD0(dw);
	pIdt[ix].Offset_16_31 = WORD1(dw);

	SENDCHAR('+');
	SENDEAX(dw);
	}

      /* And free the entry */
      pIdtList[i] = NULL;
      }
    }

  _asm popfd;	    /* End critical section */

  return;
  }

/*--------------------------------------------------------*\
*							   *
|   Function:	    IRQMON_Dynamic_Init 		   |
|							   |
|   Description:    Called once after the VxD is loaded    |
*							   *
\*--------------------------------------------------------*/

DWORD _stdcall IRQMON_Dynamic_Init(void)
  {
  Init_UART();	     /* Prepare the UART for output */
  VPICD_Hook_IRQs(); /* Hook IRQs using the VPICD service */

  return(VXD_SUCCESS);
  }

/*--------------------------------------------------------*\
*							   *
|   Function:	    IRQMON_Dynamic_Exit 		   |
|							   |
|   Description:    Called once before the VxD is unloaded |
*							   *
\*--------------------------------------------------------*/

DWORD _stdcall IRQMON_Dynamic_Exit(void)
  {
  int i;
  PCALLGATEDESCRIPTOR pIdt;

  _asm pushfd;
  _asm cli;	    /* Begin critical section */

  VPICD_Unhook_IRQs();	  /* Unhook IRQs from VPICD */

  for (i=0; i<nIdt; i++)  /* Unhook all IRQs in all IDTs */
    {
    if (pIdtList[i]) Remove_IDT(pIdtList[i]);
    }

  Reset_UART();     /* Minimal cleanup */

  _asm popfd;	    /* End critical section */

  return(VXD_SUCCESS);
  }

/*--------------------------------------------------------*\
*							   *
|   Function:	    IRQMON_Begin_PM_App 		   |
|							   |
|   Description:    Called when a prot mode applic starts  |
*							   *
\*--------------------------------------------------------*/

DWORD _stdcall IRQMON_Begin_PM_App(struct pmcb_s *pPMCB,
				   DWORD dwFlags)
    {
    Add_IDT(GetIdtBase()); /* Add the new IDT to our list */

    return VXD_SUCCESS;
    }

/*--------------------------------------------------------*\
*							   *
|   Function:	    IRQMON_End_PM_App			   |
|							   |
|   Description:    Called when a prot mode applic ends    |
*							   *
\*--------------------------------------------------------*/

DWORD _stdcall IRQMON_End_PM_App(struct pmcb_s *pPMCB)
    {
    /* Remove the IDT from the list, and make sure it is not
       added back, should it be used for a few more ints. */
    pAbandonedIdt = GetIdtBase();
    Remove_IDT(pAbandonedIdt);	 /* Remove it from the list */

    return VXD_SUCCESS;
    }
