/**********************************************************\
*							   *
*   File name:	    vxd.c				   *
*							   *
*   Description:    Manage dynamic VxDs 		   *
*							   *
\**********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define streq(s1, s2) (!strcmp(s1, s2)) // TRUE if equal str

typedef unsigned char BYTE;
typedef unsigned short WORD;
typedef unsigned long DWORD;

#define WORD0(dw) (*((WORD *)(&(dw))+0))
#define WORD1(dw) (*((WORD *)(&(dw))+1))
#define BYTE0(dw) (*((BYTE *)(&(dw))+0))
#define BYTE1(dw) (*((BYTE *)(&(dw))+1))
#define BYTE2(dw) (*((BYTE *)(&(dw))+2))
#define BYTE3(dw) (*((BYTE *)(&(dw))+3))

#define FALSE 0
#define TRUE 1

typedef WORD (_fastcall far * lpVxdEntry)();

#define VXDLDR_DEVICE_ID 0x0027 // VxD Loader device ID
enum				// V86 entry point functions
  {
  VXDLDR_GET_VERSION,
  VXDLDR_LOAD,
  VXDLDR_UNLOAD,
  };
char *pszVxdldrErrorDescription[] =
  {
  "No error",
  "Out of memory",    // VXDLDR_ERR_OUT_OF_MEMORY    1
  "DOS error",	      // VXDLDR_ERR_IN_DOS	     2
  "Can't open file",  // VXDLDR_ERR_FILE_OPEN_ERROR  3
  "Can't read file",  // VXDLDR_ERR_FILE_READ	     4
  "Duplicate VxD",    // VXDLDR_ERR_DUPLICATE_DEVICE 5
  "Not a VxD file",   // VXDLDR_ERR_BAD_DEVICE_FILE  6
  "VxD refused",      // VXDLDR_ERR_DEVICE_REFUSED   7
  "VxD not found",    // VXDLDR_ERR_NO_SUCH_DEVICE   8
  };
#define N_VXDLDR_ERROR \
  (sizeof(pszVxdldrErrorDescription) / sizeof(char *))

// Forward references

void usage(void);
lpVxdEntry GetVxdEntryPoint(WORD wID);
WORD GetVmmVersion(void);

/*--------------------------------------------------------*\
*							   *
|   Function:	    main				   |
|							   |
|   Description:    EXE program main initialization routine|
|							   |
|   Parameters:     int argc	    Number of arguments    |
|		    char *argv[]    List of arguments	   |
|							   |
|   Returns:	    The return code to pass to DOS.	   |
*							   *
\*--------------------------------------------------------*/

int main(int argc, char *argv[])
  {
  char *pszVxdName = NULL;
  lpVxdEntry lpVXDLDR;
  WORD wVersion;
  int i;
  int iLoad = FALSE;
  int iUnload = FALSE;
  WORD wErr;
  WORD wCarry;


  printf("\n");

  /* Process arguments */

  for (i=1 ; i<argc ; i++)
    {
    strlwr(argv[i]);
    if ((argv[i][0] == '-') || (argv[i][0] == '/'))
      { 			    /* It's a switch */
      if (streq(argv[i]+1, "?"))	/* -?: Help */
	{
	usage();			/* Display help */
	}
      if (streq(argv[i]+1, "l"))	/* -l: Load */
	{
	if (i < (argc-1))
	  {
	  iLoad = TRUE;
	  pszVxdName = argv[++i];
	  }
	continue;
	}
      if (streq(argv[i]+1, "u"))	/* -u: Unload */
	{
	if (i < (argc-1))
	  {
	  iUnload = TRUE;
	  pszVxdName = argv[++i];
	  }
	continue;
	}
      printf("Unrecognized switch %s. Ignored.\n", argv[i]);
      continue;
      }

    printf("Unexpected argument: %s\nIgnored.\n", argv[i]);
    }

  /* Make sure the context is correct */

  if (iLoad && iUnload)
    {
    printf("Please specify only one operation at a time.\n");
    usage();
    }

  // Make sure Windows enhanced is running
  wVersion = GetVmmVersion();
  if (!wVersion)
    {
    printf("This must run in a DOS box under Windows enhanced.\n");
    exit(1);
    }
  printf("Running under Windows Enhanced Version %d.%02d\n",
	      BYTE1(wVersion), BYTE0(wVersion));

  // Make sure the VXDLDR VxD is present
  lpVXDLDR = GetVxdEntryPoint(VXDLDR_DEVICE_ID);
  if (!lpVXDLDR)
    {
    printf("VxD Loader not present.\n");
    printf("Add \"Device=VXDLDR.386\" in SYSTEM.INI.\n");
    exit(1);
    }

  /* Do the requested operations */

  if (iLoad)
    {
    printf("Loading VxD %s\n", pszVxdName);
    wErr = lpVXDLDR(VXDLDR_LOAD, (WORD)pszVxdName);
    }

  if (iUnload)
    {
    printf("Unloading VxD %s\n", pszVxdName);
    wErr = lpVXDLDR(VXDLDR_UNLOAD, (WORD)pszVxdName, -1);
    }

  if (iLoad || iUnload)
    {
    if (wErr >= N_VXDLDR_ERROR)
      printf("Unknown error %d\n", wErr);
    else if (wErr)
      printf("Error %d: %s.\n", wErr,
	     pszVxdldrErrorDescription[wErr]);
    else
      printf("Done\n");
    }

  return 0;
  }

/*--------------------------------------------------------*\
*							   *
|   Function:	    usage				   |
|							   |
|   Description:    Display a brief help for this program  |
|							   |
|   Parameters:     None				   |
|							   |
|   Returns:	    N/A 				   |
*							   *
\*--------------------------------------------------------*/

void usage(void)
    {
    printf("\
Dynamic VxDs manager.\n\
\n\
Usage:\n\
\n\
vxd [switches]\n\
\n\
Switches:\n\
\n\
  -l {filename}     Load the given VxD.\n\
  -u {modulename}   Unload the given VxD.\n\
\n\
");

    exit(0);
    }

// Ignore the inline assembler warning
#pragma warning(disable:4704)

// Don't add useless stack checking
#pragma check_stack(off)

/*--------------------------------------------------------*\
*							   *
|   Function:	    GetVxdEntryPoint			   |
|							   |
|   Description:    Get the V86 entry point of a VxD	   |
|							   |
|   Parameters:     WORD wID	    The VxD's ID number    |
|							   |
|   Returns:	    The entry point, or NULL if VxD absent |
*							   *
\*--------------------------------------------------------*/

lpVxdEntry GetVxdEntryPoint(WORD wID)
  {
  _asm
    {
    xor   di, di
    mov   es, di	// ES:DI = NULL on entry
    mov   ax, 1684H	// Function number
    mov   bx, wID	// Device ID
    int   2Fh		// Get protected mode API address
    mov   ax, di
    mov   dx, es
    }
  // The return value is already in AX. Ignore the warning.
  }

/*--------------------------------------------------------*\
*							   *
|   Function:	    GetVmmVersion			   |
|							   |
|   Description:    Get Virtual Machine Manager's version  |
|							   |
|   Parameters:     None				   |
|							   |
|   Returns:	    High byte = major version		   |
|		    Low byte = minor version		   |
|		    Version 0 means VMM is not running.    |
|							   |
|   Notes:	    Can be used to detect if Windows is    |
|		    running in enhanced mode.		   |
*							   *
\*--------------------------------------------------------*/

WORD GetVmmVersion(void)  // Return Windows VMM version
  {
  WORD wWinVer;

  _asm mov ax, 1600h
  _asm int 2fh
  _asm xchg al, ah
  _asm mov wWinVer, ax
  if ((BYTE1(wWinVer) == 0) || (BYTE1(wWinVer) == 0x80))
    return 0;	/* Windows enhanced not present */
  return wWinVer;
  }
