/*
 *  Listing 1
 *  intr.h
 */

#ifndef __INTR_H
#define __INTR_H
#include <dos.h>    /* BYTEREGS type */

typedef unsigned int WORDREG;
typedef unsigned char BYTEREG;

struct EXT_REGPACK {    /* from Borland's REGPACK */
    WORDREG rax, rbx, rcx, rdx;
    WORDREG rbp, rsi, rdi, rds, res, rflags;
    WORDREG rss, rcs;     /* added for SegRead() */
    };

const int _CF = 1;      /* carry flag */
const int NUMWORDREGS = sizeof(EXT_REGPACK)/
                        sizeof(WORDREG);
const int NUMBYTEREGS = sizeof(BYTEREGS)/
                        sizeof(BYTEREG);

/* Modified version of REGS union; allows array-based
 * access to structure members ....... */
typedef union  {
    WORDREG wr[NUMWORDREGS];
    BYTEREG br[NUMBYTEREGS];
    EXT_REGPACK  x;
    BYTEREGS h;
    } XREGS;

enum WORDREGtag {  ax,bx,cx,dx,bp,si,
                   di,ds,es,flags,ss,cs };
enum BYTEREGtag {  al,ah,bl,bh,cl,ch,dl,dh };

class Intr {
  protected:
    unsigned _intn;    /* interrupt number */
    unsigned _errn;    /* error number on failure */
    XREGS _regs;

  protected:
    virtual int ExecFail(); //detect exec() failure

  public:
    Intr(unsigned n): _intn(n) {  Clear(); }
    virtual ~Intr() {}
    unsigned IntNumber() const { return _intn; }
    unsigned ErrNumber() const { return _errn; }

    WORDREG& operator [] (WORDREGtag t)
        {   return _regs.wr[t];   }
    BYTEREG& operator [] (BYTEREGtag t)
        {   return _regs.br[t];   }

    Intr & operator <= (Intr& (*)(Intr&));
    operator void *();
    int  operator! ();
    virtual void Clear();   //clear internal data
    virtual void Exec();     //execute interrupt
    virtual void SegRead();  // Reads CPU seg regs

    void ax(WORDREG val)  { _regs.x.rax = val; }
    void bx(WORDREG val)  { _regs.x.rbx = val; }
    void cx(WORDREG val)  { _regs.x.rcx = val; }
    void dx(WORDREG val)  { _regs.x.rdx = val; }
    void bp(WORDREG val)  { _regs.x.rbp = val; }
    void si(WORDREG val)  { _regs.x.rsi = val; }
    void di(WORDREG val)  { _regs.x.rdi = val; }
    void ds(WORDREG val)  { _regs.x.rds = val; }
    void es(WORDREG val)  { _regs.x.res = val; }
    void ss(WORDREG val)  { _regs.x.rss = val; }
    void cs(WORDREG val)  { _regs.x.rcs = val; }

    void al(WORDREG val) { _regs.h.al = (BYTEREG) val;}
    void ah(WORDREG val) { _regs.h.ah = (BYTEREG) val;}
    void bl(WORDREG val) { _regs.h.bl = (BYTEREG) val;}
    void bh(WORDREG val) { _regs.h.bh = (BYTEREG) val;}
    void cl(WORDREG val) { _regs.h.cl = (BYTEREG) val;}
    void ch(WORDREG val) { _regs.h.ch = (BYTEREG) val;}
    void dl(WORDREG val) { _regs.h.dl = (BYTEREG) val;}
    void dh(WORDREG val) { _regs.h.dh = (BYTEREG) val;}

    WORDREG ax() const { return _regs.x.rax; }
    WORDREG bx() const { return _regs.x.rbx; }
    WORDREG cx() const { return _regs.x.rcx; }
    WORDREG dx() const { return _regs.x.rdx; }
    WORDREG bp() const { return _regs.x.rbp; }
    WORDREG si() const { return _regs.x.rsi; }
    WORDREG di() const { return _regs.x.rdi; }
    WORDREG flags() const { return _regs.x.rflags; }
    WORDREG cflag() const
                  { return _regs.x.rflags & _CF;}
    WORDREG ds() const { return _regs.x.rds; }
    WORDREG es() const { return _regs.x.res; }
    WORDREG ss() const { return _regs.x.rss; }
    WORDREG cs() const { return _regs.x.rcs; }

    WORDREG al() const { return (WORDREG) _regs.h.al; }
    WORDREG ah() const { return (WORDREG) _regs.h.ah; }
    WORDREG bl() const { return (WORDREG) _regs.h.bl; }
    WORDREG bh() const { return (WORDREG) _regs.h.bh; }
    WORDREG cl() const { return (WORDREG) _regs.h.cl; }
    WORDREG ch() const { return (WORDREG) _regs.h.ch; }
    WORDREG dl() const { return (WORDREG) _regs.h.dl; }
    WORDREG dh() const { return (WORDREG) _regs.h.dh; }
};

/* Manipulators .......... */
Intr& Clear( Intr& );   /* Clear all Intr data */
Intr& Exec( Intr& );  /* Execute */

/* Intr specialized for using DOS services ....... */
class IntDos : public Intr  {
    DOSERROR _doserr;
  protected:
    int ExecFail();
  public:
    IntDos() : Intr( 0x21 ) {}
    virtual ~IntDos() {}
    int DosErrorCode() const
        { return _doserr.de_exterror;   }
    int DosErrorClass() const
        { return _doserr.de_class;      }
    int DosErrorAction() const
        { return _doserr.de_action;     }
    int DosErrorLocus() const
        { return _doserr.de_locus;      }
};

#endif
/* ----- EOF --------------------------------------- */
