# =============================================================================
#
# = DESCRIPTION
#     Awk script which performs the actual conversion from classinfo file
#     to various types of documentation.
#
# = AUTHOR(S)
#     Graham Dumpleton
# 
# = COPYRIGHT
#     Copyright 1991 OTC LIMITED
#     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
#
# =============================================================================

BEGIN {
  #
  # Set some defaults.
  #
  # device = "troff"
  access["private"] = "off"
  access["protected"] = "on"
  access["public"] = "on"

  lower["CLASS"] = "class"
  lower["STRUCT"] = "struct"
  lower["UNION"] = "union"
  lower["class"] = "class"
  lower["struct"] = "struct"
  lower["union"] = "union"
  upper["public"] = "PUBLIC"
  upper["protected"] = "PROTECTED"
  upper["private"] = "PRIVATE"

  #
  # Check that info file, class have been specified.
  #
  # if ( infile ~ /^$/ || class ~ /^$/ )
  #   exit

  #
  # Read in info file.
  #
  # FS="\n"; RS=""
  # while ( getline < infile > 0 )
  #   lines[++num] = $0

  #
  # Prepare for format file.
  #
  FS=" "; RS="\n"

  #
  # Find some defaults in info file.
  #
  # findLibrary()
  # findFilename()
}

#
# Start next pass.
#
$1 == "start" {

  if ( pass == 0)
  {
    #
    # Check that info file, class have been specified.
    #
    if ( infile ~ /^$/ || class ~ /^$/ )
      exit

    #
    # Read in info file.
    #
    FS="\n"; RS=""
    while ( getline < infile > 0 )
      lines[++num] = $0

    #
    # Prepare for format file.
    #
    FS=" "; RS="\n"

    #
    # Find some defaults in info file.
    #
    findLibrary()
    findFilename()

    #
    # Check to see if we have a version of
    # gawk which doesn't do pattern substitution
    # for backslahess as we expect.
    #
    test1 = "\\x"
    backslashworkaround = 0
    gsub( "\\\\", "\\\\", test1 )
    if ( test1 == "\\x" )
      backslashworkaround = 1
  }

  pass++
}

#
# Sets the output device to use.
#
# $1 == "set" && $2 == "device" {
#   device = $3
# }

#
# Prevents a section from being printed.
#
$1 == "hide" && $2 == "section" {
  section = $3
  for ( i=4; i<=NF; i++ )
   section = " " section
  sections[section] = ""
}

#
# Prevents a contract from being printed.
#
# Note: a contract will not be displayed if it occurs in a part of the class
# which has been disabled. For instance, if anything occuring in the private
# section of a class is being hidden, the no contracts from that section will
# be displayed either.
#
$1 == "hide" && $2 == "contract" {
  contract = $3
  for ( i=4; i<=NF; i++ )
   contract = " " contract
  contracts[contract] = ""
}

#
# Specification of whether a part of a class is hidden or not.
#
$1 == "set" && $2 in access {
  access[$2] = $3
}

#
# Print a prelude. This should produce any stuff necessary to initialise
# the document.
#
pass == "2" && $1 == "print" && $2 == "prelude" {
  "date" | getline date
  split ( date, d, " " )
  date = d[2] " " d[3] ", " d[6]
  if ( device == "troff" )
  {
    print( ".\\\" troff -man %" )
    print( ".\\\"" )
    print( ".\\\" DO NOT EDIT" )
    print( ".\\\"" )
    print( ".\\\" This manual page is automatically generated by class2man." )
    print( ".\\\"" )
    print( ".ds sL " library )
    print( ".ds sV " date )
    print( ".ds sC " class )
    print( ".ds sS 3" )
    print( ".TH \"\\*(sC\" \"\\*(sS\" \"\\*(sV\" \"\\*(sL Library\" \"\\*(sL Reference Manual\"" )
  }
  else if ( device == "mml" )
  {
    print( "<MML>" )
    print( "<Comment *** This file is automatically generated by class2mml.>" )
    print( "<!DefineTag Body>" )
    print( "<!DefineTag Heading1>" )
    print( "<!DefineTag Heading2>" )
    print( "" )
  }
}

#
# Print out a title.
# The title should be generated from the class name and any information
# in the section labelled 'TITLE" which occurs in the hdr of a class.
# On pass 1 we simply note that we intend using the 'TITLE' section so
# that it doesn't get printed out as a seperate section.
#
pass == "1" && $1 == "print" && $2 == "title" {
  sections["TITLE"] = ""
}

#
# On pass 2 we actually print the title.
# Note: even if a 'TITLE' section is not found for the class, then the class
# name at least should be output.
#
pass == "2" && $1 == "print" && $2 == "title" {
  found = 0
  for ( i=0; i<=num; i++ )
  {
    n = split( lines[i], line, "\n" )
    if ( line[1] == "INFO" &&
	 line[2] == "HDR" &&
	 line[3] == "TITLE" &&
	 line[4] == class )
    {
      found = 1
      break
    }
  }
  if ( device == "troff" )
  {
    print( ".PP" )
    print( ".SH \"NAME\"" )
    if ( found == 1 )
    {
      print( class " \\- " )
      for ( i=5; i<=n; i++ )
	outputLine( line[i] )
    }
    else
      print( class )
  }
  else if ( device == "mml" )
  {
    print( "<Heading1>" )
    print( class )
    if ( found == 1 )
    {
      print( "<Body>" )
      for ( i=5; i<=n; i++ )
	outputLine( line[i] )
    }
  }
}

#
# Print a synopsis.
# This should include a line indicating which header file needs to be included
# and a description of the members variables and functions in the class.
# A section of the class should not be output if access["section"] == "off".
# e.g.: Don't print private stuff if access["private"] == "off".
#
pass == 2 && $1 == "print" && $2 == "synopsis" {
  if ( device == "troff" )
  {
    print( ".PP" )
    print( ".SH \"SYNOPSIS\"" )
    print( ".nf\n" )
  }
  else if ( device == "mml" )
  {
    print( "" )
    print( "<Heading2>")
    print( "SYNOPSIS" )
    printf( "<Body>" )
    printf( "<Plain>" )
    printf( "<LeftIndent 0.1in>" )
    printf( "<FirstIndent 0.1in>" )
    printf( "<Alignment l>" )
  }

  if ( filename != "" && library != "C++" )
  {
    if ( device == "troff" )
    {
      print( "\\f(CO#include <" library "/" filename ">" )
      print( "" )
    }
    else if ( device == "mml" )
    {
      print( "<Family Courier>#include \\<" library "/" filename "\\>" )
      print( "<HardReturn>" )
    }
  }
  bases = ""
  baseindex = 0
  numfields = 0
  for ( i=0; i<=num; i++ )
  {
    numfields = split( lines[i], line, "\n" )
    if ( line[1] ~ /(CLASS|UNION|STRUCT)/ && line[2] == class )
    {
      type = lower[line[1]]
      baseindex = 3
      break
    }
    else if ( \
     line[1] == "TEMPLATE" && \
     line[2] ~ /(class|union|struct)/ && \
     line[3] == class \
    )
    {
      baseindex = 5
      type = "template"
      break
    }
  }
  if ( baseindex != 0 )
  {
    for ( i=baseindex; i<=numfields; i++ )
    {
      if ( bases == "" )
	bases = line[i]
      else
	bases = bases ", " line[i]
    }
  }

  if ( type == "template" )
  {
    for ( i=0; i<=num; i++ )
    {
      split( lines[i], line, "\n" )
      if ( line[1] == "TEMPLATE" && line[3] == class )
	break
    }
    if ( device == "troff" )
      printf( "template%s\n",line[4] )
    else if ( device == "mml" )
    {
      sub( "<", "\\<", line[4] )
      sub( ">", "\\>", line[4] )
      printf( "<HardReturn>template%s\n",line[4] )
    }
    typestat = line[2] " " line[3]
    if ( device == "troff" )
      printf( "%s", typestat )
    else if ( device == "mml" )
      printf( "<HardReturn>%s", typestat )
  }
  else
  {
    typestat = type " " class
    if ( device == "mml" )
      printf( "<HardReturn>" )
    printf( "%s", typestat )
  }

  if ( bases != "" )
  {
    typestat = typestat " : "
    printf( " : " )
    if ( length(typestat) + length(bases) > 70 )
    {
      print( "" )
      if ( device == "mml" )
	printf( "<HardReturn>" )
      narg = 0
      while ( match( bases, "[^<>,# ]*<[^<>]*>" ) )
      {
	narg++
	arg = substr( bases, RSTART, RLENGTH )
	sub( "[^<>,# ]*<[^<>]*>", "#" narg, bases )
	bargs["#" narg] = arg
	# Need the following to stop resubstitution of the pattern matched
	# back into the string.
	gsub( "&", "\\\\&", bargs["#" narg] )
      }
      numbases = split( bases, base, "," )
      for ( m=1; m<=numbases; m++ )
      {
	while ( match( base[m], "#[0-9]+" ) )
	{
	  arg = substr( base[m], RSTART, RLENGTH )
	  sub( arg, bargs[arg], base[m] )
	}
	sub( "^[\t ]*", "", base[m] )
	if ( device == "mml" )
	{
	  gsub( "<", "\\<", base[m] )
	  gsub( ">", "\\>", base[m] )
	}
	if ( device == "troff" )
	  printf( "  %s", base[m] )
	else
	  printf( "\\ \\ %s", base[m] )
	if ( m == numbases )
	  print( "" )
	else
	{
	  print( "," )
	  if ( device == "mml" )
	    printf( "<HardReturn>" )
	}
      }
    }
    else
    {
      if ( device == "mml" )
      {
	gsub( "<", "\\<", bases )
	gsub( ">", "\\>", bases )
      }
      print( bases )
    }
  }
  else
  {
    printf( "\n" )
  }
  if ( device == "troff" )
    print( "{\\fP" )
  else if ( device == "mml" )
    print( "<HardReturn>{<Family Times>")
  if ( access["public"] == "on" )
  {
    start = 0
    for ( i=0; i<=num; i++ )
    {
      split( lines[i], line, "\n" )
      if ( ( line[1] == "MEMBER" ||
	     line[1] == "FUNC" ||
	     line[1] == "TYPEDEF" ||
	     line[1] == "FRIEND" ) &&
	   line[2] == class &&
	   line[3] == "public" )
      {
	if ( start == 0 )
	{
	  start = 1
	  if ( device == "troff" )
	    print( "  \\f(COpublic:\\fP")
	  else if ( device == "mml" )
	    print( "<HardReturn><Family Courier>\\ \\ public:<Family Times>" )
	}
	if ( device == "troff" )
	  outputPrototype( "    " line[4] )
	else
	{
	  printf( "<HardReturn>" )
	  outputPrototype( "    " line[4] )
	}
      }
    }
  }
  if ( access["protected"] == "on" )
  {
    start = 0
    for ( i=0; i<=num; i++ )
    {
      split( lines[i], line, "\n" )
      if ( ( line[1] == "MEMBER" ||
	     line[1] == "FUNC" ||
	     line[1] == "TYPEDEF" ||
	     line[1] == "FRIEND" ) &&
	   line[2] == class &&
	   line[3] == "protected" )
      {
	if ( start == 0 )
	{
	  start = 1
	  if ( device == "troff" )
	    print( "  \\f(COprotected:\\fP")
	  else if ( device == "mml" )
	    print("<HardReturn><Family Courier>\\ \\ protected:<Family Times>" )
	}
	if ( device == "troff" )
	  outputPrototype( "    " line[4] )
	else
	{
	  printf( "<HardReturn>" )
	  outputPrototype( "    " line[4] )
	}
      }
    }
  }
  if ( access["private"] == "on" )
  {
    start = 0
    for ( i=0; i<=num; i++ )
    {
      split( lines[i], line, "\n" )
      if ( ( line[1] == "MEMBER" ||
	     line[1] == "FUNC" ||
	     line[1] == "TYPEDEF" ||
	     line[1] == "FRIEND" ) &&
	   line[2] == class &&
	   line[3] == "private" )
      {
	if ( start == 0 )
	{
	  start = 1
	  if ( device == "troff" )
	    print( "  \\f(COprivate:\\fP")
	  else if ( device == "mml" )
	    print("<HardReturn><Family Courier>\\ \\ private:<Family Times>" )
	}
	if ( device == "troff" )
	  outputPrototype( "    " line[4] )
	else
	{
	  printf( "<HardReturn>" )
	  outputPrototype( "    " line[4] )
	}
      }
    }
  }
  if ( device == "troff" )
  {
    print( "\\f(CO};" )
    print( "\\fP\n.fi" )
  }
  else if ( device == "mml" )
  {
    print( "<HardReturn><Family Courier>};<Family Times>" )
  }
}

#
# Print copyright holders. This must break the line before each copyright
# holder.
#
pass == "2" && $1 == "print" && $2 == "copyright" {
  for ( i=0; i<=num; i++ )
  {
    n = split( lines[i], line, "\n" )
    if ( line[1] == "INFO" &&
	 line[2] == "GLOBAL" &&
	 line[3] == "COPYRIGHT" )
    {
      if ( device == "troff" )
      {
	print( ".PP" )
	print( ".SH COPYRIGHT" )
      }
      else if ( device == "mml" )
      {
	print( "" )
	print( "<Heading2>" )
	print( "COPYRIGHT" )
	printf( "<Body>" )
	printf( "<Plain>" )
	printf( "<LeftIndent 0in>" )
	printf( "<FirstIndent 0in>" )
	printf( "<SpaceAfter 8pt>" )
      }
      start = 4
      for ( k=start; k<=n; k++ )
      {
	if ( k != start && line[k] ~ "[\t ]*Copyright" )
	{
	  if ( device == "troff" )
	    print( ".br" )
	  else if ( device == "mml" )
	    printf( "<HardReturn>" )
	}
	outputLine( line[k] )
      }
      break
    }
  }
}

#
# Print out any sections which have not been previously printed out and
# which are not hidden.
#
pass == "1" && $1 == "print" && $2 == "section" && $3 == "*" {
  next
}

pass == "2" && $1 == "print" && $2 == "section" && $3 == "*" {
  for ( i=0; i<=num; i++ )
  {
    n = split( lines[i], line, "\n" )
    if ( line[1] == "INFO" &&
	 ( line[2] == "GLOBAL" ||
	   ( line[2] == "HDR" && line[4] == class ) ) )
    {
      if ( line[3] in sections )
	continue
      else
      {
	section = line[3]
	sections[section] = ""
	outputSection()
      }
    }
  }
}

#
# Print out a section.
# On pass 1 only record the fact that this section will be printed.
#
pass == "1" && $1 == "print" && $2 == "section" {
  section = $3
  for ( i=4; i<=NF; i++ )
    section = section " " $i
  sections[section] = ""
}

#
# Print out a section.
# On pass 2 actually print it out.
#
pass == "2" && $1 == "print" && $2 == "section" {
  section = $3
  for ( i=4; i<=NF; i++ )
    section = section " " $i
  found = 0
  for ( i=0; i<=num; i++ )
  {
    n = split( lines[i], line, "\n" )
    if ( line[1] == "INFO" &&
	 ( line[2] == "GLOBAL" ||
	   ( line[2] == "HDR" && line[4] == class ) ) &&
	 line[3] == section )
    {
      found = 1
      break
    }
  }
  if ( found == 1 )
    outputSection()
}

#
# Print out any contracts which have not previously been printed, and
# which are not hidden.
#
# If there are no contracts in the class, then print out sections for the
# private, protected and public members; thats if they are not being hidden.
#
pass == "1" && $1 == "print" && $2 == "contract" && $3 == "\*" {
  next
}

pass == "2" && $1 == "print" && $2 == "contract" && $3 == "\*" {
  for ( i=0; i<=num; i++ )
  {
    n = split( lines[i], line, "\n" )
    if ( line[1] == "INFO" &&
	 line[2] == "BODY" &&
	 line[4] == class )
    {
      if ( line[3] in contracts )
	continue
      else if ( access[line[5]] == "on" )
      {
	contract = line[3]
	contracts[contract] = ""
	outputContract()
      }
    }
  }
  i = 0
  for ( j in contracts )
    i++
  if ( i == 0 )
  {
    if ( access["public"] == "on" )
      outputClassSection( "public" )
    if ( access["protected"] == "on" )
      outputClassSection( "protected" )
    if ( access["private"] == "on" )
      outputClassSection( "private" )
  }
}

#
# Print out a particular contract, regardless of whether it is hidden or
# not.
# On pass 1 we only note that the contract will be printed.
#
pass == "1" && $1 == "print" && $2 == "contract" {
  contract = $3
  for ( i=4; i<=NF; i++ )
    contract = contract " " $i
  contracts[contract] = ""
}

pass == "2" && $1 == "print" && $2 == "contract" {
  contract = $3
  for ( i=4; i<=NF; i++ )
    contract = contract " " $i
  found = 0
  for ( i=0; i<=num; i++ )
  {
    n = split( lines[i], line, "\n" )
    if ( line[1] == "INFO" &&
	 line[2] == "BODY" &&
	 line[4] == class &&
	 line[3] == section )
    {
      found = 1
      break
    }
  }
  if ( found == 1 )
    outputContract()
}

#
# Output a particular section of a class.
# This ignores whether that part is hidden.
#
pass == "2" && $1 == "print" && $3 in access {
  outputClassSection( $3 )
}

#
# Actually prints out a particular section of a class.
#
function outputClassSection( part )
{
  start = 0
  for ( i=0; i<=num; i++ )
  {
    n = split( lines[i], line, "\n" )
    if ( ( line[1] == "FUNC" ||
	   line[1] == "MEMBER" ||
	   line[1] == "TYPEDEF" ||
           line[1] == "FRIEND" ) &&
	 line[2] == class &&
	 line[3] == part )
    {
      if ( start == 0 )
      {
	start = 1
	if ( device == "troff" )
	{
	  print( ".PP" )
	  print( ".SH \"" upper[line[3]] " MEMBERS\"" )
	}
	else if ( device == "mml" )
	{
	  print( "" )
	  print( "<Heading2>" )
	  print( upper[line[3]] " MEMBERS" )
	  printf( "<Body>" )
	  printf( "<Plain>" )
	  printf( "<LeftIndent 0in>" )
	  printf( "<FirstIndent 0in>" )
	  printf( "<SpaceAfter 8pt>" )
	}
      }
      if ( device == "troff" )
      {
	print( ".nf" )
      }
      else if ( device == "mml" )
      {
	printf( "<Alignment l>" )
      }
      outputPrototype( line[4] )
      if ( device == "troff" )
      {
	print( ".fi" )
	print( ".RS 0.25i" )
      }
      else if ( device == "mml" )
      {
	printf( "<Alignment lr>" )
	printf( "<LeftIndent 0.1in>" )
	printf( "<FirstIndent 0.1in>" )
      }
      for ( l=5; l<=n; l++ )
	outputLine( line[l] )
      if ( device == "troff" )
      {
	print( ".RE" )
	print( "" )
      }
      else if ( device == "mml" )
      {
	printf( "<LeftIndent 0in>" )
	printf( "<FirstIndent 0in>" )
      }
    }
  }
}

#
# Once a contract has been found, this prints it out.
#
function outputContract() {
  if ( device == "troff" )
  {
    print( ".PP" )
    print( ".SH \"" contract "\"" )
  }
  else if ( device == "mml" )
  {
    print( "" )
    print( "<Heading2>" )
    print( contract )
    printf( "<Body>" )
    printf( "<Plain>" )
    printf( "<LeftIndent 0in>" )
    printf( "<FirstIndent 0in>" )
    printf( "<SpaceAfter 8pt>" )
  }

  for ( k=6; k<=n; k++ )
    outputLine( line[k] )

  # print( "" )
  for ( k=i+1; k<=num; k++ )
  {
    n = split( lines[k], line, "\n" )
    if ( ( line[1] == "INFO" &&
	   line[2] == "BODY" &&
	   line[4] == class ) ||
	 ( line[1] == "END" &&
	   line[2] == class ) ||
	 ( line[1] == "ACCESS" &&
	   line[2] == class ) )
    {
      break
    }
    else if ( ( line[1] == "FUNC" ||
		line[1] == "MEMBER" ||
		line[1] == "TYPEDEF" ||
		line[1] == "FRIEND" ) &&
		line[2] == class )
    {
      if ( device == "troff" )
      {
        print( ".nf" )
      }
      else if ( device == "mml" )
      {
        printf( "<Alignment l>" )
      }
      outputPrototype( line[4] )
      if ( device == "troff" )
      {
        print( ".fi" )
        print( ".RS 0.25i" )
      }
      else if ( device == "mml" )
      {
        printf( "<Alignment lr>" )
        printf( "<LeftIndent 0.1in>" )
        printf( "<FirstIndent 0.1in>" )
      }
      for ( l=5; l<=n; l++ )
	outputLine( line[l] )
      if ( device == "troff" )
      {
        print( ".RE" )
        print( "" )
      }
      else if ( device == "mml" )
      {
        printf( "<LeftIndent 0in>" )
        printf( "<FirstIndent 0in>" )
      }
    }
    else if ( line[1] == "COMMENT" &&
		line[2] == "BODY" &&
		line[3] == class )
    {
      for ( l=5; l<=n; l++ )
	outputLine( line[l] )
      # print( "" )
    }
  }
}

#
# Once a section has been found, this prints it out.
#
function outputSection()
{
  if ( line[2] == "GLOBAL" )
    start = 4
  else
    start = 5

  if ( device == "troff" )
  {
    print( ".PP" )
    print( ".SH \"" section "\"" )
  }
  else if ( device == "mml" )
  {
    print( "" )
    print( "<Heading2>" )
    print( section )
    if ( line[start] != "" )
    {
      printf( "<Body>" )
      printf( "<Plain>" )
      printf( "<LeftIndent 0in>" )
      printf( "<FirstIndent 0in>" )
      printf( "<Alignment lr>" )
      printf( "<SpaceAfter 8pt>" )
      printf( "<Family Times>" )
    }
  }

  blankLine = 0
  for ( k=start; k<=n; k++ )
    outputLine( line[k] )
}

#
# Removes comment delimiter from start of line.
#
function removeComment( line )
{
  sub( "^// ?", "", line )
  return line
}

#
# Output a line. This checks for various formatting requests and will
# appropriately expand them.
#
function outputLine( line )
{
  line = removeComment( line )
  if ( inComment == 0 )
  {
    if ( line ~ "^= BEGIN<CODE>" )
    {
      inCode = 1
      if ( device == "troff" )
      {
	print( ".RS 0.25i" )
	print( ".nf\n\\f(CO" )
      }
      else if ( device == "mml" )
      {
	printf( "<Plain>" )
	printf( "<LeftIndent 0.2in>" )
	printf( "<FirstIndent 0.2in>" )
	printf( "<Alignment l>" )
	printf( "<Family Courier>" )
      }
    }
    else if ( line ~ "^= END<CODE>" )
    {
      inCode = -1
      if ( device == "troff" )
      {
	print( "\\fP\n.fi" )
	print( ".RE" )
      }
      else if ( device == "mml" )
      {
	printf( "<LeftIndent 0in>" )
	printf( "<FirstIndent 0in>" )
	printf( "<Alignment lr>" )
	printf( "<Family Times>")
      }
    }
    else if ( line ~ "^= BEGIN<COMMENT>" )
      inComment = 1
    else if ( line ~ "^= BEGIN<INDENT>" )
    {
      if ( device == "troff" )
	print( ".RS 0.25i" )
      else if ( device == "mml" )
      {
	printf( "<LeftIndent 0.2in>" )
	printf( "<FirstIndent 0.2in>" )
      }
    }
    else if ( line ~ "^= END<INDENT>" )
    {
      if ( device == "troff" )
	print( ".RE" )
      else if ( device == "mml" )
      {
	printf( "<LeftIndent 0in>" )
	printf( "<FirstIndent 0in>" )
      }
    }
    else if ( line ~ "^= BEGIN<NOFILL>" )
    {
      inNoFill = 1
      if ( device == "troff" )
	print( ".nf" )
      else if ( device == "mml" )
	printf( "<Alignment l>" )
    }
    else if ( line ~ "^= END<NOFILL>" )
    {
      inNoFill = 0
      if ( device == "troff" )
	print( ".fi" )
      else if ( device == "mml" )
	printf( "<Alignment lr>" )
    }
    else
    {
      if ( inCode > 0 )
      {
	if ( device == "mml" )
	{
	  gsub( " ", "\\ ", line )
	  gsub( "<", "\\<", line )
	  gsub( ">", "\\>", line )
	  if ( inCode > 1 )
	    line = "<HardReturn>" line
	}
	print( line )
	inCode = inCode + 1
      }
      else 
      {
	gsub( "\\\\]>", "]\\fP", line )
	gsub( "\\\\}>", "]\\fP", line )
	gsub( "([^\\\\]|^)<\\[", "&<\\fB", line )
	gsub( "[^\\\\]\\]>", "&>\\fP", line )
	gsub( "<\\[<", "", line )
	gsub( "\\]>>", "", line )
	gsub( "([^\\\\]|^)<\{", "&<\\fI", line )
	gsub( "[^\\\\]}>", "&>\\fP", line )
	gsub( "<\{<", "", line )
	gsub( "}>>", "", line )
	gsub( "([^\\\\]|^)<", "&<\\f(CO", line )
	gsub( "[^\\\\]>", "&>\\fP", line )
	gsub( "<<", "", line )
	gsub( ">>", "", line )
	gsub( "\\\\<", "<", line )
	gsub( "\\\\>", ">", line )

	# Some versions of gawk do not like these
	# substitutions, the substitutions above, for
	# some reason already putting it into what
	# is required. As far as is known, it is a
	# bug in gawk, nawk behaves okay.

	if ( backslashworkaround == 0 )
	{
	  gsub( "\\\\", "\\\\", line )
	  gsub( "\\\\fB", "fB", line )
	  gsub( "\\\\fI", "fI", line )
	  gsub( "\\\\fP", "fP", line )
	  gsub( "\\\\f\\(CO", "f(CO", line )
	}

	if ( device == "mml" )
	{
	  if ( inNoFill > 0 )
	    gsub( " ", "\\ ", line )

	  gsub( "<", "\\<", line )
	  gsub( ">", "\\>", line )
	  gsub( "\\\\fP", "<Family Times><Plain>", line )
	  gsub( "\\\\fB", "<Bold>", line )
	  gsub( "\\\\fI", "<Italic>", line )
	  gsub( "\\\\f\\(CO", "<Family Courier>", line )

	  if ( inNoFill > 0 )
	  {
	    if ( inNoFill > 1 )
	      line = "<HardReturn>" line
	    inNoFill = inNoFill + 1
	  }
	}

	if ( line == "" )
	{
	  blankLine = 1
	}
	else
	{
	  if ( blankLine == 1 )
	  {
	    if ( inCode == 0 )
	      print( "" )
	    blankLine = 0
	    if ( inCode == -1 )
	      inCode = 0
	  }
	  print( line )
	}
      }
    }
  }
  else
  {
    if ( line ~ "^= END<COMMENT>" )
      inComment = 0
  }
}

#
# Searches for the 'LIBRARY' section and sets 'library' to the value found.
#
function findLibrary()
{
  for ( i=0; i<=num; i++ )
  {
    split( lines[i], line, "\n" )
    if ( line[1] == "INFO" &&
	 line[2] == "GLOBAL" &&
	 line[3] == "LIBRARY" )
    {
      library = removeComment( line[4] )
      break
    }
  }
  if ( library == "" )
    library = "C++"
}

#
# Searches for the 'FILENAME' section and sets 'filename' to the value found.
#
function findFilename()
{
  for ( i=0; i<=num; i++ )
  {
    split( lines[i], line, "\n" )
    if ( line[1] == "INFO" &&
	 line[2] == "GLOBAL" &&
	 ( line[3] == "FILENAME" ||
	   line[3] == "RCSID" ||
	   line[3] == "SCCSID" ) )
    {
      filename = removeComment( line[4] )
      if ( line[3] == "RCSID" )
      {
	sub( "^.*\\$Id:[\t ]*", "", filename )
	sub( ",v.*$", "", filename )
      }
      else if ( line[3] == "SCCSID" )
      {
	sub( "^.*@\\(#\\)", "", filename )
	sub( "[\t ]+.*$", "", filename )
      }
      break
    }
  }
}

#
# Outputs a prototype, wrapping it if required.
#
function outputPrototype( prototype )
{
  sub( "[\t ]*$", "", prototype )
  if ( prototype !~ "^.*;$" )
    prototype = prototype ";"

  if ( device == "troff" )
    width = 70
  else if ( device  == "mml" )
    width = 60

  if ( length(prototype) > width )
  {
    if ( device == "mml" )
    {
      gsub( ">", "\\>", prototype )
      gsub( "<", "\\<", prototype )
    }
    if ( ( funcptr = match( prototype, "^[^(]*\\([^)]*\\)\\(" ) ) !~ "0" )
      match( prototype, "^[^(]*\\([^)]*\\)" )
    else
      match( prototype, "^[^(]*\\(" )
    funcname = substr( prototype, RSTART, RLENGTH )
    if ( prototype ~ "^[^(]*\\(\\)[\t ]*\\(" )
    {
      funcname = funcname ")("
      sub( "^[^(]*\\(\\)[\t ]*\\(", "", prototype )
    }
    else
    {
      if ( funcptr ~ "0" )
	sub( "^[^(]*\\(", "", prototype )
      else
	sub( "^[^(]*\\([^)]*\\)\\(", "", prototype )
    }
    match( prototype, "\\)([\t ]*)?(=[\t ]*0|const[\t ]*(=[\t ]*0)?)?;[\t ]*" )
    functail = substr( prototype, RSTART, RLENGTH )
    sub( "\\)([\t ]*)?(=[\t ]*0|const[\t ]*(=[\t ]*0)?)?;.*$", "", prototype )
    sub( "\\($", "", funcname )
    match( funcname, "^[\t ]*" )
    space = substr( funcname, RSTART, RLENGTH )
    if ( device == "troff" )
    {
      printf( "\\f(CO" funcname )
      printf( "(\n" )
    }
    else if ( device == "mml" )
    {
      gsub( " ", "\\ ", space )
      gsub( " ", "\\ ", funcname )
      printf( "<Family Courier>%s(\n<HardReturn>", funcname )
    }
    narg = 0
    while ( match( prototype, "[^<>,#]*<[^<>]*>" ) )
    {
      narg++
      arg = substr( prototype, RSTART, RLENGTH )
      sub( "[^<>,#]*<[^<>]*>", "#" narg, prototype )
      fargs["#" narg] = arg
      # Need the following to stop resubstitution of the pattern matched
      # back into the string.
      gsub( "&", "\\\\&", fargs["#" narg] )
    }
    numargs = split( prototype, args, "," )
    for ( m=1; m<=numargs; m++ )
    {
      while ( match( args[m], "#[0-9]+" ) )
      {
	arg = substr( args[m], RSTART, RLENGTH )
	sub( arg, fargs[arg], args[m] )
      }
      sub( "[\t ]*", "", args[m] )
      printf( "%s  %s", space, args[m] )
      if ( m == numargs )
      {
	if ( device == "troff" )
	  print( "" )
	else if ( device == "mml" )
	  printf( "\n<HardReturn>" )
      }
      else
      {
	if ( device == "troff" )
	  print( "," )
	else if ( device == "mml" )
	  printf( ",\n<HardReturn>" )
      }
    }
    if ( device == "troff" )
      print( space functail "\\fP" )
    else if ( device == "mml" )
      print( space functail "<Family Times>" )

  }
  else
  {
    if ( device == "mml" )
    {
      gsub( ">", "\\>", prototype )
      gsub( "<", "\\<", prototype )
    }
    if ( device == "troff" )
      print( "\\f(CO" prototype "\\fP" )
    else if ( device == "mml" )
    {
      gsub( " ", "\\ ", prototype )
      print( "<Family Courier>" prototype "<Family Times>" )
    }
  }
}
