/*
// ============================================================================
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 OTC LIMITED
//
// ============================================================================
*/

#include "cell.hh"
#include "prepost.hh"

#include <OTC/debug/tracer.hh>

#include <string.h>
#include <strstream.h>

/* ------------------------------------------------------------------------- */
UQ_Cell::UQ_Cell(UQ_CellId const& anId, OTC_Set<UQ_FreqId> const& theFreqs)
 : myId(anId), myFreqs(theFreqs)
{
  OTCLIB_MARKBLOCK(UQ,"Cell::Cell()");

  UQ_INVARIANT("Cell::Cell()");
}

/* ------------------------------------------------------------------------- */
UQ_Cell::~UQ_Cell()
{
  OTCLIB_MARKBLOCK(UQ,"Cell::~Cell()");
}

/* ------------------------------------------------------------------------- */
int UQ_Cell::page(UQ_MobileId const& mid)
{
  OTCLIB_MARKBLOCK(UQ,"Cell::page()");

  OTC_Boolean pre1 = !(myAccess.contains(mid) || myAlloc.containsItem(mid));
  UQ_PRE("Cell::page",pre1);

  UQ_INVARIANT("Cell::page()");

  UQ_SUCCESS("Cell::page()");
}

/* ------------------------------------------------------------------------- */
int UQ_Cell::bePagedSeized(UQ_MobileId const& mid)
{
  OTCLIB_MARKBLOCK(UQ,"Cell::bePagedSeized()");

  OTC_Boolean pre1 = myAccess.isEmpty();
  UQ_PRE("Cell::bePagedSeized()",pre1);

  myAccess.add(mid);

  OTC_Boolean post1 = myAccess.contains(mid);
  UQ_POST("Cell::bePagedSeized()",post1);

  UQ_INVARIANT("Cell::bePagedSeized()");

  UQ_SUCCESS("Cell::bePagedSeized()");
}

/* ------------------------------------------------------------------------- */
int UQ_Cell::beCallSeized(UQ_MobileId const& mid, UQ_PhoneId const& /* pid */)
{
  OTCLIB_MARKBLOCK(UQ,"Cell::beCallSeized()");

  OTC_Boolean pre1 = myAccess.isEmpty();
  UQ_PRE("Cell::beCallSeized()",pre1);

  myAccess.add(mid);

  OTC_Boolean post1 = myAccess.contains(mid);
  UQ_POST("Cell::beCallSeized()",post1);

  UQ_INVARIANT("Cell::beCallSeized()");

  UQ_SUCCESS("Cell::beCallSeized()");
}

/* ------------------------------------------------------------------------- */
int UQ_Cell::tune(UQ_MobileId const& mid, UQ_FreqId& fid)
{
  OTCLIB_MARKBLOCK(UQ,"Cell::tune()");

  OTC_Boolean pre1 = (myAccess.population() == 1);
  OTC_Boolean pre2 = (myAlloc.population() < myFreqs.population());
  UQ_PRE("Cell::tune()",pre1 && pre2);

  OTC_Boolean pre3 = myAccess.contains(mid);
  UQ_PRE("Cell::tune()",pre3);

  myAccess.removeAll();

  OTC_Iterator<UQ_FreqId> iter1 = 0;
  iter1 = myFreqs.items();
  fid = iter1.item();
  while (myAlloc.containsKey(fid))
  {
    iter1.next();
    fid = iter1.item();
  }

  myAlloc.add(fid,mid);

  OTC_Boolean post1 = myAccess.isEmpty();
  UQ_POST("Cell::tune()",post1);

  UQ_INVARIANT("Cell::tune()");

  UQ_SUCCESS("Cell::tune()");
}

/* ------------------------------------------------------------------------- */
int UQ_Cell::handOff(UQ_MobileId const& mid)
{
  OTCLIB_MARKBLOCK(UQ,"Cell::handOff()");

  OTC_Boolean pre1 = !myAlloc.isEmpty();
  UQ_PRE("Cell::handOff()",pre1);

  OTC_Boolean pre2 = myAlloc.containsItem(mid);
  UQ_PRE("Cell::handOff()",pre2);

  myAlloc.removeItem(mid);

  UQ_INVARIANT("Cell::handOff()");

  UQ_SUCCESS("Cell::handOff()");
}

/* ------------------------------------------------------------------------- */
int UQ_Cell::handOn(UQ_MobileId const& mid, UQ_FreqId& fid)
{
  OTCLIB_MARKBLOCK(UQ,"Cell::handOn()");

  OTC_Boolean pre1 = myAccess.isEmpty();
  UQ_PRE("Cell::handOn()",pre1);

  OTC_Boolean pre2 = !myAlloc.containsItem(mid);
  UQ_PRE("Cell::handOn()",pre2);

  OTC_Boolean pre3 = myAlloc.population() != myFreqs.population();
  UQ_PRE("Cell::handOn()",pre3);

  OTC_Iterator<UQ_FreqId> iter1 = 0;
  iter1 = myFreqs.items();
  fid = iter1.item();
  while (myAlloc.containsKey(fid))
  {
    iter1.next();
    fid = iter1.item();
  }

  myAlloc.add(fid,mid);

  OTC_Boolean post1 = myAlloc.containsKey(fid);
  UQ_POST("Cell::handOn()",post1);

  UQ_INVARIANT("Cell::handOn()");

  UQ_SUCCESS("Cell::handOn()");
}

/* ------------------------------------------------------------------------- */
int UQ_Cell::disconnect(UQ_MobileId const& mid)
{
  OTCLIB_MARKBLOCK(UQ,"Cell::disconnect()");

  OTC_Boolean pre1 = !(myAlloc.isEmpty() && myAccess.isEmpty());
  UQ_PRE("Cell::disconnect()",pre1);

  OTC_Boolean pre2 = (myAccess.contains(mid) || myAlloc.containsItem(mid));
  UQ_PRE("Cell::disconnect()",pre2);

  if (myAlloc.containsItem(mid))
    myAlloc.removeItem(mid);

  if (myAccess.contains(mid))
    myAccess.remove(mid);

  UQ_INVARIANT("Cell::disconnect()");

  UQ_SUCCESS("Cell::disconnect()");
}

/* ------------------------------------------------------------------------- */
OTC_Boolean UQ_Cell::invariant() const
{
  OTCLIB_MARKBLOCK(UQ,"Cell::invariant()");

  OTC_Boolean c1 = OTCLIB_TRUE;
  OTC_Set<UQ_FreqId> set1;
  OTC_Iterator<UQ_FreqId> iter1 = 0;
  iter1 = myAlloc.keys();
  for (iter1.reset(); iter1.isValid(); iter1.next())
    set1.add(iter1.item());
  iter1 = set1.items();
  for (iter1.reset(); iter1.isValid(); iter1.next())
  {
    if (!myFreqs.contains(iter1.item()))
    {
      c1 = OTCLIB_FALSE;
      break;
    }
  }

  OTC_Boolean c2 = myAccess.population() <= 1;

  OTC_Boolean c3 = OTCLIB_FALSE;
  OTC_Set<UQ_MobileId> set2;
  OTC_Iterator<UQ_MobileId> iter2 = 0;
  iter2 = myAlloc.items();
  for (iter2.reset(); iter2.isValid(); iter2.next())
    set2.add(iter2.item());
  OTC_Set<UQ_MobileId> set3;
  iter2 = set2.items();
  for (iter2.reset(); iter2.isValid(); iter2.next())
    if (myAccess.contains(iter2.item()))
      if (!set3.contains(iter2.item()))
        set3.add(iter2.item());
  c3 = set3.isEmpty();

  return c1 && c2 && c3;
}

/* ------------------------------------------------------------------------- */
void UQ_Cell::dump(ostream& outs) const
{
  outs << "Cell::dump {" << endl;
  outs << "  id = " << id() << endl;
  if (myFreqs.population() != 0)
  {
    outs << "  freqs =";
    OTC_Iterator<UQ_FreqId> iter(myFreqs.items());
    for (iter.reset(); iter.isValid(); iter.next())
      outs << " " << iter.item();
    outs << endl;
  }
  if (myAccess.population() != 0)
  {
    outs << "  access =";
    OTC_Iterator<UQ_MobileId> iter(myAccess.items());
    for (iter.reset(); iter.isValid(); iter.next())
      outs << " " << iter.item();
    outs << endl;
  }
  if (myAlloc.population() != 0)
  {
    outs << "  alloc =";
    OTC_Iterator<UQ_FreqId> iter(myAlloc.keys());
    for (iter.reset(); iter.isValid(); iter.next())
    {
      outs << " (" << iter.item() << ","
	  << myAlloc.item(iter.item()) << ")";
    }
    outs << endl;
  }
  outs << "}" << endl;
}

/* ------------------------------------------------------------------------- */
