/*
// ============================================================================
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 OTC LIMITED
//
// ============================================================================
*/

#include "mobile.hh"
#include "prepost.hh"

#include <OTC/debug/tracer.hh>

#include <string.h>
#include <strstream.h>

/* ------------------------------------------------------------------------- */
static char const* NAMES_State[] =
{
  "IDLE",
  "PAGED",
  "PAGED_ACCESS",
  "CALL_ACCESS",
  "RINGING",
  "OFFHOOK"
};

/* ------------------------------------------------------------------------- */
UQ_Mobile::UQ_Mobile(UQ_MobileId const& anId)
 : myId(anId), myState(IDLE)
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::Mobile()");

  UQ_INVARIANT("Mobile::Mobile()");
}

/* ------------------------------------------------------------------------- */
UQ_Mobile::UQ_Mobile(UQ_Mobile const& theMobile)
 : myId(theMobile.myId), myState(theMobile.myState), myFreq(theMobile.myFreq)
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::Mobile()");

  UQ_INVARIANT("Mobile::Mobile()");
}

/* ------------------------------------------------------------------------- */
UQ_Mobile::~UQ_Mobile()
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::~Mobile()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::cell(UQ_CellId& cid) const
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::cell()");

  OTC_Boolean pre1 = (myCell.population() == 1);
  UQ_PRE("Mobile::cell()",pre1);

  cid = myCell.pickItem();

  UQ_INVARIANT("Mobile::cell()");

  UQ_SUCCESS("Mobile::cell()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::locate(UQ_CellId& cid)
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::locate()");

  OTC_Boolean pre1 = (state() == PAGED || state() == IDLE);
  UQ_PRE("Mobile::locate()",pre1);

  myCell.removeAll();
  cid = UQ_CellId::any();
  myCell.add(cid);

  UQ_INVARIANT("Mobile::locate()");

  UQ_SUCCESS("Mobile::locate()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::bePaged()
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::bePaged()");

  OTC_Boolean pre1 = (state() == IDLE);
  UQ_PRE("Mobile::bePaged()",pre1);

  myState = PAGED;

  OTC_Boolean post1 = (state() == PAGED);
  UQ_POST("Mobile::bePaged()",post1);

  UQ_INVARIANT("Mobile::bePaged()");

  UQ_SUCCESS("Mobile::bePaged()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::pagedSeize()
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::pagedSeize()");

  OTC_Boolean pre1 = (state() == PAGED);
  UQ_PRE("Mobile::pagedSeize()",pre1);

  OTC_Boolean pre2 = (myCell.population() == 1);
  UQ_PRE("Mobile::pagedSeize()",pre2);

  myState = PAGED_ACCESS;

  OTC_Boolean post1 = (state() == PAGED_ACCESS);
  UQ_POST("Mobile::pagedSeize()",post1);

  UQ_INVARIANT("Mobile::pagedSeize()");

  UQ_SUCCESS("Mobile::pagedSeize()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::callSeize(UQ_PhoneId& pid)
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::callSeize()");

  OTC_Boolean pre1 = (state() == IDLE);
  UQ_PRE("Mobile::callSeize()",pre1);

  pid = UQ_PhoneId::any();
  myState = CALL_ACCESS;

  OTC_Boolean post1 = (state() == CALL_ACCESS);
  UQ_POST("Mobile::callSeize()",post1);

  UQ_INVARIANT("Mobile::callSeize()");

  UQ_SUCCESS("Mobile::callSeize()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::beTuned(UQ_FreqId const& fid)
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::beTuned()");

  OTC_Boolean pre1 = (state() == PAGED_ACCESS || state() == CALL_ACCESS);
  UQ_PRE("Mobile::beTuned()",pre1);

  myFreq.add(fid);
  if (state() == PAGED_ACCESS)
    myState = RINGING;
  if (state() == CALL_ACCESS)
    myState = OFFHOOK;

  OTC_Boolean post1 = (state() == RINGING || state() == OFFHOOK);
  UQ_POST("Mobile::beTuned()",post1);

  UQ_INVARIANT("Mobile::beTuned()");

  UQ_SUCCESS("Mobile::beTuned()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::answer()
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::answer()");

  OTC_Boolean pre1 = (state() == RINGING);
  UQ_PRE("Mobile::answer()",pre1);

  myState = OFFHOOK;

  OTC_Boolean post1 = (state() == OFFHOOK);
  UQ_POST("Mobile::answer()",post1);

  UQ_INVARIANT("Mobile::answer()");

  UQ_SUCCESS("Mobile::answer()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::beHandedOver(UQ_CellId const& cid, UQ_FreqId const& fid)
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::beHandedOver()");

  OTC_Boolean pre1 = !myFreq.contains(fid);
  OTC_Boolean pre2 = (state() == OFFHOOK);
  UQ_PRE("Mobile::beHandedOver()",pre1 && pre2);

  myFreq.removeAll();
  myFreq.add(fid);
  myCell.removeAll();
  myCell.add(cid);

  OTC_Boolean post1 = (state() == OFFHOOK);
  UQ_POST("Mobile::beHandedOver()",post1);

  UQ_INVARIANT("Mobile::beHandedOver()");

  UQ_SUCCESS("Mobile::beHandedOver()");
}

/* ------------------------------------------------------------------------- */
int UQ_Mobile::beDisconnected()
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::beDisconnected()");

  OTC_Boolean pre1 = (state() != IDLE && state() != PAGED);
  UQ_PRE("Mobile::beDisconnected()",pre1);

  myFreq.removeAll();
  myCell.removeAll();
  myState = IDLE;

  OTC_Boolean post1 = (state() == IDLE) && myFreq.isEmpty();
  UQ_POST("Mobile::beDisconnected()",post1);

  UQ_INVARIANT("Mobile::beDisconnected()");

  UQ_SUCCESS("Mobile::beDisconnected()");
}

/* ------------------------------------------------------------------------- */
OTC_Boolean UQ_Mobile::invariant() const
{
  OTCLIB_MARKBLOCK(UQ,"Mobile::invariant()");

  OTC_Boolean c1 = myFreq.population() <= 1;
  OTC_Boolean c2a = !myFreq.isEmpty() || !(myState == RINGING || myState == OFFHOOK);
  OTC_Boolean c2b = (myState == RINGING || myState == OFFHOOK) || myFreq.isEmpty();
  OTC_Boolean c2 = c2a && c2b;

  return c1 && c2;
}

/* ------------------------------------------------------------------------- */
UQ_FreqId UQ_Mobile::freq() const
{
  if (myFreq.population() == 1)
    return myFreq.pickItem();

  else
    return int(0);
}

/* ------------------------------------------------------------------------- */
void UQ_Mobile::dump(ostream& outs) const
{
  outs << "Mobile::dump {" << endl;
  outs << "  id = " << id() << endl;
  outs << "  state = " << NAMES_State[state()] << endl;
  if (myFreq.population() == 1)
    outs << "  freq = " << myFreq.pickItem() << endl;
  if (myCell.population() == 1)
    outs << "  cell = " << myCell.pickItem() << endl;
  outs << "}" << endl;
}

/* ------------------------------------------------------------------------- */
