/*
// ============================================================================
//
// = AUTHOR(S)
//     Graham Dumpleton
//
// = COPYRIGHT
//     Copyright 1991 OTC LIMITED
//
// ============================================================================
*/

#include "mps.hh"
#include "prepost.hh"

#include <OTC/debug/tracer.hh>
#include <OTC/text/string.hh>

#include <stdio.h>
#include <string.h>
#include <iostream.h>
#include <strstream.h>
#include <fstream.h>

/* ------------------------------------------------------------------------- */
UQ_MPS::UQ_MPS()
{
  OTCLIB_MARKBLOCK(UQ,"MPS::MPS()");
  
  OTC_Set<UQ_FreqId> set1;

  UQ_CellId cid1 = UQ_CellId::any();
  UQ_CellId cid2;
  char buf[64];
  do {
    cid2 = UQ_CellId::any();
    set1.removeAll();
    UQ_FreqId fid;

    sprintf(buf,"cell%d.dat",int(cid2));
    ifstream ins(buf);

    while (ins.good())
    {
      ins >> fid;
      if (!ins.fail())
	set1.add(fid);
    }
    UQ_Cell* cell1 = new UQ_Cell(cid2,set1);
    OTCLIB_ASSERT(cell1 != 0);
    myCells.add(cid2,cell1);
    ins.close();
  } while (cid1 != cid2);
  
  UQ_INVARIANT("MPS::MPS()");
}

/* ------------------------------------------------------------------------- */
UQ_MPS::UQ_MPS(UQ_MPS const& theMPS)
 : myMobiles(theMPS.myMobiles), myCells(theMPS.myCells)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::MPS()");

  UQ_INVARIANT("MPS::MPS()");
}

/* ------------------------------------------------------------------------- */
UQ_MPS::~UQ_MPS()
{
  OTCLIB_MARKBLOCK(UQ,"MPS::~MPS()");
}

/* ------------------------------------------------------------------------- */
int UQ_MPS::allCellsPage(UQ_MobileId const& mid)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::allCellsPage()");

  OTC_Iterator<UQ_Cell*> iter1(myCells.items());
  for (iter1.reset(); iter1.isValid(); iter1.next())
  {
    int call1 = iter1.item()->page(mid);
    UQ_CALL("MPS::allCellsPage()",call1);
  }

  UQ_INVARIANT("MPS::allCellsPage()");

  UQ_SUCCESS("MPS::allCellsPage()");
}

/* ------------------------------------------------------------------------- */
int UQ_MPS::page(UQ_MobileId const& mid)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::page()");

  int call1 = allCellsPage(mid);
  UQ_CALL("MPS::page()",call1);

  int call2 = myMobiles.bePaged(mid);
  UQ_CALL("MPS::page()",call2);

  UQ_INVARIANT("MPS::page()");

  UQ_SUCCESS("MPS::page()");
}

/* ------------------------------------------------------------------------- */
int UQ_MPS::pagedSeize(UQ_MobileId const& mid, UQ_CellId& cid)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::pagedSeize()");

  int call1 = myMobiles.locate(mid,cid);
  UQ_CALL("MPS::pagedSeize()",call1);

  UQ_Cell* cell1 = myCells.item(cid);
  int call2 = cell1->bePagedSeized(mid);
  UQ_CALL("MPS::pagedSeize()",call2);

  int call3 = myMobiles.pagedSeize(mid);
  UQ_CALL("MPS::pagedSeize()",call3);

  UQ_INVARIANT("MPS::pagedSeize()");

  UQ_SUCCESS("MPS::pagedSeize()");
}

/* ------------------------------------------------------------------------- */
int UQ_MPS::callSeize(UQ_MobileId const& mid, UQ_CellId& cid, UQ_PhoneId& pid)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::callSeize()");

  int call1 = myMobiles.locate(mid,cid);
  UQ_CALL("MPS::callSeize()",call1);

  UQ_Cell* cell1 = myCells.item(cid);
  int call2 = cell1->beCallSeized(mid,pid);
  UQ_CALL("MPS::callSeize()",call2);

  // XXX - Call seize on a mobile which is in paged state, leaves mobile
  // in the access list of cell for which seize failed. The mobile can
  // be removed from the access list using a failed disconnect, however
  // this leaves the mobile in an unsable state. To fix this would need
  // to add new function to cell to forcibly remove mobile from access list.

  int call3 = myMobiles.callSeize(mid,pid);
  UQ_CALL("MPS::callSeize()",call3);

  UQ_INVARIANT("MPS::callSeize()");

  UQ_SUCCESS("MPS::callSeize()");
}

/* ------------------------------------------------------------------------- */
int UQ_MPS::tune(UQ_MobileId const& mid, UQ_FreqId& fid)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::tune()");

  UQ_CellId cid;
  int call1 = myMobiles.cell(mid,cid);
  UQ_CALL("MPS::tune()",call1);

  OTC_Boolean pre1 = myCells.contains(cid);
  UQ_PRE("MPS::tune()",pre1);

  UQ_Cell* cell1 = myCells.item(cid);

  int call2 = cell1->tune(mid,fid);
  UQ_CALL("MPS::tune()",call2);

  int call3 = myMobiles.beTuned(mid,fid);
  UQ_CALL("MPS::tune()",call3);

  UQ_INVARIANT("MPS::tune()");

  UQ_SUCCESS("MPS::tune()");
}

/* ------------------------------------------------------------------------- */
int UQ_MPS::answer(UQ_MobileId const& mid)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::answer()");

  int call1 = myMobiles.answer(mid);
  UQ_CALL("MPS::answer()",call1);

  UQ_INVARIANT("MPS::answer()");

  UQ_SUCCESS("MPS::answer()");
}

/* ------------------------------------------------------------------------- */
int UQ_MPS::handOver(
 UQ_MobileId const& mid,
 UQ_CellId const& cid,
 UQ_FreqId& fid
)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::handOver()");

  UQ_CellId cid1;
  int call1 = myMobiles.cell(mid,cid1);
  UQ_CALL("MPS::handover()",call1);

  OTC_Boolean pre1 = (cid != cid1);
  UQ_PRE("MPS::handOver()",pre1);

  UQ_Cell* cell1 = myCells.item(cid1);
  UQ_Cell* cell2 = myCells.item(cid);

  int call2 = cell2->handOn(mid,fid);
  UQ_CALL("MPS::handOver()",call2);

  int call3 = myMobiles.beHandedOver(mid,cid,fid);
  if (call3 == -1)
  {
    int call4 = cell2->handOff(mid);
    if (call4 == -1)
      OTCLIB_EXCEPTION("MPS::handOver() - error during recovery");
  }
  UQ_CALL("MPS::handOver()",call3);

  int call5 = cell1->handOff(mid);
  UQ_CALL("MPS::handOver()",call5);

  UQ_INVARIANT("MPS::handOver()");

  UQ_SUCCESS("MPS::handOver()");
}

/* ------------------------------------------------------------------------- */
int UQ_MPS::disconnect(UQ_MobileId const& mid)
{
  OTCLIB_MARKBLOCK(UQ,"MPS::disconnect()");

  UQ_CellId cid;
  int call1 = myMobiles.cell(mid,cid);
  UQ_CALL("MPS::disconnect()",call1);

  UQ_Cell* cell1 = myCells.item(cid);
  int call2 = cell1->disconnect(mid);
  UQ_CALL("MPS::disconnect()",call2);

  int call3 = myMobiles.beDisconnected(mid);
  UQ_CALL("MPS::disconnect()",call3);

  UQ_INVARIANT("MPS::disconnect()");

  UQ_SUCCESS("MPS::disconnect()");
}

/* ------------------------------------------------------------------------- */
OTC_Boolean UQ_MPS::invariant() const
{
  OTCLIB_MARKBLOCK(UQ,"MPS::invariant()");

  OTC_Boolean c1 = (myCells.population() == UQ_CellId::population());

  OTC_Boolean c2 = OTCLIB_TRUE;
  OTC_Iterator<UQ_CellId> iter1(myCells.keys());
  for (iter1.reset(); iter1.isValid(); iter1.next())
  {
    if (!UQ_CellId::contains(iter1.item()))
    {
      c2 = OTCLIB_FALSE;
      break;
    }
  }

  return c1 && c2;
}

/* ------------------------------------------------------------------------- */
void UQ_MPS::dump(ostream& outs) const
{
  outs << "MPS::dump {" << endl;
  myMobiles.dump(outs);
  OTC_Iterator<UQ_Cell*> iter(myCells.items());
  for (iter.reset(); iter.isValid(); iter.next())
    iter.item()->dump(outs);
  outs << "}" << endl;
}

/* ------------------------------------------------------------------------- */
