#ifndef OTC_COLLCTN_AVLTREE_HH
#define OTC_COLLCTN_AVLTREE_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/avltree.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/avlnode.hh>
#include <OTC/refcnt/resource.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/avltree.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_AVLTree : public OTC_Resource
    // = TITLE
    //	   The root class of a height balanced AVL tree.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   This class is the root class of a height balanced AVL tree.
    //	   It is this class which holds the root node in the tree and
    //	   maintains the population count for the tree.
    //	   
    //	   The generic AVL tree structure does not actually provide the
    //	   capability to hold any items. Instead it only maintains the
    //	   balance of the tree and ensures the integrity of the tree
    //	   structure. In order for the tree structure to actually hold
    //	   items, a specific node class must be derived from the
    //	   <OTC_AVLNode> class. An example of such a class is the
    //	   <OTC_AVLLinkNode> class, which holds a pointer to a link
    //	   within a linked list.
    //	   
    // = SEE ALSO
    //	   <OTC_AVLNode>, <OTC_AVLLinkNode>
{
    friend		class OTC_AVLNode;

  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    // = DESTRUCTION

			~OTC_AVLTree();
				// Deletes all nodes in the tree.

			OTC_AVLTree();
				// Creates an empty tree. When the tree
				// is empty, the user must explicitly add
				// the initial root node.

    OTC_Boolean		isEmpty() const
				{ return myPopulation == 0; }
				// Returns <OTCLIB_TRUE> if the tree is empty.

    u_int		population() const
				{ return myPopulation; }
				// Returns the number of nodes in the tree.

    void		removeAll();
				// Removes and deletes all nodes in the tree.

    OTC_AVLNode*	root()
				{ return myRoot; }
				// Returns the root node, or <0> if the tree
				// is empty.

    OTC_AVLNode const*	root() const
				{ return myRoot; }
				// Returns the root node, or <0> if the tree
				// is empty.

    void		addRoot(OTC_AVLNode* theNode);
				// Adds the initial root node to the tree
				// and sets the population count to <1>.
				// If this is invoked when the tree is not
				// empty, an exception is raised.

    int			depth() const;
				// Returns the depth of the tree. A tree
				// with a single root node is regarded as
				// having a depth of <0>. If the tree is
				// empty, then <-1> is returned.

    OTC_AVLNode*	node(u_int theIndex)
				{ return _node(theIndex); }
				// If the tree is empty, then <0> is returned.
				// Otherwise the node with index <theIndex>,
				// based on an inorder traversal, is returned.
				// Indexes start at <0>.

    OTC_AVLNode const*	node(u_int theIndex) const
				{ return _node(theIndex); }
				// If the tree is empty, then <0> is returned.
				// Otherwise the node with index <theIndex>,
				// based on an inorder traversal, is returned.
				// Indexes start at <0>.

    OTC_AVLNode*	first()
				{ return _first(); }
				// Returns the first node, base on an inorder
				// traversal, in the tree.

    OTC_AVLNode const*	first() const
				{ return _first(); }
				// Returns the first node, base on an inorder
				// traversal, in the tree.

    OTC_AVLNode*	last()
				{ return _last(); }
				// Returns the last node, base on an inorder
				// traversal, in the tree.

    OTC_AVLNode const*	last() const
				{ return _last(); }
				// Returns the last node, base on an inorder
				// traversal, in the tree.

  private:

			OTC_AVLTree(OTC_AVLTree const&);
				// Do not define an implementation for this.

    OTC_AVLTree&	operator=(OTC_AVLTree const&);
				// Do not define an implementation for this.

    void		_setRoot(OTC_AVLNode* theNode)
				{ myRoot = theNode; }
				// Sets the root node to <theNode>.

    void		_setPopulation(u_int thePopulation)
				{ myPopulation = thePopulation; }
				// Sets the population to <thePopulation>.

    OTC_AVLNode*	_node(u_int theIndex) const;
				// If the tree is empty, then <0> is returned.
				// Otherwise the node with index <theIndex>,
				// based on an inorder traversal, is returned.
				// Indexes start at <0>.

    OTC_AVLNode*	_first() const;
				// Returns the first node, base on an inorder
				// traversal, in the tree.

    OTC_AVLNode*	_last() const;
				// Returns the last node, base on an inorder
				// traversal, in the tree.

    OTC_AVLNode*	myRoot;
				// Pointer to the root node of the tree.

    u_int		myPopulation;
				// The number of nodes in the tree.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_AVLTREE_HH */
