#ifndef OTC_COLLCTN_BAG_C
#define OTC_COLLCTN_BAG_C
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     collctn/bag.c
// 
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/linklist.hh>
#include <OTC/collctn/bucket.hh>
#include <OTC/collctn/icursor.hh>
#include <OTC/collctn/tblentry.hh>
#include <OTC/collctn/hashdefs.hh>

/* ------------------------------------------------------------------------- */
#ifdef __OSE_TEMPLATES__
template<class T> OTC_Bag
{
  OSE_DECLARE OTC_Bucket<T>;
  OSE_DECLARE OTC_ItemCursor< T,OTC_Bucket<T> >;
  OSE_IMPLEMENT OTC_Iterator<T>;
  OSE_IMPLEMENT OTC_HashActions<T>;
  OSE_IMPLEMENT OTC_RankActions<T>;
  OSE_IMPLEMENT OTC_Bucket<T>;
  OSE_IMPLEMENT OTC_ItemCursor< T,OTC_Bucket<T> >;
};
#endif

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Bag<T>::~OTC_Bag()
{
  myPopulation = 0;

  // Null out pointer to table so that member functions can check
  // for access to partially destroyed object from destructors of
  // objects deleted.

  delete [] myTable;
  myTable = 0;

  myItems->unReference();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Bag<T>::OTC_Bag(OTC_Bag<T> const& theBag)
  : myItems(0),
    myTable(0),
    myTableSize(theBag.myTableSize),
    myPopulation(0),
    myLowThreshold(theBag.myLowThreshold),
    myHighThreshold(theBag.myHighThreshold),
    myRankFn(theBag.myRankFn)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myItems = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myItems = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myItems = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myItems != 0);
  myItems->reference();

#if defined(ENV_OSTORE_DML)
  myTable = new ((os_segment*)theLocality) OTC_TableEntry[myTableSize];
#else
#if defined(ENV_OSTORE)
  myTable = new (theLocality,OTC_TableEntry::get_os_typespec(),myTableSize)
   OTC_TableEntry[myTableSize];
#else
  myTable = new OTC_TableEntry[myTableSize];
#endif
#endif
  OTCLIB_ASSERT(myTable != 0);

  OTC_Iterator<T> iter = 0;
  iter = theBag.items(OTCLIB_FORWARD,OTCLIB_UNSAFE);
  for (iter.reset(); iter.isValid(); iter.next())
    add(iter.item());
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Bag<T>::OTC_Bag(int (*theRankFn)(T const&,T const&))
  : myItems(0),
    myTable(0),
    myTableSize(OTCLIB_START_TBLSZ),
    myPopulation(0),
    myLowThreshold(OTCLIB_THRESHOLD(OTCLIB_START_TBLSZ,OTCLIB_LOW_THRESHOLD)),
    myHighThreshold(OTCLIB_THRESHOLD(OTCLIB_START_TBLSZ,OTCLIB_HIGH_THRESHOLD)),
    myRankFn(theRankFn)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myItems = new ((os_segment*)theLocality) OTC_LinkList;
#else
#if defined(ENV_OSTORE)
  myItems = new (theLocality,OTC_LinkList::get_os_typespec()) OTC_LinkList;
#else
  myItems = new OTC_LinkList;
#endif
#endif
  OTCLIB_ASSERT(myItems != 0);
  myItems->reference();

#if defined(ENV_OSTORE_DML)
  myTable = new ((os_segment*)theLocality) OTC_TableEntry[myTableSize];
#else
#if defined(ENV_OSTORE)
  myTable = new (theLocality,OTC_TableEntry::get_os_typespec(),myTableSize)
   OTC_TableEntry[myTableSize];
#else
  myTable = new OTC_TableEntry[myTableSize];
#endif
#endif
  OTCLIB_ASSERT(myTable != 0);
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_Bag<T>::removeAll()
{
  // Make sure we mark all items in index as deleted before killing objects
  // in list in case deleted objects try to access this collection from a
  // destructor.

  myPopulation = 0;
  for (u_int i=0; i<myTableSize; i++)
    myTable[i].markDeleted();

  myItems->removeAll();

  // If index is greater than the defined floor size then delete it and
  // go back to the start size so that space will be reclaimed.

  if (myTableSize > OTCLIB_FLOOR_TBLSZ)
  {
    delete [] myTable;
    myTableSize = OTCLIB_START_TBLSZ;
    myLowThreshold = OTCLIB_THRESHOLD(myTableSize,OTCLIB_LOW_THRESHOLD);
    myHighThreshold = OTCLIB_THRESHOLD(myTableSize,OTCLIB_HIGH_THRESHOLD);

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
    OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
    myTable = new ((os_segment*)theLocality) OTC_TableEntry[myTableSize];
#else
#if defined(ENV_OSTORE)
    myTable = new (theLocality,OTC_TableEntry::get_os_typespec(),myTableSize)
     OTC_TableEntry[myTableSize];
#else
    myTable = new OTC_TableEntry[myTableSize];
#endif
#endif
    OTCLIB_ASSERT(myTable != 0);
  }
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Bag<T>& OTC_Bag<T>::operator=(OTC_Bag<T> const& theBag)
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the bag and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  if (this != &theBag)
  {
    removeAll();

    myRankFn = theBag.myRankFn;
    OTC_Iterator<T> iter = 0;
    iter = theBag.items(OTCLIB_FORWARD,OTCLIB_UNSAFE);
    for (iter.reset(); iter.isValid(); iter.next())
      add(iter.item());
  }

  return *this;
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_Bag<T>::add(T const& theItem)
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the bag and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // If this addition to the bag will mean that we were over the threshold
  // then resize the bag before adding in this entry.

  if (population() + 1 > myHighThreshold)
    resize(otclib_hashtable_grow(myTableSize));

  // In a bag we allow duplicates so simply search for a location where we
  // can stick this item. If <theFreeIndex> is still <-1> after a search
  // then we probably have a corrupted bag as there should always be sufficient
  // room.

  int theFreeIndex = -1;

  // Get the hash value for <theItem>.

  int theHashValue = OTC_HashActions<T>::hash(theItem);

  // Calculate the start of the search.

  u_int anIndex = otclib_hashtable_mask(theHashValue,myTableSize);

  // Do the search. Note that we cannot simply search till the end of the
  // table but must wrap around to the start and search any entries prior
  // to <theOffset> in the table.

  for (u_int i=0; i<=myTableSize-1; i++)
  {
    OTC_TableEntry& anEntry = myTable[anIndex];

    if (
     anEntry.isEmpty() != OTCLIB_FALSE
     ||
     anEntry.isDeleted() != OTCLIB_FALSE
    )
    {
      theFreeIndex = anIndex;
      break;
    }
    if (++anIndex == myTableSize)
      anIndex = 0;
  }

  OTCLIB_ASSERT(theFreeIndex != -1);

  // Add the item to the table and list.

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

  OTC_Bucket<T>* theBucket;
#if defined(ENV_OSTORE_DML)
  theBucket = new ((os_segment*)theLocality) OTC_Bucket<T>(theItem);
#else
#if defined(ENV_OSTORE)
  theBucket = new (theLocality,OTC_Bucket<T>::get_os_typespec())
   OTC_Bucket<T>(theItem);
#else
  theBucket = new OTC_Bucket<T>(theItem);
#endif
#endif
  OTCLIB_ASSERT(theBucket != 0);

  OTC_TableEntry theEntry(theHashValue,theBucket);

  myTable[theFreeIndex] = theEntry;
  myItems->addLast(theBucket);
  myPopulation++;
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_Bag<T>::remove(T const& theItem)
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the bag and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // Search for item.

  int theHashValue = 0;
  int theIndex = -1;

  OTC_Boolean itemExists = search(theItem,theIndex,theHashValue);
  OTCLIB_ENSURE((itemExists == OTCLIB_TRUE),
   "OTC_Bag::remove() - No such item");

  OTCLIB_ASSERT(theIndex != -1);

  OTC_TableEntry& theEntry = myTable[theIndex];
  theEntry.link()->kill();
  theEntry.markDeleted();
  myTable[theIndex].markDeleted();
  myPopulation--;

  // Check population against table floor size and low threshold. If we
  // need to, reduce the size of the table.

  if (myTableSize > OTCLIB_FLOOR_TBLSZ && myPopulation < myLowThreshold)
    resize(otclib_hashtable_shrink(myTableSize));
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_Bag<T>::removeAll(T const& theItem)
{
  while (contains(theItem))
    remove(theItem);
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Boolean OTC_Bag<T>::contains(T const& theItem) const
{
  int theHashValue = 0;
  int theIndex = -1;

  return search(theItem,theIndex,theHashValue);
}

/* ------------------------------------------------------------------------- */
template<class T>
u_int OTC_Bag<T>::frequency(T const& theItem) const
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the bag and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // If the population is zero then the frequency must be zero.

  if (myPopulation == 0)
    return 0;

  // Get the hash value for <theItem>.

  int theHashValue = OTC_HashActions<T>::hash(theItem);

  // Calculate the start of the search.

  u_int anIndex = otclib_hashtable_mask(theHashValue,myTableSize);

  // Do the search. Note that we cannot simply search till the end of the
  // table but must wrap around to the start and search any entries prior
  // to <theOffset> in the table.

  u_int theOccurences = 0;
  for (u_int i=0; i<=myTableSize-1; i++)
  {
    OTC_TableEntry& anEntry = myTable[anIndex];

    if (anEntry.isEmpty() != OTCLIB_FALSE)
      break;

    else if (anEntry.isOccupied() != OTCLIB_FALSE)
    {
      if (anEntry.hashValue() == theHashValue)
      {
	T& aItem = ((OTC_Bucket<T>*)anEntry.link())->item();
	OTC_Boolean theSame = (rank(theItem,aItem) == 0);
	if (theSame != OTCLIB_FALSE)
	  theOccurences++;
      }
    }
    if (++anIndex == myTableSize)
      anIndex = 0;
  }

  return theOccurences;
}

/* ------------------------------------------------------------------------- */
template<class T>
T const& OTC_Bag<T>::pickItem() const
{
  OTCLIB_ENSURE((population() != 0),
   "OTC_Bag::pickItem() - collection must not be empty");
   
  return ((OTC_Bucket<T>*)myItems->first())->item();
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Iterator<T> OTC_Bag<T>::items(
 OTC_Direction theDirection,
 OTC_Protection theProtection
) const
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
  if (!theLocality.isTransientSegment())
  {
    os_transaction* theTXN = os_transaction::get_current();
    if (theTXN != 0 && theTXN->get_type() == os_transaction::read_only)
      theProtection = OTCLIB_UNSAFE;

    if (theProtection == OTCLIB_UNSAFE)
      theLocality = os_segment::get_transient_segment();
  }
#endif

  OTC_Cursor<T>* theIter;
#if defined(ENV_OSTORE_DML)
  theIter = new ((os_segment*)theLocality) OTC_ItemCursor< T,OTC_Bucket<T> >(
   myItems,theDirection,theProtection);
#else
#if defined(ENV_OSTORE)
  theIter = new (theLocality,
   OTC_ItemCursor< T,OTC_Bucket<T> >::get_os_typespec())
   OTC_ItemCursor< T,OTC_Bucket<T> >(myItems,theDirection,theProtection);
#else
  theIter = new OTC_ItemCursor< T,OTC_Bucket<T> >(myItems,theDirection,
   theProtection);
#endif
#endif
  OTCLIB_ASSERT(theIter != 0);
  return theIter;
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Boolean OTC_Bag<T>::search(
 T const& theItem,
 int& theIndex,
 int& theHashValue
) const
{
  // Check to see if table still exists. If table has been nulled it means
  // that we are part way through destroying the bag and thus nothing should
  // be trying to access it.

  OTCLIB_ASSERT(myTable != 0);

  // Get the hash value for <theItem>.

  theHashValue = OTC_HashActions<T>::hash(theItem);

  // Calculate the start of the search.

  u_int anIndex = otclib_hashtable_mask(theHashValue,myTableSize);

  // Do the search. Note that we cannot simply search till the end of the
  // table but must wrap around to the start and search any entries prior
  // to <theOffset> in the table.

  theIndex = -1;
  int theAvailable = -1;

  for (u_int i=0; i<=myTableSize-1; i++)
  {
    OTC_TableEntry& anEntry = myTable[anIndex];

    if (anEntry.isEmpty() != OTCLIB_FALSE)
    {
      if (theAvailable == -1)
	theAvailable = anIndex;

      break;
    }
    else if (anEntry.isDeleted() != OTCLIB_FALSE)
    {
      if (theAvailable == -1)
	theAvailable = anIndex;
    }
    else if (anEntry.isOccupied() != OTCLIB_FALSE)
    {
      if (anEntry.hashValue() == theHashValue)
      {
	T& aItem = ((OTC_Bucket<T>*)anEntry.link())->item();
	OTC_Boolean theSame = (rank(theItem,aItem) == 0);
	if (theSame != OTCLIB_FALSE)
	{
	  theIndex = anIndex;

	  break;
	}
      }
    }
    if (++anIndex == myTableSize)
      anIndex = 0;
  }

  if (theIndex == -1)
  {
    theIndex = theAvailable;
    return OTCLIB_FALSE;
  }

  return OTCLIB_TRUE;
}

/* ------------------------------------------------------------------------- */
template<class T>
void OTC_Bag<T>::resize(u_int theNewSize)
{
  u_int oldTableSize = myTableSize;
  myTableSize = theNewSize;
  myLowThreshold = OTCLIB_THRESHOLD(myTableSize,OTCLIB_LOW_THRESHOLD);
  myHighThreshold = OTCLIB_THRESHOLD(myTableSize,OTCLIB_HIGH_THRESHOLD);
  OTC_TableEntry* oldTable = myTable;

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

#if defined(ENV_OSTORE_DML)
  myTable = new ((os_segment*)theLocality) OTC_TableEntry[myTableSize];
#else
#if defined(ENV_OSTORE)
  myTable = new (theLocality,OTC_TableEntry::get_os_typespec(),myTableSize)
   OTC_TableEntry[myTableSize];
#else
  myTable = new OTC_TableEntry[myTableSize];
#endif
#endif
  OTCLIB_ASSERT(myTable != 0);

  // Traverse through the old table and put each entry in the new table.
  // Note that we cannot simply call <add()> to add the entry into the
  // new table as it will result in another copy of the item being placed
  // in the list. Also note that we do not have to check that an item
  // already exists as we are already guaranteed that they are unique, thus
  // simply search for first empty slot.

  for (u_int j=0; j<oldTableSize; j++)
  {
    OTC_TableEntry& oldEntry = oldTable[j];
    if (oldEntry.isOccupied() != OTCLIB_FALSE)
    {
      int theHashValue = oldEntry.hashValue();
      u_int theOffset = otclib_hashtable_mask(theHashValue,myTableSize);

      for (u_int i=0; i<=myTableSize-1; i++)
      {
	u_int theIndex = otclib_hashtable_mask(theOffset+i,myTableSize);
	OTC_TableEntry& anEntry = myTable[theIndex];

	if (anEntry.isEmpty() != OTCLIB_FALSE)
	{
	  anEntry = oldEntry;
	  break;
	}
      }
    }
  }

  delete [] oldTable;
}

/* ------------------------------------------------------------------------- */
template<class T>
int OTC_Bag<T>::rank(T const& item1, T const& item2) const
{
  return (myRankFn != 0) ? (*myRankFn)(item1,item2) :
   OTC_RankActions<T>::rank(item1,item2);
}

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_BAG_C */
