#ifndef OTC_COLLCTN_BITSETC_HH
#define OTC_COLLCTN_BITSETC_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/bitsetc.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1994 OTC LIMITED
//     Copyright 1994 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/hashactn.hh>
#include <OTC/collctn/rankactn.hh>

#include <string.h>

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/bitsetc.hh"
#else
#pragma interface
#endif
#endif

#ifdef NEED_MEMSET
extern "C" void* memset(void*, int, int);
#endif
#ifdef NEED_MEMCPY
extern "C" void* memcpy(void*, void const*, int);
#endif
#ifdef NEED_MEMCMP
extern "C" int memcmp(void const*, void const*, int);
#endif

/* ------------------------------------------------------------------------- */

class OTC_BitSetC
    // = TITLE
    //     A bit set of size 256.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     <OTC_BitSetC> is a bit set for the range of values <0> to <255>.
    //     The class is designed for tracking character classes in parsing
    //     type applications.
{
  private:

    u_char		byte(u_char theBit) const
				{ return theBit >> 3; }
				// Returns the index of the byte holding the
				// bit for <theBit>.

    u_char		mask(u_char theBit) const
				{ return 1 << (theBit & 0x07); }
				// Returns the mask for <theBit> within its
				// byte.

  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* typespec();
    static os_typespec* get_os_typespec() { return typespec(); }
#endif

    static int		rank(
			 OTC_BitSetC const& theSet1,
			 OTC_BitSetC const& theSet2
			)
				{
				  return memcmp(theSet1.mySet,
				   theSet2.mySet,32);
				}
				// Ranks <theSet1> and <theSet2>.

    static int		hash(OTC_BitSetC const& theSet)
				{
				  return otclib_hash((char const*)
				   theSet.mySet,32);
				}
				// Returns a hash value for <theSet>.

    // = INITIALISATION

    			OTC_BitSetC()
				{ memset(mySet,0,32); }
				// Creates a bit set with all bits set to <0>.

			OTC_BitSetC(OTC_BitSetC const& theSet)
				{ memcpy(mySet,theSet.mySet,32); }
				// Creates a bit set which is a copy of
				// <theSet>.

			OTC_BitSetC(char const* theString);
				// Creates a bit set which has bits set
				// corresponding to the characters listed
				// in <theString>. <theString> should be
				// null terminated. The null terminator
				// does not result in bit <0> being set.

			OTC_BitSetC(char const* theString, u_int theNum);
				// Creates a bit set which has bits set
				// corresponding to first <theNum> characters
				// listed in <theString>.

    OTC_BitSetC&	operator=(OTC_BitSetC const& theSet)
				{
				  memcpy(mySet,theSet.mySet,32);
				  return *this;
				}
				// Replaces this set with <theSet>.

    // = QUERY

    int			count() const;
				// Returns the number of bits set.

    int			test(char theBit) const
				{ return mySet[byte(theBit)] & mask(theBit); }
				// Returns a non zero value if <theBit> is set.

    int			operator[](char theBit) const
				{ return mySet[byte(theBit)] & mask(theBit); }
				// Returns a non zero value if <theBit> is set.

    OTC_Boolean		operator==(OTC_BitSetC const& theSet) const
				{ return memcmp(mySet,theSet.mySet,32) == 0; }
				// Returns <OTCLIB_TRUE> if this bit set
				// is equivalent to <theSet>.

    OTC_Boolean		operator!=(OTC_BitSetC const& theSet) const
				{ return memcmp(mySet,theSet.mySet,32) != 0; }
				// Returns <OTCLIB_TRUE> if this bit set
				// is not equivalent to <theSet>.

    // = MANIPULATORS

    void		reset()
				{ memset(mySet,0,32); }
				// Resets all bits to <0>.

    void		reset(u_char theBit)
				{ mySet[byte(theBit)] &= ~mask(theBit); }
				// Resets <theBit> to <0>.

    void		reset(u_char theStart, u_char theNum);
				// Resets <theNum> bits starting at <theStart>
				// to <0>.

    void		set()
				{ memset(mySet,~0,32); }
				// Sets all bits to <1>.

    void		set(u_char theBit)
				{ mySet[byte(theBit)] |= mask(theBit); }
				// Sets <theBit> to <1>.

    void		set(u_char theStart, u_char theNum);
				// Sets <theNum> bits starting at <theStart>
				// to <1>.

    void		complement();
				// Complements, or inverts all bits.

    void		complement(u_char theBit)
				{ mySet[byte(theBit)] ^= mask(theBit); }
				// Complements, or inverts <theBit>.

    void		complement(u_char theStart, u_char theNum);
				// Complements <theNum> bits starting at
				// <theStart>.

    // = OPERATORS

    OTC_BitSetC		operator~() const;
				// Returns a set which is the complement of
				// this set, ie. with all bits inverted.

    OTC_BitSetC		operator&(OTC_BitSetC const& theSet);
				// Returns the result of performing an
				// intersection between <theSet> and this
				// set.

    OTC_BitSetC&	operator&=(OTC_BitSetC const& theSet);
				// Performs an intersection of <theSet> and
				// this set with the result being left in
				// this set.

    OTC_BitSetC		operator|(OTC_BitSetC const& theSet);
				// Returns the result of performing a union
				// of <theSet> and this set.

    OTC_BitSetC&	operator|=(OTC_BitSetC const& theSet);
				// Performs a union of <theSet> and this set
				// with the result being left in this set.

    OTC_BitSetC		operator^(OTC_BitSetC const& theSet);
				// Returns the result of performing an
				// exclusive or of <theSet> and this set.

    OTC_BitSetC&	operator^=(OTC_BitSetC const& theSet);
				// Performs an exclusive or between <theSet>
				// and this set with the result being left in
				// this set.

  private:

    static u_char	globCounts[256];
				// Look up table for quickly determining
				// the number of bits set.

    u_char		mySet[32];
				// The actual data, distributed across <32>
				// unsigned chars.
};

#ifdef __OSE_TEMPLATES__
OSE_MARK_TEMPLATE OTC_RankActions<OTC_BitSetC>;
OSE_MARK_TEMPLATE OTC_HashActions<OTC_BitSetC>;
#endif

#if defined(CXX_OS)
typedef OTC_RankActions<OTC_BitSetC> otc_collctn_bitsetc_hh_typedef1;
#pragma ObjectStore exclude_instantiations OTC_RankActions<OTC_BitSetC>;
typedef OTC_HashActions<OTC_BitSetC> otc_collctn_bitsetc_hh_typedef2;
#pragma ObjectStore exclude_instantiations OTC_HashActions<OTC_BitSetC>;
#endif

#if !defined(CXX_OS)
class OTC_RankActions<OTC_BitSetC>
{
  public:
    static int          rank(OTC_BitSetC const& s1, OTC_BitSetC const& s2)
				{ return OTC_BitSetC::rank(s1,s2); }
};

class OTC_HashActions<OTC_BitSetC>
{
  public:
    static int          hash(OTC_BitSetC const& theSet)
				{ return OTC_BitSetC::hash(theSet); }
};
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_BITSETC_HH */
