#ifndef OTC_COLLCTN_BNDQUEUE_HH
#define OTC_COLLCTN_BNDQUEUE_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/bndqueue.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/OTC.h>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/bndqueue.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/bndqueue.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

template<class T>
class OTC_BoundedQueue
    // = TITLE
    //     Implements a FIFO or queue with fixed capacity.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     <OTC_BoundedQueue> implements a FIFO or queue with fixed capacity.
    //     Note, that it is the user's responsibility to deal with deletion
    //     of objects held in the queue when it is parameterised over a
    //     pointer type; ie., this class works independently of the
    //     <OTC_BaseActions> class.
    //     
    // = NOTES
    //     Being a bounded queue, for efficiency, an array is used in the
    //     internal implementation instead of a linked list. The consequences
    //     are, that when parameterised over a class type, the class
    //     must have a constructor which accepts no arguments and also be
    //     capable of having assignment performed upon it.
{
  public:

			~OTC_BoundedQueue();

    // = INITIALISATION

			OTC_BoundedQueue(u_int theCapacity, T* theMem=0);
				// Creates an empty queue which has
				// capacity to hold <theCapacity> items.
				// If memory is provided through <theMem>
				// it will be used, else memory will
				// be allocated from the free store. If
				// the memory is provided by the user, it
				// is the user's responsibility to delete
				// it.

			OTC_BoundedQueue(OTC_BoundedQueue<T> const& theQueue);
				// Creates a queue which is a copy of
				// <theQueue>. Space required by this queue
				// will always be allocated from the free
				// store. If pointers are held in the queue,
				// only the value of the pointers is copied,
				// not what the pointer is pointing at.

    // = ASSIGNMENT

    OTC_BoundedQueue<T>&	operator=(OTC_BoundedQueue<T> const& theQueue);
				// Replaces this queue with the contents of
				// <theQueue>. Space required by this queue
				// will always be allocated from the free
				// store. The capacity of this queue will
				// be changed to that of <theQueue>.
				// If pointers are held in the queue,
				// only the value of the pointers is copied,
				// not what the pointers point at.

    // = QUERY

    OTC_Boolean		isEmpty() const
				{ return myCount == 0; }
				// Returns <OTCLIB_TRUE> if the queue is empty.

    OTC_Boolean		isFull() const
				{ return myCount >= myCapacity; }
				// Returns <OTCLIB_TRUE> if the queue is full.

    u_int		count() const
				{ return myCount; }
				// Returns the number of items in the queue.

    u_int		capacity() const
				{ return myCapacity; }
				// Returns the maximum number of items which
				// the queue can hold.

    T&			head()
				{ return _head(); }
				// Returns a reference to the item on
				// the head of the queue. If the queue
				// is empty, an exception is raised.

    T const&		head() const
				{ return _head(); }
				// Returns a reference to the item on
				// the head of the queue. If the queue
				// is empty, an exception is raised.

    T&			peek(u_int theIndex)
				{ return _peek(theIndex); }
				// Returns the item in the queue given by
				// <theIndex>. An index of <0> represents the
				// head of the queue. Successive items are
				// numbered from <1> onwards. If
				// <theIndex> is outside the bounds of the
				// queue, an exception is raised.

    T const&		peek(u_int theIndex) const
				{ return _peek(theIndex); }
				// Returns the item in the queue given by
				// <theIndex>. An index of <0> represents the
				// head of the queue. Successive items are
				// numbered from <1> onwards. If
				// <theIndex> is outside the bounds of the
				// queue, an exception is raised.

    // = MODIFICATION

    void		add(T const& theItem);
				// Adds <theItem> onto the tail of the
				// queue. If the capacity of the queue would
				// be exceeded, an exception is raised.

    T			remove();
				// Removes and returns the item at
				// the head of the queue. If the queue
				// is empty, an exception is raised.

    void		clear()
				{ myCount = 0; myGet = 0; myPut = 0; }
				// Clears the queue.

    void		discard(u_int theCount);
				// Discards the first <theCount> items
				// from the head of the queue. If there
				// are not that many items, an
				// exception is raised.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  private:

    T&			_head() const;
				// Returns a reference to the item on
				// the head of the queue. If the queue
				// is empty an exception is raised.

    T&			_peek(u_int theIndex) const;
				// Returns the item in the queue given by
				// <theIndex>. An index of <0> represents the
				// head of the queue. Successive items are
				// numbered from <1> onwards. If
				// <theIndex> is outside the bounds of the
				// queue an exception is raised.

    u_int		myCapacity;
				// The maximum number of items which can
				// be held.

    u_int		myCount;
				// Number of items held.

    u_int		myGet;
				// Index of next item to be fetched.

    u_int		myPut;
				// Index of next available location
				// to put an item.

    T*			myItems;
				// Items in the queue.

    OTC_Boolean		myDestroy;
				// <OTCLIB_TRUE> if <myItems> needs to be
				// destroyed.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/bndqueue.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_BNDQUEUE_HH */
