#ifndef OTC_COLLCTN_ILIST_HH
#define OTC_COLLCTN_ILIST_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/ilist.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/dequei.hh>
#include <OTC/collctn/linkiter.hh>
#include <OTC/collctn/worker.hh>
#include <OTC/collctn/visitor.hh>
#include <OTC/collctn/dirction.hh>
#include <OTC/collctn/prtction.hh>
#include <OTC/collctn/shlwcopy.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/ilist.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/ilist.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T> OTC_IList
{
  OSE_DECLARE OTC_Visitor<T>;
  OSE_DECLARE OTC_Worker<T>;
};
#endif

template<class T>
class OTC_IList
    // = TITLE
    //	   Linked list with inbuilt iterator.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   This class is a templated, minimal linked list, incorporating an
    //	   inbuilt iterator. This would be used where the extra functionality
    //	   of <OTC_Deque> or <OTC_List> is not required. This class
    //	   replaces the <OTC_SimpleList> class.
    //	   
    //	   Items may be inserted at any position in the list by moving the
    //	   inbuilt iterator to the required location and then adding the
    //	   item, either before, or after the iterator position. It is also
    //	   possible to add items directly onto the head or tail of the list.
    //
    //	   The class uses <OTC_Bucket> to hold items internally. Therefore,
    //	   <OTC_BaseActions> can be used to define what occurs when items
    //	   are added and removed from the list.
    //	   
    // = SEE ALSO
    //	   <OTC_Bucket>, <OTC_BaseActions>
{
  public:

    // = CONSTRUCTION

			OTC_IList();
				// Creates an empty list.

			OTC_IList(OTC_IList<T> const& theList);
				// Creates a list which is a copy of
				// <theList>. If the list holds pointers,
				// only the pointer is copied and not
				// what the pointer points at.

			OTC_IList(OTC_IList<T> const& theList, OTC_ShallowCopy);
				// Creates an alias for <theList>.
				// The same items will be referenced
				// by each list, however, the iterators
				// will be independent.

    // = DESTRUCTION

			~OTC_IList();
				// Invokes <removeAll()> to kill all buckets
				// in the list.

    // = ASSIGNMENT

    OTC_IList<T>&	operator=(OTC_IList<T> const& theList);
				// Replaces this list with a copy of
				// <theList>. The iterator is reset
				// to being off the start of the list.

    // = POPULATION

    u_int		population() const
				{ return myList.population(); }
				// Returns the number of items in the list.

    OTC_Boolean		isEmpty() const
				{ return myList.population() == 0; }
				// Returns <OTCLIB_TRUE> if the list is empty.

    // = ITERATION

    T&			item()
				{ return _item(); }
				// If the iterator is located over a valid
				// item, a reference to the item is returned.
				// If there is not a valid item under the
				// iterator, an exception is raised.

    T const&		item() const
				{ return _item(); }
				// If the iterator is located over a valid
				// item, a reference to the item is returned.
				// If there is not a valid item under the
				// iterator, an exception is raised.

    OTC_Boolean		isValid() const
				{
				  return myIter->isLink() &&
				   !myIter->link()->isDead();
				}
				// Returns <OTCLIB_TRUE> if the iterator is
				// located over a live item.

    OTC_Boolean		isStart() const
				{ return myIter->isStart(); }
				// Returns <OTCLIB_TRUE> if the iterator is
				// located off the start of the list.

    OTC_Boolean		isEnd() const
				{ return myIter->isEnd(); }
				// Returns <OTCLIB_TRUE> if the iterator is
				// located off the end of the list.

    void		next() const
				{ myIter->next(); }
				// Moves the iterator onto the next item.

    void		prev() const
				{ myIter->prev(); }
				// Moves the iterator onto the previous item.

    void		resetFirst() const
				{ myIter->resetFirst(); }
				// Moves the iterator to the first item in
				// the list. If the list is empty, the
				// iterator will be placed off the end of
				// the list.

    void		resetLast() const
				{ myIter->resetLast(); }
				// Moves the iterator to the last item in
				// the list. If the list is empty, the
				// iterator will be placed off the start of
				// the list.

    // = APPLICATORS

    void		apply(
			 OTC_Visitor<T>& theApplicator,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Applies <theApplicator> to each of the
				// items in the collection. The direction
				// being determined by <theDirection>. Valid
				// values are <OTCLIB_FORWARD> and
				// <OTCLIB_BACKWARD>.

    void		apply(
			 OTC_Worker<T>& theApplicator,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			);
				// Applies <theApplicator> to each of the
				// items in the collection. The direction
				// being determined by <theDirection>. Valid
				// values are <OTCLIB_FORWARD> and
				// <OTCLIB_BACKWARD>.

    // = ADDITION

    void		addBefore(T const& theItem);
				// Adds <theItem> before the item where the
				// iterator is currently located. If the
				// iterator is located off the end of the
				// list, the effect is that <theItem> is
				// appended to the list. If the iterator is
				// off the start of the list, the effect
				// is that <theItem> is prepended to the
				// list. After the operation, the iterator
				// will be located on the newly inserted
				// item.

    void		addAfter(T const& theItem);
				// Adds <theItem> after the item where the
				// iterator is currently located. If the
				// iterator is located off the end of the
				// list, the effect is that <theItem> is
				// appended to the list. If the iterator is
				// off the start of the list, the effect
				// is that <theItem> is prepended to the
				// list. After the operation, the iterator
				// will be located on the newly inserted
				// item.

    void		addFirst(T const& theItem)
				{ myList.addFirst(_link(theItem)); }
				// Adds <theItem> at the head of the list.
				// The location of the iterator is unchanged.

    void		addLast(T const& theItem)
				{ myList.addLast(_link(theItem)); }
				// Adds <theItem> after the last item in
				// the list. The location of the iterator
				// is unchanged.

    // = LOOKING

    T&			first()
				{ return _first(); }
				// If the list is not empty, a reference
				// is returned to the item on the head of
				// the list. If the list is empty, an
				// exception is raised.

    T const&		first() const
				{ return _first(); }
				// If the list is not empty, a reference is
				// returned to the item on the head of the
				// list. If the list is empty, an exception
				// is raised.

    T&			last()
				{ return _last(); }
				// If the list is not empty, a reference is
				// returned to the last item in the list. If
				// the list is empty, an exception is raised.

    T const&		last() const
				{ return _last(); }
				// If the list is not empty, a reference is
				// returned to the last item in the list. If
				// the list is empty, an exception is raised.

    T*			pFirst()
				{ return myList.isEmpty() ? 0 : &_first(); }
				// If the list is not empty, a pointer
				// is returned to the item on the head of
				// the list. If the list is empty, a null
				// pointer is returned.

    T const*		pFirst() const
				{ return myList.isEmpty() ? 0 : &_first(); }
				// If the list is not empty, a pointer
				// is returned to the item on the head of
				// the list. If the list is empty, a null
				// pointer is returned.

    T*			pLast()
				{ return myList.isEmpty() ? 0 : &_last(); }
				// If the list is not empty, a pointer is
				// returned to the last item in the list. If
				// the list is empty, a null pointer is
				// returned.

    T const*		pLast() const
				{ return myList.isEmpty() ? 0 : &_last(); }
				// If the list is not empty, a pointer is
				// returned to the last item in the list. If
				// the list is empty, a null pointer is
				// returned.

    // = REMOVING

    void		removeAll()
				{ myIter->resetStart(); myList.removeAll(); }
				// Kills all buckets holding items in the
				// list. The iterator is reset to being off
				// the start of the list.

    void		remove();
				// If the iterator is located over a valid
				// item, the bucket which is holding the item
				// is killed. The bucket is only unlinked
				// from the list and destroyed when the
				// iterator is moved. If the iterator is
				// not over a valid link within the list,
				// an exception is raised.

    void		removeFirst()
				{ myList.removeFirst(); }
				// If the list is not empty, the first
				// item in the list is killed. If the
				// internal iterator is not located on
				// the first item, it will be removed
				// immediately. If the list were empty,
				// an exception is raised.

    void		removeLast()
				{ myList.removeLast(); }
				// If the list is not empty, the last
				// item in the list is killed. If the
				// internal iterator is not located on
				// the first item, it will be removed
				// immediately. If the list were empty,
				// an exception is raised.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  private:

    T&			_item() const;
				// If the iterator is located over a valid
				// item, a reference to the item is
				// returned. If there is not a valid item
				// under the iterator, an exception is
				// raised.

    T&			_first() const;
				// If the list is not empty, a reference
				// is returned to the item on the head of
				// the list. If the list is empty, an
				// exception is raised.

    T&			_last() const;
				// If the list is not empty, a reference
				// is returned to the last item in
				// the list. If the list is empty, an
				// exception is raised.

    OTC_Link*		_link(T const& theItem);
				// Creates an instance of <OTC_Bucket>
				// to hold <theItem>.

    OTC_DequeI		myList;
				// The actual items.

    OTC_LinkIterator*	myIter;
				// Internal iterator over items in the list.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/ilist.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_ILIST_HH */
