#ifndef OTC_COLLCTN_IMAP_HH
#define OTC_COLLCTN_IMAP_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/imap.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/prtction.hh>
#include <OTC/collctn/hashactn.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/collctn/dirction.hh>
#include <OTC/collctn/visitor.hh>
#include <OTC/collctn/worker.hh>
#include <OTC/collctn/prvisitr.hh>
#include <OTC/collctn/prworker.hh>
#include <OTC/collctn/linklist.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/imap.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/imap.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T1, class T2> OTC_IMap
{
  OSE_DECLARE OTC_HashActions<T1>;
  OSE_DECLARE OTC_RankActions<T1>;
  OSE_DECLARE OTC_Visitor<T1>;
  OSE_DECLARE OTC_Visitor<T2>;
  OSE_DECLARE OTC_Worker<T2>;
  OSE_DECLARE OTC_PairVisitor<T1,T2>;
  OSE_DECLARE OTC_PairWorker<T1,T2>;
};
#endif

class OTC_TableEntry;

template<class T1, class T2>
class OTC_IMap
    // = TITLE
    //	   Class implementing a one way map between two objects.
    //	   
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   This class implements a one way mapping between two objects. The
    //	   first of these objects, the key, must be unique and must be able
    //	   to have a hash value generated for it using the <OTC_HashActions>
    //	   class. It must also be possible to compare the keys. The method
    //	   of comparing keys can be determined by providing a template
    //	   override version of <OTC_RankActions> for the type of the key.
    //	   
    // = NOTES
    //	   When a pointer type is used as a key and the hash value generated
    //	   from it, is based not on the pointer, but on some part of the
    //	   object being pointed at, then it is important that the pointer be
    //	   of type pointer to const object. In other words you should define
    //	   the map as <OTC_IMap\<EX_Foo const*,EX_Bar*\>> and not just
    //	   <OTC_IMap\<EX_Foo*,EX_Bar*\>>. If this is not done, it would be
    //	   possible for a user to modify parts of the key from which the hash
    //	   value is generated. If the hash value of an object changes, it
    //	   will no longer be accessible.
    //	   
    //	   The <OTC_Bucket> class is used internally to hold both key and
    //	   items. Thus the <OTC_BaseActions> class may be used to provide
    //	   actions to be performed, when items are inserted or removed from
    //	   the map.
    //
    //	   Since an attempt to access or remove an item from the map, which
    //	   isn't there, will raise an exception, it is important to first
    //	   check that an item is actually in the map by calling the
    //	   <contains()> function.
    //	   
    // = SEE ALSO
    //	   <OTC_HashActions>, <OTC_Bucket>, <OTC_BaseActions>,
    //	   <OTC_RankActions>
{
  public:

			~OTC_IMap();

    // = INITIALISATION

			OTC_IMap(int (*theRankFn)(T1 const&,T1 const&)=0);
				// Creates an empty map. <theRankFn> is an
				// optional comparison function to be used in
				// place of <OTC_RankActions> for the key.

			OTC_IMap(OTC_IMap<T1,T2> const& theMap);
				// Creates a copy of <theMap>.

    // = DESTRUCTION

    void		removeAll();
				// Removes all key/item pairs in this map.

    // = INSERTION

    OTC_IMap<T1,T2>&	operator=(OTC_IMap<T1,T2> const& theMap);
				// Replaces the contents of this map with
				// the contents of <theMap>.

    void		add(T1 const& theKey, T2 const& theItem);
				// If an item corresponding to <theKey> does
				// not exist in this map, then <theItem> will
				// be entered into this map. If an item
				// corresponding to <theKey> already exists,
				// an exception will be raised.

    // = RETRIEVAL/SEARCH

    void		remove(T1 const& theKey);
				// If an item corresponding to <theKey>
				// exists in this map, it will be removed
				// and deleted. If an item corresponding to
				// <theKey> does not exist, an exception
				// will be raised.

    OTC_Boolean		contains(T1 const& theKey) const;
				// Returns <OTCLIB_TRUE> if this map contains
				// a item corresponding to <theKey>.

    T2&			item(T1 const& theKey)
				{ return _item(theKey); }
				// If an item corresponding to <theKey>
				// exists in this map, a reference to it will
				// be returned. If an item corresponding to
				// <theKey> does not exist, an exception will
				// be raised.

    T2 const&		item(T1 const& theKey) const
				{ return _item(theKey); }
				// If an item corresponding to <theKey>
				// exists in this map, a reference to it will
				// be returned. If an item corresponding to
				// <theKey> does not exist, an exception will
				// be raised.

    T2*			pItem(T1 const& theKey)
				{ return _pItem(theKey); }
				// If an item corresponding to <theKey>
				// exists in this map, a pointer to it will
				// be returned. If an item corresponding to
				// <theKey> does not exist, a null pointer
				// is returned.

    T2 const*		pItem(T1 const& theKey) const
				{ return _pItem(theKey); }
				// If an item corresponding to <theKey>
				// exists in this map, a pointer to it will
				// be returned. If an item corresponding to
				// <theKey> does not exist, a null pointer
				// is returned.

    T1 const&		pickKey() const;
				// Returns a key from the collection. If
				// the map is empty, an exception is raised.
				// The function is not guaranteed to
				// return a different key on each call.

    T2 const&		pickItem() const;
				// Returns an item from the collection. If
				// the map is empty, an exception is raised.
				// The function is not guaranteed to
				// return a different item on each call.

    // = QUERY

    u_int		population() const
				{ return myPopulation; }
				// Returns the number of key/item mappings
				// contained in this map.

    OTC_Boolean		isEmpty() const
				{ return myPopulation == 0; }
				// Returns <OTCLIB_TRUE> if the collection
				// is empty.

    // = ITERATION

    T1 const&		key() const
				{ return _key(); }
				// If the iterator is located over a valid
				// key/item pair, a reference to the key is
				// returned. If there is not a valid key/item
				// pair under the iterator, an exception is
				// raised.

    T2&			item()
				{ return _item(); }
				// If the iterator is located over a valid
				// key/item pair, a reference to the item is
				// returned. If there is not a valid key/item
				// pair under the iterator, an exception is
				// raised.

    T2 const&		item() const
				{ return _item(); }
				// If the iterator is located over a valid
				// key/item pair, a reference to the item is
				// returned. If there is not a valid key/item
				// pair under the iterator, an exception is
				// raised.

    OTC_Boolean		isValid() const
				{
				  return myIter->isLink() &&
				   !myIter->link()->isDead();
				}
				// Returns <OTCLIB_TRUE> if the iterator is
				// located over a live key/item pair.

    OTC_Boolean		isStart() const
				{ return myIter->isStart(); }
				// Returns <OTCLIB_TRUE> if the iterator is
				// located off the start of the list of
				// key/item pairs.

    OTC_Boolean		isEnd() const
				{ return myIter->isEnd(); }
				// Returns <OTCLIB_TRUE> if the iterator is
				// located off the end of the list of key/item
				// pairs.

    void		next() const
				{ myIter->next(); }
				// Moves the iterator onto the next key/item
				// pair.

    void		prev() const
				{ myIter->prev(); }
				// Moves the iterator onto the previous
				// key/item pair.

    void		resetFirst() const
				{ myIter->resetFirst(); }
				// Moves the iterator to the first key/item
				// pair. If the map is empty, the iterator
				// will be placed off the conceptual end of
				// the list.

    void		resetLast() const
				{ myIter->resetLast(); }
				// Moves the iterator to the last key/item
				// pair. If the map is empty, the iterator
				// will be placed off the conceptual start of
				// the list.

    void		remove();
				// Removes the key/item pair currently
				// located under the iterator. If there
				// is no valid key/item pair currently
				// under the iterator, an exception is
				// raised.

    // = APPLICATORS

    void		applyKeys(
			 OTC_Visitor<T1>& theApplicator,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Applies <theApplicator> to each of the
				// keys in the collection. The direction
				// being determined by <theDirection>. Valid
				// values are <OTCLIB_FORWARD> and
				// <OTCLIB_BACKWARD>.

    void		applyItems(
			 OTC_Visitor<T2>& theApplicator,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Applies <theApplicator> to each of the
				// items in the collection. The direction
				// being determined by <theDirection>. Valid
				// values are <OTCLIB_FORWARD> and
				// <OTCLIB_BACKWARD>.

    void		applyItems(
			 OTC_Worker<T2>& theApplicator,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			);
				// Applies <theApplicator> to each of the
				// items in the collection. The direction
				// being determined by <theDirection>. Valid
				// values are <OTCLIB_FORWARD> and
				// <OTCLIB_BACKWARD>.

    void		applyPairs(
			 OTC_PairVisitor<T1,T2>& theApplicator,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Applies <theApplicator> to each of the
				// key/item pairs in the collection. The
				// direction being determined by
				// <theDirection>. Valid values are
				// <OTCLIB_FORWARD> and <OTCLIB_BACKWARD>.

    void		applyPairs(
			 OTC_PairWorker<T1,T2>& theApplicator,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			);
				// Applies <theApplicator> to each of the
				// key/item pairs in the collection. The
				// direction being determined by
				// <theDirection>. Valid values are
				// <OTCLIB_FORWARD> and <OTCLIB_BACKWARD>.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  private:

    T1&                  _key() const;
				// If the iterator is located over a valid
				// key/item pair, a reference to the key is
				// returned. If there is not a valid key/item
				// pair under the iterator, an exception is
				// raised.

    T2&                  _item() const;
				// If the iterator is located over a valid
				// key/item pair, a reference to the item is
				// returned. If there is not a valid key/item
				// pair under the iterator, an exception is
				// raised.

    T2&			_item(T1 const& theKey) const;
				// If an item corresponding to <theKey>
				// exists in this map, a reference to
				// it will be returned. If an item
				// corresponding to <theKey> does not exist,
				// an exception will be raised.

    T2*			_pItem(T1 const& theKey) const;
				// If an item corresponding to <theKey>
				// exists in this map, a pointer to
				// it will be returned. If an item
				// corresponding to <theKey> does not exist,
				// a null pointer will be returned.

    OTC_Boolean		search(
			 T1 const& theKey,
			 int& theIndex,
			 int& theHashValue
			) const;
				// Searches for <theKey> in the hashtable.
				// If the key is in the hashtable index,
				// <OTCLIB_TRUE> is returned and <theIndex>
				// is set to the location in the table where
				// the key can be found. If the key is not
				// in the table, <OTCLIB_FALSE> is returned
				// and <theIndex> is set to a location in
				// the table where the key could be placed.
				// In both cases, <theHashValue> is set to
				// the hash value of the key.

    void		resize(u_int theNewSize);
				// Change the capacity of the map.

    int			rank(T1 const& key1, T1 const& key2) const;
				// Ranks two items.

    OTC_LinkList*	myPairs;
				// List of pairs.

    OTC_TableEntry*	myTable;
				// Key index table.

    u_int		myTableSize;
				// This must be a power of two.

    u_int		myPopulation;
				// Count of the number of keys in the map.

    u_int		myLowThreshold;
				// Numbers of items which if less than
				// will result in the table being
				// reduced.

    u_int		myHighThreshold;
				// Numbers of items which if greater than
				// will result in the table being
				// increased.

    int			(*myRankFn)(T1 const&,T1 const&);
				// Comparison function to be used in
				// place of <OTC_RankActions> for key.

    OTC_LinkIterator*	myIter;
				// Internal iterator over key/item pairs
				// in the map.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/imap.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_IMAP_HH */
