#ifndef OTC_COLLCTN_INDEX_HH
#define OTC_COLLCTN_INDEX_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/index.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/avltree.hh>
#include <OTC/collctn/rankactn.hh>
#include <OTC/collctn/indxnode.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/index.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/index.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template <class T> OTC_Index
{
  OSE_DECLARE OTC_RankActions<T>;
  OSE_DECLARE OTC_IndexNode<T>;
};
#endif

template<class T>
class OTC_Index
    // = TITLE
    //     Index mapping objects to void pointers.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     This class is provided to allow the construction of maps between a
    //     key and an item stored as a void pointer. The index uses a height
    //     balance AVL tree, maintaining in order the lookup keys to
    //     facilitate searching. The index does not in any way manage the
    //     items referenced as void pointers from the index. This means that
    //     when the index is destroyed, it is your responsibility to ensure
    //     the referenced items are destroyed. Normally the items would
    //     reside in a separate list and thus this can be achieved by
    //     deleting the list. The keys are managed through the use of the
    //     <OTC_BaseActions> class. Thus, if the keys are pointers, you can
    //     define actions to be performed when keys are added or removed
    //     from the index. Searching with keys is carried out using the
    //     <OTC_RankActions> class.
    //     
    // = SEE ALSO
    //     <OTC_IndexNode>, <OTC_AVLTree>, <OTC_BaseActions>,
    //     <OTC_RankActions>
{
  public:

    // = DESTRUCTION

			~OTC_Index() {}
				// Index is destroyed.

    // = INITIALISATION

			OTC_Index(int (*theRankFn)(T const&,T const&)=0)
			 : _rankFn(theRankFn) {}
				// Creates an empty index. <theRankFn>
				// is an alternate rank function to use
				// in place of <OTC_RankActions>.

    // = QUERY

    u_int		population() const
				{ return _index.population(); }
				// Returns the number of keys in the
				// index.

    OTC_Boolean		isEmpty() const
				{ return _index.isEmpty(); }
				// Returns true of the index is empty.

    OTC_Boolean		contains(T const& theKey) const;
				// Returns <OTCLIB_TRUE> if <theKey>
				// is in the index.

    OTC_IndexNode<T>*	node(T const& theKey)
				{ return _node(theKey); }
				// Returns a pointer to the node in
				// the tree corresponding to <theKey>.
				// Returns <0> if <theKey> doesn't
				// exist.

    OTC_IndexNode<T> const*	node(T const& theKey) const
				{ return _node(theKey); }
				// Returns a pointer to the node in
				// the tree corresponding to <theKey>.
				// Returns <0> if <theKey> doesn't
				// exist.

    OTC_IndexNode<T>*	nodeAtIndex(u_int theIndex)
				{ return _nodeAtIndex(theIndex); }
				// Returns a pointer to the node in
				// the tree corresponding to <theIndex>.
				// Returns <0> if <theIndex> doesn't
				// exist.

    OTC_IndexNode<T> const*	nodeAtIndex(u_int theIndex) const
				{ return _nodeAtIndex(theIndex); }
				// Returns a pointer to the node in
				// the tree corresponding to <theIndex>.
				// Returns <0> if <theIndex> doesn't
				// exist.

    int			index(T const& theKey) const;
				// Returns the position of <theKey> within
				// the tree. Keys are maintained in an order
				// as defined by <OTC_RankActions>. If
				// <theKey> doesn't exist, a value of <-1> is
				// returned.

    OTC_IndexNode<T>*	root()
				{ return (OTC_IndexNode<T>*)_index.root(); }
				// Returns the root node of the index.

    OTC_IndexNode<T> const*	root() const
				{
				  return
				   (OTC_IndexNode<T> const*)_index.root();
				}
				// Returns the root node of the index.

    // = INSERTION

    OTC_IndexNode<T>*	insert(T const& theKey);
				// Inserts into the index a node
				// corresponding to <theKey> and
				// returns it. If a node corresponding
				// to <theKey> already exists, it
				// will be returned. If the node didn't
				// already exist, the item field will
				// be <0>, presuming it was set to a
				// non zero value when the key was first
				// added to the index.

    // = REMOVAL
    //     In addition to the following, a node can be removed by
    //     calling <unlink()> on it and then deleting the node
    //     by using <operator delete()>.

    void		removeAll()
				{ _index.removeAll(); }
				// Index is destroyed.

    void		removeAtIndex(u_int theIndex);
				// Removes from the index the node in
				// position <theIndex>. If there exists no
				// node corresponding to the index
				// <theIndex>, no action is taken.

    void		remove(T const& theKey);
				// Removes from the index the node
				// corresponding to <theKey>. If there exists
				// no node corresponding to <theKey>, no
				// action is taken.

  public:

#if defined(ENV_OSTORE) && !defined(SCHEMA_GENERATION)
    static os_typespec* get_os_typespec();
#endif

  private:

			OTC_Index(OTC_Index<T> const&);
				// Do not provide an implementation for this.

    OTC_Index<T>&	operator=(OTC_Index<T> const&);
				// Do not provide an implementation for this.

    int			_rank(T const& item1, T const& item2) const;
				// Rank two items.

    OTC_IndexNode<T>*	_node(T const& theKey) const;
				// Returns a pointer to the node in
				// the tree corresponding to <theKey>.
				// Returns <0> if <theKey> doesn't exist.

    OTC_IndexNode<T>*	_nodeAtIndex(u_int theIndex) const;
				// Returns a pointer to the node in
				// the tree corresponding to <theIndex>.
				// Returns <0> if <theIndex> doesn't exist.

    OTC_AVLTree		_index;
				// The AVL tree forming the index.

    int			(*_rankFn)(T const&,T const&);
                                // Comparison function to be used in
                                // place of <OTC_RankActions>.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/index.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_INDEX_HH */
