#ifndef OTC_COLLCTN_ITERATOR_HH
#define OTC_COLLCTN_ITERATOR_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     collctn/iterator.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/cursor.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/iterator.hh"
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/iterator.hh"
#else
#pragma interface
#endif
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T> OTC_Iterator
{
  OSE_DECLARE OTC_Cursor<T>;
};
#endif

template<class T>
class OTC_Iterator
    // = TITLE
    //     A smart pointer for iterators.
    //
    // = CLASS TYPE
    //     Concrete
    //
    // = DESCRIPTION
    //     This class exists so that the user of an iterator for a particular
    //     type of collection, does not have to worry about deleting the
    //     iterator when finished with it, nor need to know how to create an
    //     iterator for a specific type of collection. This class will ensure
    //     that the iterator is destroyed, when this class goes out of scope.
    //     In order for this to work correctly, an instance of this class
    //     should never be created using <new>.
    //     
    // = EXAMPLE
    //
    // = BEGIN<CODE>
    //     void sum(OTC_Collection<int> const& aCollection)
    //     {
    //       OTC_Iterator<int> anIter = 0;
    //       anIter = aCollection.items();
    //       int i=0;
    //       for (anIter.reset(); anIter.isValid(); anIter.next())
    //         i += anIter.item();
    //       cout << i << endl;
    //     }
    // = END<CODE>
{
  protected:

    T&			_item() const
				{
				  OTCLIB_ENSURE((myIter != 0),
				   "OTC_Iterator::item() - Null iterator");
				  return myIter->item();
				}
				// Returns the item under the current
				// location of the iterator. Raises an
				// exception if there is no item under the
				// iterator, or if this is a null iterator.

    OTC_Cursor<T>*	_clone() const
				{ return myIter ? myIter->clone() : 0; }
				// Returns a clone of this iterator.

#if !defined(ENV_NOAPPLICATORS)

    void		_apply(OTC_Visitor<T>& theApplicator)
				{ OTC_Cursor<T>::apply(myIter,theApplicator); }
				// Applies <theApplicator> to each item
				// accessible by this iterator.

    void		_apply(OTC_Worker<T>& theApplicator)
				{ OTC_Cursor<T>::apply(myIter,theApplicator); }
				// Applies <theApplicator> to each item
				// accessible by this iterator.

#endif

  public:

			~OTC_Iterator()
				{ OTC_Cursor<T>::replace(myIter,0); }

    // = INITIALISATION

			OTC_Iterator()
			  : myIter(0) {}
				// Creates a null iterator.

			OTC_Iterator(OTC_Cursor<T>* anIter)
				{ OTC_Cursor<T>::assign(myIter,anIter); }
				// Constructor used when an iterator
				// for a particular type of collection is
				// first created. Passing in <0> will
				// result in a null iterator being
				// constructed.

			OTC_Iterator(OTC_Iterator<T> const& anIter)
				{ OTC_Cursor<T>::assign(myIter,anIter.myIter); }
				// Copy constructor for when an iterator is
				// used as return type.

    OTC_Iterator<T>&    operator=(OTC_Cursor<T>* anIter)
				{
				  OTC_Cursor<T>::replace(myIter,anIter);
				  return *this;
				}
                                // Sets this iterator to <anIter>.
                                // If <anIter> is <0>, the iterator
                                // is turned into a null iterator and
                                // all associations with the previous
                                // collection are lost.

    OTC_Iterator<T>&	operator=(OTC_Iterator<T> const& anIter)
				{
				  OTC_Cursor<T>::replace(myIter,anIter.myIter);
				  return *this;
				}
				// Sets this iterator to <anIter>.

    // = MOVEMENT

    void		next()
				{ if (myIter) myIter->next(); }
				// Moves the iterator to the next item.

    void		reset()
				{ if (myIter) myIter->reset(); }
				// Resets the iterator to the start.

    // = RETRIEVAL

    T const&		item() const
				{ return _item(); }
				// Returns the item under the current
				// location of the iterator. Raises an
				// exception if there is no item under the
				// iterator, or if this is a null iterator.

    OTC_Boolean		isValid() const
				{
				  return myIter ?
				   myIter->isValid() : OTCLIB_FALSE;
				}
				// Returns <OTCLIB_TRUE> while there is a
				// valid data item under the current location
				// of the iterator.

    // = CLONING

    OTC_Iterator<T>	clone() const
				{ return _clone(); }
				// Returns an iterator which is a clone of
				// this iterator. The new iterator will
				// initially be located over the same item as
				// this iterator, however subsequent
				// operations will be independent of this
				// iterator.

    // = APPLICATOR

#if !defined(ENV_NOAPPLICATORS)

    void		apply(OTC_Visitor<T>& theApplicator)
				{ _apply(theApplicator); }
				// Applies <theApplicator> to each item
				// accessible by this iterator.

#endif

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  private:

    OTC_Cursor<T>*	myIter;
				// The real iterator.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#ifdef __OSE_TEMPLATES__
template<class T> OTC_Iterator
{
  OSE_IMPLEMENT OTC_Cursor<T>;
};
#endif
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_ITERATOR_HH */
