#ifndef OTC_COLLCTN_LINKABLE_HH
#define OTC_COLLCTN_LINKABLE_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/linkable.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1992 1993 OTC LIMITED
//     Copyright 1994 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/memory/mpobject.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/linkable.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_Linkable : public OTC_MPObject
    // = TITLE
    //     Base class for objects which can be linked together.
    //
    // = CLASS TYPE
    //     Abstract
    //
    // = DESCRIPTION
    //     The <OTC_Linkable> class, is the basic building block for creating
    //     a doubly linked list of objects. Derived from <OTC_Linkable> are
    //     two classes which can be used when constructing a linked list.
    //     These two classes are <OTC_Link> and <OTC_Anchor>. The first
    //     of these is the base class for any user objects being placed
    //     into a linked list. The second class is used as an anchor point,
    //     from which to hang a linked list of objects. The <OTC_Anchor>
    //     class is also used in pairs to mark the boundaries for the
    //     traversal of an iterator.
    //     
    // = NOTES
    //     All instances of classes derived from <OTC_Linkable> must
    //     be created on the free store, using <new>.
    //
    // = SEE ALSO
    //     <OTC_Link>, <OTC_Anchor>, <OTC_LinkIterator>
{
  public:

    // = UNLINKING

    void		unlink();
				// Detaches this object from the list it is
				// contained in.

    // = KILLING

    void		kill();
				// If this object is unreferenced, the object
				// will be deleted, otherwise it will be
				// marked as dead.

    // = QUERY

    OTC_Boolean		isLink() const
				{ return myIsLink; }
				// Returns <OTCLIB_TRUE> if this object is
				// not an anchor point but a link.

    OTC_Boolean		isDead() const
				{ return myIsDead; }
				// Returns <OTCLIB_TRUE> if the object has
				// been killed but references still exist.
				// If there are no references, you shouldn't
				// be calling this function as no references
				// implies this objects has been deleted.

    OTC_Linkable*	prev() const
				{ return myPrev; }
				// Returns a pointer to the object before
				// this object.

    OTC_Linkable*	next() const
				{ return myNext; }
				// Returns a pointer to the object after
				// this object.

    // = ADDITIONS

    void		addBefore(OTC_Linkable* theObject);
				// Inserts <theObject> before this object in
				// the list. If <theObject> is already
				// contained within a list, it will first be
				// unlinked from that list.

    void		addAfter(OTC_Linkable* theObject);
				// Inserts <theObject> after this object in
				// the list. If <theObject> is already
				// contained within a list, it will first be
				// unlinked from that list.

    // = REFERENCES

    void		reference();
				// Increments the reference count, ie., the
				// number of references to this object.

    void		unReference();
				// Decrements the reference count. If the
				// reference count reaches <0>, the object
				// deletes itself.

  protected:

    // = INITIALISATION

			OTC_Linkable(OTC_Boolean theIsLink=OTCLIB_FALSE);
				// By default, initialises the class to be an
				// anchor. <theIsLink> should be set to
				// <OTCLIB_TRUE> if the derived class is not
				// an anchor point, but a link.

    // = DESTRUCTION

    virtual		~OTC_Linkable();
				// Invokes <unlink()> to detach this object
				// from the list in which it is contained.

  private:

			OTC_Linkable(OTC_Linkable const& theLink);
				// Do not define an implementation for this.

    OTC_Linkable const&	operator=(OTC_Linkable const& theLink);
				// Do not define an implementation for this.

    void		setPrev(OTC_Linkable* thePrev)
				{ myPrev = thePrev; }
				// Sets the pointer to the previous object
				// to be <thePrev>.

    void		setNext(OTC_Linkable* theNext)
				{ myNext = theNext; }
				// Sets the pointer to the next object
				// to be <theNext>.

    OTC_Linkable*	myPrev;
				// Pointer to object before to this object.

    OTC_Linkable*	myNext;
				// Pointer to object after this object.

    short		myCount;
				// Absolute value is the count of the number
				// of references to this object.

    char		myIsLink;
				// <OTCLIB_TRUE> is derived class is
				// not an anchor but a link.

    char		myIsDead;
				// <OTCLIB_TRUE> if link is dead.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_LINKABLE_HH */
