#ifndef OTC_COLLCTN_LIST_C
#define OTC_COLLCTN_LIST_C
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     collctn/list.c
// 
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/avltree.hh>
#include <OTC/collctn/linklist.hh>
#include <OTC/collctn/avllnode.hh>
#include <OTC/collctn/icursor.hh>

/* ------------------------------------------------------------------------- */
#ifdef __OSE_TEMPLATES__
template<class T> OTC_List
{
  OSE_DECLARE OTC_ItemCursor< T,OTC_Bucket<T> >;
  OSE_IMPLEMENT OTC_Iterator<T>;
  OSE_IMPLEMENT OTC_Modifier<T>;
  OSE_IMPLEMENT OTC_Bucket<T>;
  OSE_IMPLEMENT OTC_ItemCursor< T,OTC_Bucket<T> >;
};
#endif

/* ------------------------------------------------------------------------- */
template<class T>
OTC_List<T>::OTC_List(OTC_List<T> const& theList)
{
  OTC_Iterator<T> iter = 0;
  iter = theList.items(OTCLIB_FORWARD,OTCLIB_UNSAFE);
  for (iter.reset(); iter.isValid(); iter.next())
    addLast(iter.item());
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_List<T>::OTC_List(OTC_List<T>& theList, OTC_ShallowCopy)
  : myList(theList.myList)
{
  // Nothing to do.
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_List<T>& OTC_List<T>::operator=(OTC_List<T> const& theList)
{
  if (&theList != this)
  {
    removeAll();
    OTC_Iterator<T> iter = 0;
    iter = theList.items(OTCLIB_FORWARD,OTCLIB_UNSAFE);
    for (iter.reset(); iter.isValid(); iter.next())
      addLast(iter.item());
  }

  return *this;
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Cursor<T>* OTC_List<T>::_items(
 OTC_Direction theDirection,
 OTC_Protection theProtection
) const
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
  if (!theLocality.isTransientSegment())
  {
    os_transaction* theTXN = os_transaction::get_current();
    if (theTXN != 0 && theTXN->get_type() == os_transaction::read_only)
      theProtection = OTCLIB_UNSAFE;

    if (theProtection == OTCLIB_UNSAFE)
      theLocality = os_segment::get_transient_segment();
  }
#endif

  OTC_Cursor<T>* theIter;
#if defined(ENV_OSTORE_DML)
  theIter = new ((os_segment*)theLocality) OTC_ItemCursor< T,OTC_Bucket<T> >(
   myList.list(),theDirection,theProtection);
#else
#if defined(ENV_OSTORE)
  theIter = new (theLocality,
   OTC_ItemCursor< T,OTC_Bucket<T> >::get_os_typespec())
   OTC_ItemCursor< T,OTC_Bucket<T> >(myList.list(),theDirection,
   theProtection);
#else
  theIter = new OTC_ItemCursor< T,OTC_Bucket<T> >(myList.list(),theDirection,
   theProtection);
#endif
#endif
  OTCLIB_ASSERT(theIter != 0);
  return theIter;
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Cursor<T>* OTC_List<T>::_items(
 int theStart,
 u_int theLength,
 OTC_Direction theDirection,
 OTC_Protection theProtection
) const
{
  if (theLength == 0)
    return 0;

  OTCLIB_ENSURE((theStart >= 0 && theStart+theLength <= population()),
   "OTC_List::_items() - Invalid range");

#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
  if (!theLocality.isTransientSegment())
  {
    os_transaction* theTXN = os_transaction::get_current();
    if (theTXN != 0 && theTXN->get_type() == os_transaction::read_only)
      theProtection = OTCLIB_UNSAFE;

    if (theProtection == OTCLIB_UNSAFE)
      theLocality = os_segment::get_transient_segment();
  }
#endif

  OTC_Anchor* theStartAnchor = 0;
  OTC_Anchor* theEndAnchor = 0;

  if (theProtection == OTCLIB_SAFE)
  {
#if defined(ENV_OSTORE_DML)
    theStartAnchor = new ((os_segment*)theLocality) OTC_Anchor;
    OTCLIB_ASSERT(theStartAnchor != 0);
    theEndAnchor = new ((os_segment*)theLocality) OTC_Anchor;
    OTCLIB_ASSERT(theEndAnchor != 0);
#else
#if defined(ENV_OSTORE)
    theStartAnchor = new (theLocality,OTC_Anchor::get_os_typespec()) OTC_Anchor;
    OTCLIB_ASSERT(theStartAnchor != 0);
    theEndAnchor = new (theLocality,OTC_Anchor::get_os_typespec()) OTC_Anchor;
    OTCLIB_ASSERT(theEndAnchor != 0);
#else
    theStartAnchor = new OTC_Anchor;
    OTCLIB_ASSERT(theStartAnchor != 0);
    theEndAnchor = new OTC_Anchor;
    OTCLIB_ASSERT(theEndAnchor != 0);
#endif
#endif
  }

  OTC_Linkable* theStartLink = 0;
  OTC_Linkable* theEndLink = 0;

  theStartLink = ((OTC_AVLLinkNode*)
   myList.tree()->node(theStart))->link();
  theEndLink = ((OTC_AVLLinkNode*)
   myList.tree()->node(theStart+theLength-1))->link();

  if (theStartAnchor != 0)
  {
    theStartLink->addBefore(theStartAnchor);
    theStartLink = theStartAnchor;
  }
  else
  {
    theStartLink = theStartLink->prev();
  }

  if (theEndAnchor != 0)
  {
    theEndLink->addAfter(theEndAnchor);
    theEndLink = theEndAnchor;
  }
  else
  {
    theEndLink = theEndLink->next();
  }

  OTC_Cursor<T>* theIter;
#if defined(ENV_OSTORE_DML)
  theIter = new ((os_segment*)theLocality) OTC_ItemCursor< T,OTC_Bucket<T> >(
   myList.list(),theStartLink,theEndLink,theDirection,theProtection);
#else
#if defined(ENV_OSTORE)
  theIter = new (theLocality,
   OTC_ItemCursor< T,OTC_Bucket<T> >::get_os_typespec())
   OTC_ItemCursor< T,OTC_Bucket<T> >(myList.list(),theStartLink,theEndLink,
   theDirection,theProtection);
#else
  theIter = new OTC_ItemCursor< T,OTC_Bucket<T> >(myList.list(),theStartLink,
   theEndLink,theDirection,theProtection);
#endif
#endif
  OTCLIB_ASSERT(theIter != 0);

  if (theStartAnchor != 0)
    theStartAnchor->kill();
  if (theEndAnchor != 0)
    theEndAnchor->kill();

  return theIter;
}

/* ------------------------------------------------------------------------- */
template<class T>
OTC_Link* OTC_List<T>::_link(T const& theItem)
{
#if defined(ENV_OSTORE_DML) || defined(ENV_OSTORE)
  OTC_Locality theLocality = OTC_Locality::of(this);
#endif

  OTC_Bucket<T>* theBucket;
#if defined(ENV_OSTORE_DML)
  theBucket = new ((os_segment*)theLocality) OTC_Bucket<T>(theItem);
#else
#if defined(ENV_OSTORE)
  theBucket = new (theLocality,OTC_Bucket<T>::get_os_typespec())
   OTC_Bucket<T>(theItem);
#else
  theBucket = new OTC_Bucket<T>(theItem);
#endif
#endif
  OTCLIB_ASSERT(theBucket != 0);

  return theBucket;
}

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_LIST_C */
