#ifndef OTC_COLLCTN_LIST_HH
#define OTC_COLLCTN_LIST_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
//
// = FILENAME
//     collctn/list.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1991 1992 1993 OTC LIMITED
//     Copyright 1994 1995 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/prtction.hh>
#include <OTC/collctn/iterator.hh>
#include <OTC/collctn/modifier.hh>
#include <OTC/collctn/dirction.hh>
#include <OTC/collctn/range.hh>
#include <OTC/collctn/bucket.hh>
#include <OTC/collctn/listi.hh>
#include <OTC/collctn/shlwcopy.hh>

#ifdef __GNUG__
#if defined(EXPAND_TEMPLATES)
#pragma implementation "OTC/collctn/list.hh"
#endif
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/list.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

#ifdef __OSE_TEMPLATES__
template<class T> OTC_List
{
  OSE_DECLARE OTC_Iterator<T>;
  OSE_DECLARE OTC_Modifier<T>;
  OSE_DECLARE OTC_Bucket<T>;
};
#endif

template<class T>
class OTC_List
    // = TITLE
    //	   A collection class based on a list.
    //
    // = CLASS TYPE
    //	   Concrete
    //
    // = DESCRIPTION
    //	   Class implementing a list of objects. Objects can be accessed at
    //	   the ends of the list, or by their location in the list. Location
    //	   indexes start at <0>.
    //
    // = NOTES
    //	   Indexes are not an inherent part of the list structure, but are
    //	   implemented by a height balanced AVL tree. This means that the
    //	   space consumed by the list is twice that of a list which doesn't
    //	   allow access by indexes. Thus, if access is not required by index
    //	   it is suggested that <OTC_Deque> be used instead.
    //	   
    //	   The <OTC_Bucket> class is used internally to hold items in
    //	   the list. Thus the <OTC_BaseActions> class may be used to provide
    //	   actions to be performed, when items are inserted or removed from
    //	   the list.
    //
    //	   Common functionality for lists holding items of any type, is
    //	   factored out into the <OTC_BList> base class. The base class
    //	   should be consulted as to additional functions which are
    //	   available.
    //
    // = SEE ALSO
    //	   <OTC_Iterator>, <OTC_Modifier>, <OTC_Bucket>, <OTC_BaseActions>,
    //	   <OTC_BList>
{
  public:

#if defined(CXX_LC)
    typedef		OTC_Modifier<T> type1;
    type1		dummy1();
#endif

			~OTC_List() {}

    // = INITIALISATION

			OTC_List() {}
				// Creates an empty list.

			OTC_List(OTC_List<T> const& theList);
				// Creates a copy of <theList>.

			OTC_List(OTC_List<T>& theList, OTC_ShallowCopy);
				// Creates a copy of <theList>.

    // = ASSIGNMENT

    OTC_List<T>&	operator=(OTC_List<T> const& theList);
				// Replaces the contents of this list with
				// items contained in <theList>.

    // = QUERY

    u_int		population() const
				{ return myList.population(); }
				// Returns the number of items in the
				// list.

    OTC_Boolean		isEmpty() const
				{ return myList.isEmpty(); }
				// Returns <OTCLIB_TRUE> if the the list
				// is empty.

    // = ADDITION

    void		addFirst(T const& theItem)
				{ myList.addFirst(_link(theItem)); }
				// Inserts <theItem> at the head of the list.

    void		addLast(T const& theItem)
				{ myList.addLast(_link(theItem)); }
				// Appends <theItem> to the tail of the list.

    void		addBeforeItem(T const& theItem, u_int theIndex)
				{
				  myList.addBeforeItem(_link(theItem),
				   theIndex);
				}
				// Adds <theItem> into the list before
				// the item at location <theIndex>.
				// If <theIndex> is equal to the population
				// of the list, then <theItem> is added
				// at the end of the list. If <theIndex>
				// is greater than the population of the
				// list, an exception is raised.

    // = ACCESS

    T&			first()
				{
				  return ((OTC_Bucket<T>*)
				   myList.first())->item();
				}
				// Returns a reference to the first item in
				// the list. If the list is empty, an
				// exception is raised.

    T const&		first() const
				{
				  return ((OTC_Bucket<T>*)
				   myList.first())->item();
				}
				// Returns a reference to the first item in
				// the list. If the list is empty, an
				// exception is raised.

    T&			last()
				{
				  return ((OTC_Bucket<T>*)
				   myList.last())->item();
				}
				// Returns a reference to the last item in
				// the list. If the list is empty, an
				// exception is raised.

    T const&		last() const
				{
				  return ((OTC_Bucket<T>*)
				   myList.last())->item();
				}
				// Returns a reference to the last item in
				// the list. If the list is empty, an
				// exception is raised.

    T&			item(u_int theIndex)
				{
				  return ((OTC_Bucket<T>*)
				   myList.item(theIndex))->item();
				}
				// Returns a reference to the item in the
				// list at the location <theIndex>. If the
				// list is empty, or the index is outside the
				// bounds of the list, an exception is
				// raised.

    T const&		item(u_int theIndex) const
				{
				  return ((OTC_Bucket<T>*)
				   myList.item(theIndex))->item();
				}
				// Returns a reference to the item in the
				// list at the location <theIndex>. If the
				// list is empty, or the index is outside the
				// bounds of the list, an exception is
				// raised.

    T*			pFirst()
				{
				  return myList.isEmpty() ? 0 :
				   &((OTC_Bucket<T>*)myList.first())->item();
				}
				// Returns a pointer to the first item in
				// the list. If the list is empty, a null
				// pointer is returned.

    T const*		pFirst() const
				{
				  return myList.isEmpty() ? 0 :
				   &((OTC_Bucket<T>*)myList.first())->item();
				}
				// Returns a pointer to the first item in
				// the list. If the list is empty, a null
				// pointer is returned.

    T*			pLast()
				{
				  return myList.isEmpty() ? 0 :
				   &((OTC_Bucket<T>*)myList.last())->item();
				}
				// Returns a pointer to the last item in
				// the list. If the list is empty, a null
				// pointer is returned.

    T const*		pLast() const
				{
				  return myList.isEmpty() ? 0 :
				   &((OTC_Bucket<T>*)myList.last())->item();
				}
				// Returns a pointer to the last item in
				// the list. If the list is empty, a null
				// pointer is returned.

    T*			pItem(u_int theIndex)
				{
				  return theIndex >= myList.population()
				   ? 0 : &((OTC_Bucket<T>*)myList.item(
				   theIndex))->item();
				}
				// Returns a pointer to the item in the
				// list at the location <theIndex>. If the
				// list is empty, or the index is outside the
				// bounds of the list, a null pointer is
				// returned.

    T const*		pItem(u_int theIndex) const
				{
				  return theIndex >= myList.population()
				   ? 0 : &((OTC_Bucket<T>*)myList.item(
				   theIndex))->item();
				}
				// Returns a reference to the item in the
				// list at the location <theIndex>. If the
				// list is empty, or the index is outside the
				// bounds of the list, a null pointer is
				// returned.

    // = REMOVAL

    void		removeAll()
				{ myList.removeAll(); }
				// Removes all items from the list.

    void		removeFirst()
				{ myList.removeItem(0); }
				// Removes the first item in the list. If
				// the list is empty, an exception is
				// raised.

    void		removeLast()
				{ myList.removeItem(myList.population()-1); }
				// Removes the last item in the list. If
				// the list is empty, an exception is
				// raised.

    void		removeItem(u_int theIndex)
				{ myList.removeItem(theIndex); }
				// Removes the item at location <theIndex>.
				// Raises an exception if there is no item at
				// location <theIndex>.

    void		removeRange(u_int theStart, u_int theLength)
				{ myList.removeRange(theStart,theLength); }
				// Removes <theLength> items starting at
				// location <theStart>.

    void		removeRange(OTC_Range theRange)
				{
				  myList.removeRange(theRange.lower(),
				   theRange.length());
				}
				// Removes <theLength> items starting at
				// location <theStart>.

    // = ITERATION
    //	   By default, iterators will perform reference counts on the
    //	   buckets in the collection as the iterator moves over the items.
    //	   Performing the reference counts ensures that the iterator
    //	   is not corrupted by additions or removals to the collection.
    //	   If an unsafe iterator is required, for example, to avoid
    //	   grabbing a write lock when using ObjectStore, a second
    //	   argument can be passed to the following functions. The value
    //	   of this argument is either <OTCLIB_SAFE> or <OTCLIB_UNSAFE>.
    //	   To get an unsafe iterator, the <OTCLIB_UNSAFE> argument should
    //	   be used.
    //
    //	   The first argument to the following functions indicates the
    //	   direction of traversal of the iterator. Traversal in the
    //	   direction of first to last item is indicated by a value of
    //	   <OTCLIB_FORWARD>. Traversal in the reverse direction is
    //	   indicated by a value of <OTCLIB_BACKWARD>. The default value
    //	   is <OTCLIB_FORWARD>.

    OTC_Iterator<T>	items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const
				{ return _items(theDirection,theProtection); }
				// Returns an iterator over the list.

    OTC_Modifier<T>	items(
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			)
				{ return _items(theDirection,theProtection); }
				// Returns an iterator over the list.

    OTC_Iterator<T>	items(
			 OTC_Range const& theRange,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const
				{
				  return _items(theRange.lower(),
				   theRange.length(),theDirection,
				   theProtection);
				}
				// Returns an iterator over a range of
				// items in the list.

    OTC_Modifier<T>	items(
			 OTC_Range const& theRange,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			)
				{
				  return _items(theRange.lower(),
				   theRange.length(),theDirection,
				   theProtection);
				}
				// Returns an iterator over a range of
				// items in the list.

    OTC_Iterator<T>	items(
			 u_int theStart,
			 u_int theLength,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const
				{
				  return _items(theStart,theLength,
				   theDirection,theProtection);
				}
				// Returns an iterator over a range of
				// items in the list.

    OTC_Modifier<T>	items(
			 u_int theStart,
			 u_int theLength,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			)
				{
				  return _items(theStart,theLength,
				   theDirection,theProtection);
				}
				// Returns an iterator over a range of
				// items in the list.

  public:

#if defined(ENV_OSTORE)
    static os_typespec* get_os_typespec();
#endif

  protected:

    OTC_Cursor<T>*	_items(
			 OTC_Direction theDirection,
			 OTC_Protection theProtection
			) const;
				// Returns a cursor over the list.

    OTC_Cursor<T>*	_items(
			 int theStart,
			 u_int theLength,
			 OTC_Direction theDirection=OTCLIB_FORWARD,
			 OTC_Protection theProtection=OTCLIB_SAFE
			) const;
				// Returns a cursor over a portion of
				// the list.

    OTC_Link*		_link(T const& theItem);
				// Creates an instance of <OTC_Bucket> to
				// hold <theItem>.

  private:

    OTC_ListI		myList;
				// The actual list.
};

/* ------------------------------------------------------------------------- */

#if defined(EXPAND_TEMPLATES)
#include <OTC/collctn/list.c>
#endif

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_LIST_HH */
