#ifndef OTC_COLLCTN_LISTI_HH
#define OTC_COLLCTN_LISTI_HH
/*
// ============================================================================
//
// = LIBRARY
//     OTC
// 
// = FILENAME
//     collctn/listi.hh
//
// = AUTHOR(S)
//     Graham Dumpleton
// 
// = COPYRIGHT
//     Copyright 1994 DUMPLETON SOFTWARE CONSULTING PTY LIMITED
//
// ============================================================================
*/

#include <OTC/collctn/avltree.hh>
#include <OTC/collctn/range.hh>
#include <OTC/collctn/linklist.hh>
#include <OTC/refcnt/rcount.hh>

#ifdef __GNUG__
#if (__GNUC__ >= 3 || __GNUC_MINOR__ >= 6) || defined(CXX_CYGNUS)
#pragma interface "OTC/collctn/listi.hh"
#else
#pragma interface
#endif
#endif

/* ------------------------------------------------------------------------- */

class OTC_ListI
    // = TITLE
    //	   Core for implementing lists including reference by index position.
    //
    // = CLASS TYPE
    //	  Concrete
    //
    // = DESCRIPTION
    //	   The <OTC_ListI> class encapsulates the base functionality
    //	   for implementing <OTC_List>.
{
  public:

			~OTC_ListI();

    // = INITIALISATION

			OTC_ListI();
				// Creates an empty list.

			OTC_ListI(OTC_ListI const& theList);
				// Creates a list which is an alias for
				// <theList>. Ie., each shares and access
				// the same implementation.

    // = QUERY

    OTC_LinkList*	list() const
				{ return myList; }
				// Returns the underlying list.

    OTC_AVLTree*	tree() const
				{ return myTree; }
				// Returns the underlying index tree.

    u_int		population() const
				{ return myTree->population(); }
				// Returns the number of links in the
				// list.

    OTC_Boolean		isEmpty() const
				{ return myTree->population() == 0; }
				// Returns <OTCLIB_TRUE> if the the list
				// is empty.

    // = REMOVAL

    void		removeAll()
				{ myList->removeAll(); myTree->removeAll(); }
				// Removes all links from the list.

    void		removeFirst()
				{ removeItem(0); }
				// Removes the first link in the list. If
				// the list is empty, an exception is
				// raised.

    void		removeLast()
				{ removeItem(myTree->population()-1); }
				// Removes the last link in the list. If
				// the list is empty, an exception is
				// raised.

    void		removeItem(int theIndex);
				// Removes the link at location <theIndex>.
				// Raises an exception if there is no link at
				// location <theIndex>.

    void		removeRange(int theStart, u_int theLength);
				// Removes <theLength> links starting at
				// location <theStart>. Raises an exception
				// of the index or link are invalid.

    void		removeRange(OTC_Range theRange)
				{
				  removeRange(theRange.lower(),
				   theRange.length());
				}
				// Removes <theLength> links starting at
				// location <theStart>. Raises an exception
				// of the index or link are invalid.

    // = RETRIEVAL

    OTC_Link*		first() const;
				// Returns the first link in the list.
				// Raises an exception if the list is empty.

    OTC_Link*		last() const;
				// Returns the last link in the list. Raises
				// an exception if the list is empty.

    OTC_Link*		item(int theIndex) const;
				// Returns the link an location <theIndex>.
				// Raises an exception if <theIndex> is
				// invalid.

    // = ADDITION

    void		addFirst(OTC_Link* theLink)
				{ addBeforeItem(theLink,0); }
				// Adds <theLink> at the start of the
				// the list.

    void		addLast(OTC_Link* theLink)
				{
				  addBeforeItem(theLink,
				   myTree->population());
				}
				// Adds <theLink> at the end of the
				// the list.

    void		addBeforeItem(OTC_Link* theLink, int theIndex);
				// Adds <theLink> in the list at a location
				// <theIndex>.

  private:

    OTC_ListI&		operator=(OTC_ListI const&);
				// Do not define an implementation for this.

    OTC_LinkList*	myList;
				// The actual list.

    OTC_AVLTree*	myTree;
				// The index tree.
};

/* ------------------------------------------------------------------------- */

#endif /* OTC_COLLCTN_LISTI_HH */
